/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __CDRIPPER_H__
#define __CDRIPPER_H__

#include <qvaluelist.h>

#include "docwidget.h"
#include "cdwriter.h"
#include "cdinfo.h"

/*==================================*/
/*	DEFINES
 *==================================*/

enum {
	RIPPERFMT_WAV,
	RIPPERFMT_MP3,
	RIPPERFMT_OGG,
#ifdef FLAC
	RIPPERFMT_FLAC,
#endif
	RIPPERFMT_AU,
	RIPPERFMT_CDR,
	RIPPERFMT_AIFF,
	RIPPERFMT_AIFC,
	_RIPPERFMT_MAX,
};

enum {
	MP3ENCODER_BLADEENC,
	MP3ENCODER_LAME,
	_MP3ENCODER_MAX,
};

/*========================================================*/

class ArsonEncoderProcess;
class ArsonRipperMgr;
class QComboBox;
class QLineEdit;

struct RIPPERFORMAT
{
	enum {
		encoderNone,
		encoderMp3,
		encoderOgg,
#ifdef FLAC
		encoderFlac,
#endif
	};

	RIPPERFORMAT(const QString &desc, const char *ext, uint enc = encoderNone)
		: description(desc), extension(ext), encoder(enc) { }

	ArsonEncoderProcess *createEncoder(
		ArsonRipperMgr *pMgr,
		const char *infile,
		const char *outfile) const;

	static void fillComboBox (QComboBox *pWnd);

	const QString description;
	const char *extension;
	const uint encoder;
};

/*========================================================*/

struct MP3ENCODER {
	MP3ENCODER(const QString &desc) : description(desc) {}

	static ArsonEncoderProcess *createMp3Encoder(
		ArsonRipperMgr *pMgr,
		const char *infile,
		const char *outfile);

	const QString description;
};

/*========================================================*/

class ArsonRipperListItem : public ArsonListItem
{
public:
	ArsonRipperListItem (int trackNo);

	virtual QString display (void) const { return QString(); }
	virtual QString persist (void) const { return QString(); }
	
	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);
	virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd);

	void applyTo (QListViewItem *ptr, ArsonListWnd *pWnd);

	int trackNo (void) const { return m_trackNo; }

private:
	QString defaultFileName (const RIPPERFORMAT *pFmt) const;

	int m_trackNo;
};

/*========================================================*/

class ArsonCdInfo;

class ArsonRipperDoc : public ArsonListDoc
{
	Q_OBJECT
	
public:
	ArsonRipperDoc (QWidget *parent, const char *name = NULL);

	virtual void create (void);
	virtual void deleteContents (void);

	const ArsonCdInfo &info (void) const { return m_info; }
	
	static MP3ENCODER *mp3encoder (int index) { return (mp3encoders + index); }
	static RIPPERFORMAT *format (int index) { return (formats + index); }

	struct ITEMPAIR { QCheckListItem *pListItem; ArsonRipperListItem *pRipperItem; };
	typedef QValueList<ITEMPAIR> ITEMLIST;
	ITEMLIST checkedItems (void) const;

	QString filename (QListViewItem *pi) const;
	const RIPPERFORMAT *outputFormat (void) const;
	void setOutputPath (const QString &path);
	QString outputPath (void) const;

protected:
	virtual void connectTo (ArsonActionConnector &ac);
	virtual Status *initStatus (QStatusBar *psb);
	
	void applyDiskInfo (void);
	void readTrackInfo (void);
	void cdIndex (void);

	ArsonCdInfo m_info;
	
	static MP3ENCODER mp3encoders[_MP3ENCODER_MAX];
	static RIPPERFORMAT formats[_RIPPERFMT_MAX];

	virtual ArsonListWnd *createListWnd (void);
	virtual bool buildContextMenu (QPopupMenu *pm);

	QString totalSelected (int *total = NULL, int *secs = NULL) const;

	QComboBox *m_pFormat;
	QLineEdit *m_pPathField;

protected slots:
	void slotItemStateChange (QListViewItem *ptr);
	void slotFormatChanged (int index);
	void slotBrowse (void);

public slots:
	void toggleChecked (void);
	void slotEditDiskInfo (void);
	void slotRescan (void);
	void slotRip (void);
};

/*========================================================*/

class ArsonRipperProcess;
class ArsonFifo;

class ArsonRipperMgr : public ArsonProcessMgr
{
	struct TRACKINFO {
		TRACKINFO(const char *outfile = NULL, int track = -1)
			: filename(outfile),
			trackNo(track) {}

		bool operator== (const TRACKINFO &info) const
		{ return (filename == info.filename); }
		
		QString filename;
		int trackNo;
	};

	typedef QValueList<TRACKINFO> TRACKLIST;

public:
	ArsonRipperMgr (ArsonProcessUI *pUI,
		ArsonRipperDoc *pDoc,
		const RIPPERFORMAT *pFmt);
	virtual ~ArsonRipperMgr (void);

	virtual void taskComplete (ArsonProcess *ptr);
	virtual void begin (const ArsonProcessOpts &opts);

	void ripTrack (int trackNo = -1, const char *outfile = NULL);
	void next (void);

	void addFile (int trackNo, const char *outfile);
	
private:
	ArsonRipperProcess *createRipperProcess (ArsonRipperMgr *pThis,
		int trackNo, const char *outfile);

	bool verifyDirectory (const char *outfile) const;
	void tagFile (void);
	
	const RIPPERFORMAT *m_pFmt;
	TRACKLIST m_tracks;
	int m_currentTrack;
	ArsonRipperDoc *m_pDoc;
	ArsonFifo *m_pFifo;
};

/*========================================================*/
#endif	/*	__CDRIPPER_H__	*/
