/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	CDDB interface for arson
 *
 *	by Tony Sideris	(08:39PM Feb 13, 2002)
 *================================================*/
#include "arson.h"

#include <unistd.h>
#include <stdio.h>

#include <qregexp.h>

#include <klocale.h>

#include "konfig.h"
#include "cddb.h"

/*========================================================*/
/*	const commands
 *========================================================*/

class helloCmd : public ArsonHttpCommand
{
public:
	helloCmd (void)
		: ArsonHttpCommand("hello")
	{
		char buf[512];
		const char *username = getlogin();

		if (!username && !(username = getenv("USER")))
			username = "arsonuser";

		gethostname(buf, sizeof(buf));

		(*this) << username << buf << PACKAGE << VERSION;
	}
};

class protoCmd : public ArsonHttpCommand
{
public:
	protoCmd (void)
		: ArsonHttpCommand("proto")
	{
		(*this) << 5;
	}
};

const helloCmd HELLO;
const protoCmd PROTO;

/*========================================================*/
/*	Base class for CDDB sockets
 *========================================================*/

ArsonCddbBaseSocket::ArsonCddbBaseSocket (const ArsonCdInfo &info,
	const QString &host, short port)
	: ArsonHttpGetSocket(host, port),
	m_info(info)
{
	//	Nothing...
}

/*========================================================*/

QString ArsonCddbBaseSocket::path (void) const
{
	return QString("/~cddb/cddb.cgi");
}

/*========================================================*/

QString ArsonCddbBaseSocket::request (void) const
{
	ArsonHttpCommand req = cddbCommand();

	req << HELLO << PROTO;
	return req.command();
}

/*========================================================*/
/*	Socket class which performs the cddb query command
 *========================================================*/

ArsonCddbQuerySocket::ArsonCddbQuerySocket (const ArsonCdInfo &info,
	const QString &host, short port)
	: ArsonCddbBaseSocket(info, host, port)
{
	//	Nothing...
}

ArsonHttpCommand ArsonCddbQuerySocket::cddbCommand (void) const
{
	ArsonHttpCommand cmd ("cmd");
	const int count = m_info.trackCount();
	const int elead = (m_info.leadout() / 75);

	cmd << "cddb" << "query" << m_info.cddbID() << count;
	
	for (int index = 0; index < count; ++index)
		cmd << m_info.track(index).offset();

	Trace("CDDB Query:\n count = %d\n leadout = %d\n e = %d\n",
		count, m_info.leadout(), elead);
	
	cmd << elead;
	return cmd;
}

/*========================================================*/
/*	Socket class which performs cddb read command
 *========================================================*/

ArsonCddbReadSocket::ArsonCddbReadSocket (const ArsonCdInfo &info,
	const QString &categ, const QString &host, short port)
	: ArsonCddbBaseSocket(info, host, port),
	m_categ(categ)
{
	//	Nothing...
}

/*========================================================*/

ArsonHttpCommand ArsonCddbReadSocket::cddbCommand (void) const
{
	ArsonHttpCommand cmd ("cmd");

	cmd << "cddb" << "read" << m_categ << m_info.cddbID();
	return cmd;
}

/*========================================================*/
/*	Repsonse parser class
 *========================================================*/

ArsonCddbParserBase::ArsonCddbParserBase (QTextStream *ps)
	: m_ps(ps)
{
	Assert(m_ps != NULL);
}

/*========================================================*/

bool ArsonCddbParserBase::checkResponse (void)
{
	const QString line = nextLine();
	bool valid;

	/*	This == vary cheesy code...
	 *	i should actually parse this
	 *	line.
	 */
	valid = (line[0] >= '1' && line[0] <= '3');

	Trace("Response line: %s (%d)\n",
		line.ascii(), valid);

	return valid;
}
	
/*========================================================*/

QString ArsonCddbParserBase::nextLine (void)
{
	QString line = m_ps->readLine();//.stripWhiteSpace();

	if (line == ".")
		line = QString::null;

	return line;
}

/*========================================================*/
/*	Split the input string into two strings at the
 *	first instance of the sep string.
 *========================================================*/

bool ArsonCddbParserBase::split (const QString &str,
	QString &lhs, QString &rhs, const QString &sep)
{
	const int pos = str.find(sep);

	if (pos != -1)
	{
		lhs = str.left(pos);//.stripWhiteSpace();
		rhs = str.mid(pos + sep.length());//.stripWhiteSpace();

		return true;
	}

	return false;
}

/*========================================================*/
/*	Parses the cddb query response
 *========================================================*/

ArsonCddbQueryParser::ArsonCddbQueryParser (QTextStream *ps)
	: ArsonCddbParserBase(ps)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonCddbQueryParser::checkResponse (void)
{
	const QString line = nextLine();
	const int code = line.left(3).toInt();

	Trace("Cddb query response: %d (%s)\n", code, line.latin1());

	if (code == 200)
		parseLine(line.latin1() + 4);

	return (code >= 200 && code < 400);
}

/*========================================================*/

bool ArsonCddbQueryParser::parse (void)
{
	for (QString line = nextLine(); line != QString::null; line = nextLine())
		if (!parseLine(line)) break;

	return (m_cats.count() > 0);
}

/*========================================================*/

bool ArsonCddbQueryParser::parseLine (const QString &line)
{
	QString temp;
	Category cat;

	Trace("ArsonCddbQueryParser::parseLine: %s\n",
		line.ascii());
		
	if (!split(line, cat.categ, temp, " "))
		return false;

	split(temp, cat.id, cat.title, " ");

	Trace("Adding %s:%s:%s\n",
		cat.categ.latin1(),
		cat.id.latin1(),
		cat.title.latin1());

	m_cats.append(cat);
	return true;
}

/*========================================================*/

QString ArsonCddbQueryParser::category (void) const
{
	Assert(m_cats.count() > 0);
	return m_cats[0].categ;
}

/*========================================================*/
/*	Parses the cddb read response
 *========================================================*/

ArsonCddbReadParser::ArsonCddbReadParser (QTextStream *ps)
	: ArsonCddbParserBase(ps)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonCddbReadParser::parse (void)
{
	for (QString line = nextLine(); line != QString::null; line = nextLine())
	{
		QString lhs, rhs;

//		Trace("CDDB Line: %s\n", line.ascii());
		
		//	Ignore blanks lines and comments
		if (line.isEmpty() || line[0] == '#')
			continue;

		//	Seperate key from value
		if (split(line, lhs, rhs, "=") && !rhs.isEmpty())
		{
			VALMAP::Iterator it = m_map.find(lhs.ascii());

			if (it == m_map.end())
				it = m_map.insert(lhs.ascii(), rhs);
			else
				it.data().append(rhs);
		}
	}

	return m_map.contains("DISCID") &&
		m_map.contains("DTITLE");
}

/*========================================================*/

QString ArsonCddbReadParser::value (const char *key) const
{
	if (m_map.contains(key))
		return m_map[key];

	return QString::null;
}

/*========================================================*/

QCString ArsonCddbParserBase::encode (const QCString &str)
{
	QCString result (str.length() * 2 + 1);
	int index, ctr;
	char ch;

	for (index = ctr = 0; index < str.length(); ++index, ++ctr)
	{
		switch (str[index])
		{
		case '\n': ch = 'n'; break;
		case '\t': ch = 't'; break;
		case '\\': ch = '\\'; break;
		default: result[ctr] = str[index]; continue;
		}

		result[ctr++] = '\\';
		result[ctr] = ch;
	}

	result[ctr] = '\0';
	return result;
}

QCString ArsonCddbParserBase::decode (const QCString &str)
{
	QCString result (str.length() + 1);
	int index, ctr;
	char ch;

	for (index = ctr = 0; index < str.length(); ++index, ++ctr)
	{
		ch = str[index];

		if (str[index] == '\\')
		{
			switch (str[++index])
			{
			case 'n': ch = '\n'; break;
			case 't': ch = '\t'; break;
			case '\\': ch = '\\'; break;
			}
		}

		result[ctr] = ch;
	}

	result[ctr] = '\0';
	return result;
}

/*========================================================*/

void ArsonCddbReadParser::applyTo (ArsonCdInfo &info) const
{
	bool various = false;
	char buf[10] = {0,};
	QString artist, title;

	if (!split(value("DTITLE"), artist, title, " / "))
		artist = title = value("DTITLE");

	info.setArtist(artist);
	info.setTitle(title);

	info.setGenre(value("DGENRE"));
	info.setYear(value("DYEAR"));
	
	for (int index = 0; index < info.trackCount(); ++index)
	{
		ArsonCdInfo::Track &track = info.track(index);

		sprintf(buf, "TTITLE%d", index);

		if (split(value(buf), artist, title, " / "))
		{
			track.setArtist(artist);
			track.setTitle(title);
			various = true;
		}
		else
			track.setTitle(value(buf));

		sprintf(buf, "EXTT%d", index);
		track.setComment(decode(value(buf).latin1()));
	}

	info.setComment(decode(value("EXTD").latin1()));
	
	if (various)
		info.setArtist(QString::null);
}

/*========================================================*/
/*	CDDB interface dialog
 *========================================================*/

ArsonCddbDlg::ArsonCddbDlg (const ArsonCdInfo &info, const QString &addr, short port)
	: ArsonCdInfoWaitDlg(info), m_bQueried(false),
	m_addr(addr), m_port(port)
{
	setMessage(
		i18n("Getting CD information from freedb server..."));

	setCaption(
		i18n("CDDB Query"));
}

/*========================================================*/

void ArsonCddbDlg::handleStream (QTextStream *pStream)
{
	if (!m_bQueried)
	{
		ArsonCddbQueryParser parser (pStream);

		if (parser.checkResponse() && parser.parse())
		{
			Trace("CDDB query successful\n");
			
			m_categ = parser.category();
			m_bQueried = true;

			beginConnection();
			return;
		}
	}
	else
	{
		ArsonCddbReadParser parser (pStream);

		if (parser.checkResponse() && parser.parse())
		{
			m_info.setCateg(m_categ);
			parser.applyTo(m_info);
			
			accept();
			return;
		}
	}

	reject();
}

/*========================================================*/

ArsonSocket *ArsonCddbDlg::createSocket (void)
{
	ArsonRetrSocket *ptr = NULL;

	if (!m_bQueried)
		ptr = new ArsonCddbQuerySocket(m_info, m_addr, m_port);
	else
	{
		Assert(m_categ != QString::null);

		Trace("Connecting to %s:%d for CDDB read\n",
			m_addr.latin1(), m_port);
		
		ptr = new ArsonCddbReadSocket(
			m_info, m_categ,
			m_addr, m_port);
	}

	return ptr;
}

/*========================================================*/
