//audio.h:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2013
 *
 *  This file is part of aroarfw, a RoarAudio framework for
 *  embedded systems (Controlers).
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  aroarfw is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef _AROARFW_AUDIO_H_
#define _AROARFW_AUDIO_H_

//! Channel setups for RoarAudio
typedef enum {
 //! No channels
 RCHANNELS_NONE   = 0,
 //! Mono stream (one channel)
 RCHANNELS_MONO   = 1,
 //! Stereo stream (two channels)
 RCHANNELS_STEREO = 2
} rchannels_t;

//! Codec used by streams or other audio data
typedef enum {
 //! Signed PCM data in little endian
 RCODEC_PCM_S_LE     = 0x01,
 //! Signed PCM data in big endian
 RCODEC_PCM_S_BE     = 0x02,
 //! Signed PCM data in PDP (middle) endian
 RCODEC_PCM_S_PDP    = 0x03,
 //! Unsigned PCM data in little endian
 RCODEC_PCM_U_LE     = 0x05,
 //! Unsigned PCM data in big endian
 RCODEC_PCM_U_BE     = 0x06,
 //! Unsigned PCM data in PDP (middle) endian
 RCODEC_PCM_U_PDP    = 0x07,
 //! Ogg Vorbis
 RCODEC_OGG_VORBIS   = 0x10,
 //! Native FLAC (.flac)
 RCODEC_FLAC         = 0x11,
 //! Ogg Speex (.spx)
 RCODEC_OGG_SPEEX    = 0x12,
 //! Ogg FLAC
 RCODEC_OGG_FLAC     = 0x14,
 //! Ogg CELT
 RCODEC_OGG_CELT     = 0x16,
 //! General Ogg (mixed/unknown content)
 RCODEC_OGG_GENERAL  = 0x15,
 //! Ogg container itself (to be used with virtual streams)
 RCODEC_OGG          = 0x17,
 //! Opus in Ogg
 RCODEC_OGG_OPUS     = 0x18,
 //! RoarOpus (.ro)
 RCODEC_ROAR_OPUS    = 0x19,
 //! RoarCELT (.rc)
 RCODEC_ROAR_CELT    = 0x1a,
 //! RoarSpeex (.rs)
 RCODEC_ROAR_SPEEX   = 0x1b,
 //! RIFF WAVE container (.wav) (may be used with virtual streams)
 RCODEC_RIFF_WAVE    = 0x20,
 //! Big endian RIFF WAVE file
 RCODEC_RIFX         = 0x22,
 //! Sun Audio file format (.au) (may be used with virtual streams)
 RCODEC_AU           = 0x24,
 //! Audio Interchange File Format (may be used with virtual streams)
 RCODEC_AIFF         = 0x28,
 //! A-Law
 RCODEC_ALAW         = 0x30,
 //! A-Law alike 16 bit codec in little endian (uncommon)
 RCODEC_AUTLAW_LE    = 0x31,
 //! A-Law alike 16 bit codec in big endian
 RCODEC_AUTLAW_BE    = 0x32,
 //! Alias for A-Law alike 16 bit codec
 RCODEC_AUTLAW       = RCODEC_AUTLAW_BE,
 //! mu-Law
 RCODEC_MULAW        = 0x34,
 //! mu-Law alike 16 bit codec in little endian (uncommon)
 RCODEC_MUUTLAW_LE   = 0x35,
 //! mu-Law alike 16 bit codec in big endian
 RCODEC_MUUTLAW_BE   = 0x35,
 //! Alias for mu-Law alike 16 bit codec
 RCODEC_MUUTLAW      = RCODEC_MUUTLAW_BE,
 //! GSM Codec
 RCODEC_GSM          = 0x38,
 //! GSM Codec for RIFF_WAVE files
 RCODEC_GSM49        = 0x39,
 //! SPC-700 Bit Rate Reduction of Super Nintendo Entertainment System (SNES)
 RCODEC_BRR          = 0x3c,
 //! MIDI File
 RCODEC_MIDI_FILE    = 0x08,
 //! MIDI (on-wire) protocol
 RCODEC_MIDI         = 0x60,
 //! RoarMIDI protocol
 RCODEC_ROARMIDI     = 0x64,
 //! DMX512
 RCODEC_DMX512       = 0x70,
 //! RoarDMX
 RCODEC_ROARDMX      = 0x71,

 //! RAUM Files
 RCODEC_RAUM         = 0x1c,
 //! RAUM Files with Vorbis stream
 RCODEC_RAUM_VORBIS  = 0x1d,
 //! RAUM Files with FLAC stream
 RCODEC_RAUM_FLAC    = 0x1e,
 //! Vorbis Comment Like Text
 RCODEC_META_VCLT    = 0x40,
 //! RoarAudio Like Text
 RCODEC_META_RALT    = 0x44,
 //! RoarAudio Like Binary
 RCODEC_META_RALB    = 0x4c,
 //! RoarAudio Like Binary (little endian)
 RCODEC_META_RALB_LE = 0x4d,
 //! RoarAudio Like Binary (big endian)
 RCODEC_META_RALB_BE = 0x4e,
 //! RoarAudio Like Binary (PDP (middle) endian)
 RCODEC_META_RALB_PDP= 0x4f,
 //! Null container
 RCODEC_CONT_NULL    = 0x50,
 //! GnuZIP Container
 RCODEC_CONT_GZIP    = 0x51,
 //! bzip2 container
 RCODEC_CONT_BZIP2   = 0x52,
 //! OpenPGP binary data container
 RCODEC_CONT_OPGPBIN = 0x53,
 //! OpenPGP ascii armored container
 RCODEC_CONT_OPGPASC = 0x54,
 //! TAR container
 RCODEC_CONT_TAR     = 0x55,
 //! RDS (on-wire) protocol
 RCODEC_RDS          = 0x80,
 //! User defined codec 0
 RCODEC_USER0        = 0x90,
 //! User defined codec 1
 RCODEC_USER1        = 0x91,
 //! User defined codec 2
 RCODEC_USER2        = 0x92,
 //! User defined codec 3
 RCODEC_USER3        = 0x93,
 //! User defined codec 4
 RCODEC_USER4        = 0x94,
 //! User defined codec 5
 RCODEC_USER5        = 0x95,
 //! User defined codec 6
 RCODEC_USER6        = 0x96,
 //! User defined codec 7
 RCODEC_USER7        = 0x97,
 //! User defined codec 8
 RCODEC_USER8        = 0x98,
 //! User defined codec 9
 RCODEC_USER9        = 0x99,
 //! User defined codec 10
 RCODEC_USER10       = 0x9a,
 //! User defined codec 11
 RCODEC_USER11       = 0x9b,
 //! User defined codec 12
 RCODEC_USER12       = 0x9c,
 //! User defined codec 13
 RCODEC_USER13       = 0x9d,
 //! User defined codec 14
 RCODEC_USER14       = 0x9e,
 //! User defined codec 15
 RCODEC_USER15       = 0x9f
 //! User defined codec 16
} rcodec_t;

//! RoarSpeex mode
typedef enum {
 //! Narrowband (8 kHz)
 RSPEEX_MODE_NB  = 1,
 //! Wideband (16 kHz)
 RSPEEX_MODE_WB  = 2,
 //! Ultra-wideband (32 kHz)
 RSPEEX_MODE_UWB = 3
} rspeex_mode_t;

//! Default sample rate for audio streams
#define RDEFAULT_AU_RATE         44100

//! Default number of bits per sample for audio streams
#define RDEFAULT_AU_BITS            16

//! Default channel setup for audio streams
#define RDEFAULT_AU_CHANNELS     RCHANNELS_STEREO

//! RoarSpeex file magic string
#define RSPEEX_MAGIC           "RoarSpeex"
//! Length of RoarSpeex file magic string
#define RSPEEX_MAGIC_LEN       9
//! RoarCELT file magic string
#define RCELT_MAGIC            "RoarCELT0"
//! Length of RoarCELT file magic string
#define RCELT_MAGIC_LEN        9

//! Struct to store Audio Signal information
typedef struct raudio_info {
 //! Sample rate
 unsigned int rate;
 //! Number of bits per sample
 unsigned int bits;
 //! Number of channels per frame
 rchannels_t channels;
 //! Used codec
 rcodec_t codec;
} raudio_info_t;

#endif

//ll
