/*
 * Copyright (c) 2019 ARM Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"

#include <cstdint>
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_smallK_hybrid_u8u32_dot_1VLx8(const uint8_t *A, int lda, const uint8_t *B, uint32_t *C, int ldc, int M, int N, int K, const uint32_t *, Activation, bool) {
    const long loops_count = iceildiv(N, (int)get_vector_length<uint32_t>()) - 1;
    const long ldab = lda * sizeof(uint8_t);
    const long ldcb = ldc * sizeof(uint32_t);
    const long odd_depth  = (K % 16) ? (K % 16) : 16;
    const long last_width = N - (loops_count * get_vector_length<uint32_t>());
    const long odds_count = K % 4;
    K = (K + 3) / 4;

    for (int y0=0; y0<M; y0+=8) {
        long loops = loops_count;
        long oob_rows = std::max(8 - (M-y0), 0);
        long odds = odds_count;
        long temp = 0;
        const uint8_t *b_ptr0 = B;
        const uint8_t *a_ptr0 = A + (y0 * lda);

        uint32_t *c_ptr0 = C + (y0 * ldc);

        switch(K) {
            case 1:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0]]\n"
                    "mov z29.s, #0\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1]\n"
                    "mov z30.s, #0\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2]\n"
                    "mov z31.s, #0\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "cbz %[loops], 2f\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 2:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0]]\n"
                    "mov z29.s, #0\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1]\n"
                    "mov z30.s, #0\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2]\n"
                    "mov z31.s, #0\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 3:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0]]\n"
                    "mov z29.s, #0\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1]\n"
                    "mov z30.s, #0\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2]\n"
                    "mov z31.s, #0\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 4:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0]]\n"
                    "mov z29.s, #0\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1]\n"
                    "mov z30.s, #0\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2]\n"
                    "mov z31.s, #0\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 5:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 6:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 7:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 8:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 9:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 10:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #2\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 11:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #3\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 12:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #4\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 13:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #5\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 14:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #6\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            case 15:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "mov z24.s, #0\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #7\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
            default:
            case 16:
                __asm __volatile (
                    "a_ptr1 .req X0\n"
                    "a_ptr2 .req X1\n"
                    "a_ptr3 .req X2\n"
                    "a_ptr4 .req X3\n"
                    "a_ptr5 .req X4\n"
                    "a_ptr6 .req X5\n"
                    "a_ptr7 .req X6\n"
                    "c_ptr1 .req X7\n"
                    "c_ptr2 .req X8\n"
                    "c_ptr3 .req X9\n"
                    "c_ptr4 .req X10\n"
                    "c_ptr5 .req X11\n"
                    "c_ptr6 .req X12\n"
                    "c_ptr7 .req X13\n"
                    "add a_ptr1, %[a_ptr0], %[lda]\n"
                    "add c_ptr1, %[c_ptr0], %[ldc]\n"
                    "add a_ptr2, a_ptr1, %[lda]\n"
                    "add c_ptr2, c_ptr1, %[ldc]\n"
                    "add a_ptr3, a_ptr2, %[lda]\n"
                    "add c_ptr3, c_ptr2, %[ldc]\n"
                    "add a_ptr4, a_ptr3, %[lda]\n"
                    "add c_ptr4, c_ptr3, %[ldc]\n"
                    "add a_ptr5, a_ptr4, %[lda]\n"
                    "add c_ptr5, c_ptr4, %[ldc]\n"
                    "add a_ptr6, a_ptr5, %[lda]\n"
                    "add c_ptr6, c_ptr5, %[ldc]\n"
                    "add a_ptr7, a_ptr6, %[lda]\n"
                    "add c_ptr7, c_ptr6, %[ldc]\n"
                    "cbz %[oob_rows], 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr7, %[c_ptr0], #0x0\n"
                    "add a_ptr7, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr6, %[c_ptr0], #0x0\n"
                    "add a_ptr6, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr5, %[c_ptr0], #0x0\n"
                    "add a_ptr5, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr4, %[c_ptr0], #0x0\n"
                    "add a_ptr4, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr3, %[c_ptr0], #0x0\n"
                    "add a_ptr3, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr2, %[c_ptr0], #0x0\n"
                    "add a_ptr2, %[a_ptr0], #0x0\n"
                    "b.eq 1f\n"
                    "subs %[oob_rows], %[oob_rows], #0x1\n"
                    "add c_ptr1, %[c_ptr0], #0x0\n"
                    "add a_ptr1, %[a_ptr0], #0x0\n"
                    "1:\n"
                    "mov z24.s, #0\n"
                    "ptrue p7.b\n"
                    "mov z25.s, #0\n"
                    "whilelt p6.b, %[temp], %[odd_depth]\n"
                    "mov z26.s, #0\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "mov z27.s, #0\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "mov z28.s, #0\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "mov z29.s, #0\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "mov z30.s, #0\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "mov z31.s, #0\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "whilelt p0.s, %[temp], %[last_width]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x40]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x80]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0xc0]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x100]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x140]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "prfm PLDL1KEEP, [a_ptr7, #0x180]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "cbz %[loops], 2f\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "b.eq 3f\n"
                    "4:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "subs %[loops], %[loops], #0x1\n"
                    "mov z24.s, #0\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "prfm PSTL1KEEP, [%[c_ptr0], #0x40]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr1, #0x40]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "prfm PSTL1KEEP, [c_ptr2, #0x40]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr3, #0x40]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr4, #0x40]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr5, #0x40]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr6, #0x40]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "prfm PSTL1KEEP, [c_ptr7, #0x40]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "b.ne 4b\n"
                    "3:\n"
                    "st1w z24.s, p7, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "mov z24.s, #0\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "st1w z25.s, p7, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "mov z25.s, #0\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "st1w z26.s, p7, [c_ptr2]\n"
                    "mov z26.s, #0\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "st1w z27.s, p7, [c_ptr3]\n"
                    "mov z27.s, #0\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4]\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "st1w z28.s, p7, [c_ptr4]\n"
                    "mov z28.s, #0\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "st1w z29.s, p7, [c_ptr5]\n"
                    "mov z29.s, #0\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "st1w z30.s, p7, [c_ptr6]\n"
                    "mov z30.s, #0\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "st1w z31.s, p7, [c_ptr7]\n"
                    "mov z31.s, #0\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "ld1b z16.b, p7/z, [%[b_ptr0]]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "ld1b z17.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "ld1b z18.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x10]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x10]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x10]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x10]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x10]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x10]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x10]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x10]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "ld1b z19.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "ld1b z20.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "ld1b z21.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "ld1b z22.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "ld1rqb z0.b, p7/z, [%[a_ptr0], #0x20]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "ld1rqb z1.b, p7/z, [a_ptr1, #0x20]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "ld1rqb z2.b, p7/z, [a_ptr2, #0x20]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "ld1rqb z3.b, p7/z, [a_ptr3, #0x20]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "ld1rqb z4.b, p7/z, [a_ptr4, #0x20]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "ld1rqb z5.b, p7/z, [a_ptr5, #0x20]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "ld1rqb z6.b, p7/z, [a_ptr6, #0x20]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "ld1b z23.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                    "udot z24.s, z16.b, z0.b[0]\n"
                    "ld1rqb z7.b, p7/z, [a_ptr7, #0x20]\n"
                    "udot z25.s, z16.b, z1.b[0]\n"
                    "addvl %[b_ptr0], %[b_ptr0], #8\n"
                    "udot z26.s, z16.b, z2.b[0]\n"
                    "udot z27.s, z16.b, z3.b[0]\n"
                    "udot z28.s, z16.b, z4.b[0]\n"
                    "udot z29.s, z16.b, z5.b[0]\n"
                    "udot z30.s, z16.b, z6.b[0]\n"
                    "udot z31.s, z16.b, z7.b[0]\n"
                    "udot z24.s, z17.b, z0.b[1]\n"
                    "udot z25.s, z17.b, z1.b[1]\n"
                    "udot z26.s, z17.b, z2.b[1]\n"
                    "udot z27.s, z17.b, z3.b[1]\n"
                    "udot z28.s, z17.b, z4.b[1]\n"
                    "udot z29.s, z17.b, z5.b[1]\n"
                    "udot z30.s, z17.b, z6.b[1]\n"
                    "udot z31.s, z17.b, z7.b[1]\n"
                    "udot z24.s, z18.b, z0.b[2]\n"
                    "udot z25.s, z18.b, z1.b[2]\n"
                    "udot z26.s, z18.b, z2.b[2]\n"
                    "udot z27.s, z18.b, z3.b[2]\n"
                    "udot z28.s, z18.b, z4.b[2]\n"
                    "udot z29.s, z18.b, z5.b[2]\n"
                    "udot z30.s, z18.b, z6.b[2]\n"
                    "udot z31.s, z18.b, z7.b[2]\n"
                    "udot z24.s, z19.b, z0.b[3]\n"
                    "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x30]\n"
                    "udot z25.s, z19.b, z1.b[3]\n"
                    "ld1rqb z1.b, p6/z, [a_ptr1, #0x30]\n"
                    "udot z26.s, z19.b, z2.b[3]\n"
                    "ld1rqb z2.b, p6/z, [a_ptr2, #0x30]\n"
                    "udot z27.s, z19.b, z3.b[3]\n"
                    "ld1rqb z3.b, p6/z, [a_ptr3, #0x30]\n"
                    "udot z28.s, z19.b, z4.b[3]\n"
                    "ld1rqb z4.b, p6/z, [a_ptr4, #0x30]\n"
                    "udot z29.s, z19.b, z5.b[3]\n"
                    "ld1rqb z5.b, p6/z, [a_ptr5, #0x30]\n"
                    "udot z30.s, z19.b, z6.b[3]\n"
                    "ld1rqb z6.b, p6/z, [a_ptr6, #0x30]\n"
                    "udot z31.s, z19.b, z7.b[3]\n"
                    "ld1rqb z7.b, p6/z, [a_ptr7, #0x30]\n"
                    "udot z24.s, z20.b, z0.b[0]\n"
                    "udot z25.s, z20.b, z1.b[0]\n"
                    "udot z26.s, z20.b, z2.b[0]\n"
                    "udot z27.s, z20.b, z3.b[0]\n"
                    "udot z28.s, z20.b, z4.b[0]\n"
                    "udot z29.s, z20.b, z5.b[0]\n"
                    "udot z30.s, z20.b, z6.b[0]\n"
                    "udot z31.s, z20.b, z7.b[0]\n"
                    "udot z24.s, z21.b, z0.b[1]\n"
                    "udot z25.s, z21.b, z1.b[1]\n"
                    "udot z26.s, z21.b, z2.b[1]\n"
                    "udot z27.s, z21.b, z3.b[1]\n"
                    "udot z28.s, z21.b, z4.b[1]\n"
                    "udot z29.s, z21.b, z5.b[1]\n"
                    "udot z30.s, z21.b, z6.b[1]\n"
                    "udot z31.s, z21.b, z7.b[1]\n"
                    "udot z24.s, z22.b, z0.b[2]\n"
                    "udot z25.s, z22.b, z1.b[2]\n"
                    "udot z26.s, z22.b, z2.b[2]\n"
                    "udot z27.s, z22.b, z3.b[2]\n"
                    "udot z28.s, z22.b, z4.b[2]\n"
                    "udot z29.s, z22.b, z5.b[2]\n"
                    "udot z30.s, z22.b, z6.b[2]\n"
                    "udot z31.s, z22.b, z7.b[2]\n"
                    "udot z24.s, z23.b, z0.b[3]\n"
                    "udot z25.s, z23.b, z1.b[3]\n"
                    "udot z26.s, z23.b, z2.b[3]\n"
                    "udot z27.s, z23.b, z3.b[3]\n"
                    "udot z28.s, z23.b, z4.b[3]\n"
                    "udot z29.s, z23.b, z5.b[3]\n"
                    "udot z30.s, z23.b, z6.b[3]\n"
                    "udot z31.s, z23.b, z7.b[3]\n"
                    "2:\n"
                    "st1w z24.s, p0, [%[c_ptr0]]\n"
                    "addvl %[c_ptr0], %[c_ptr0], #1\n"
                    "st1w z25.s, p0, [c_ptr1]\n"
                    "addvl c_ptr1, c_ptr1, #1\n"
                    "st1w z26.s, p0, [c_ptr2]\n"
                    "addvl c_ptr2, c_ptr2, #1\n"
                    "st1w z27.s, p0, [c_ptr3]\n"
                    "addvl c_ptr3, c_ptr3, #1\n"
                    "st1w z28.s, p0, [c_ptr4]\n"
                    "addvl c_ptr4, c_ptr4, #1\n"
                    "st1w z29.s, p0, [c_ptr5]\n"
                    "addvl c_ptr5, c_ptr5, #1\n"
                    "st1w z30.s, p0, [c_ptr6]\n"
                    "addvl c_ptr6, c_ptr6, #1\n"
                    "st1w z31.s, p0, [c_ptr7]\n"
                    "addvl c_ptr7, c_ptr7, #1\n"
                    ".unreq a_ptr1\n"
                    ".unreq a_ptr2\n"
                    ".unreq a_ptr3\n"
                    ".unreq a_ptr4\n"
                    ".unreq a_ptr5\n"
                    ".unreq a_ptr6\n"
                    ".unreq a_ptr7\n"
                    ".unreq c_ptr1\n"
                    ".unreq c_ptr2\n"
                    ".unreq c_ptr3\n"
                    ".unreq c_ptr4\n"
                    ".unreq c_ptr5\n"
                    ".unreq c_ptr6\n"
                    ".unreq c_ptr7\n"
                    : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [oob_rows] "+r" (oob_rows), [temp] "+r" (temp), [odds] "+r" (odds)
                    : [lda] "r" (ldab), [ldc] "r" (ldcb), [odd_depth] "r" (odd_depth), [last_width] "r" (last_width)
                    : "x0", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                );
                break;
        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
