/* $Id: ArkWorld.h,v 1.33 2003/03/15 17:49:04 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_WORLD_H
#define ARK_WORLD_H

#include <Ark/ArkCache.h>
#include <Ark/ArkCollision.h>
#include <Ark/ArkFactory.h>
#include <Ark/ArkMath.h>
#include <Ark/ArkPath.h>
#include <Ark/ArkRay.h>
#include <Ark/ArkTimer.h>
#include <Ark/ArkLight.h>

#include <vector>

namespace Ark
{

   class Cache;
   class Camera;
   class Entity;
   class Light;
   class Material;
   class WorldUpdater;
      
   enum
   {
      WORLD_HAS_PATHFINDING = (1<<0),
      WORLD_HAS_RENDERING = (1<<1),
      WORLD_HAS_COLLISION = (1<<2),
   };

   /// A list of entities.
   typedef std::vector<Entity*> EntityList;
   typedef std::vector<Light> LightList;

   //  ========================================================================
   /** The World class is an abstract class embedding any type of
    * world. It provides function to do anything needed in a world :
    *     \arg Pathfinding
    *     \arg Collision detection
    *     \arg (Gravity & physics)
    */
   //  ========================================================================

   class ARK_DLL_API World
   {
      protected:

	 /**
	  * Cache associated with this world. It cant change, and all the
	  * entity models/textures/etc must have been created using it.
	  */
	 Cache *m_Cache;

	 /// Updater this world is bound to.
	 WorldUpdater *m_Updater;

	 /// List of entities contained in this world.
	 EntityList m_Entities;

	 /**
	  * List of lights contained in this world.
	  * FIXME: ArkEntity should be made more generic so that
	  * the lights can be put in EntityList (would require to split
	  * ArkEntity in an ArkRenderableObject class which would be the
	  * parent of ArkEntity, and ArkLight would also derive from 
	  * ArkRenderableObject
	  */
	 LightList m_Lights;

	 /// Frame timer.
	 Timer m_Timer;

	 /// Current numerical entity IDentifier.
	 int m_CurID;

	 /// This is set to true if we are in the World's destructor.
	 bool m_Quit;
	 
      public:
	 /// Create a new world
	 World (Cache *cache, WorldUpdater *updater = 0);
	 
	 /// Destroy the given world
	 virtual ~World ();

	 /** Load the world from the directory pointed to by \c path */
	 virtual bool Load (const String &path) = 0;
	 
	 /** Initialize some "functions of the world (ie pathfinding,
	  * rendering, collision)..
	  */
	 virtual bool Init (int init_flags) = 0;

	 /** Render world using the renderer given as argument. The given
	  * camera might be changed to ensure the "lookat" point can bee
	  * seen from the point of view.
	  */
	 virtual bool Render (Renderer &renderer, const Camera& camera) = 0;
	 
	 /**
	  * Find a path. Origin and goal are contained in path, along with
	  * other informations.
	  */
	 virtual bool FindPath (Path &path) = 0;

	 /** Tell if \c B can be reached if we are at point \c A. Gravity has
	  * the same meaning as with FindPath.
	  *   \return true if A is reacheable from B.
	  */
	 virtual bool IsReachable (const Vector3 &A, const Vector3 &B,
				   bool gravity = true) = 0;
	 
	 /** Tell if this bounding box collides with one or more object in the
	  * world.
	  *
	  *   \return True if it succeeded (box is in the world).
	  *   \param colflags tell which data are needed, so useless
	  *          computations can  be avoided..
	  *   \param collisions is a vector we'll fill with the collision that
	  *          may occur (if its size is zero, there was no collision).
	  */
	 virtual bool TestCollision (const BBox &box, int8 colflags,
				     std::vector<Collision> &collisions) = 0;
	 
	 /** Trace a ray from point \c A to point \c B and return collision
	  * infos for objects it hits. This can be used for several things :
	  *   \arg Test if an object can be seen.
	  *   \arg Apply the gravity to an object.
	  *   \arg Find pointed object from mouse pos,
	  *   \arg ...
	  *
	  *   \return True on success.
	  *   \param colflags tell which data are needed, so useless
	  *          computations can be avoided..
	  *   \param collisions is a vector we'll fill with the collision that
	  *           may occur (if its size is zero, there was no collision).
	  */
	 virtual bool RayTrace (const Ray &ray, int8 colflags,
				std::vector<Collision> &collisions) = 0;
	 
	 /** Finds the rest position for a given point.
	  *
	  * This is used for basic positioning of objects on the ground for example.
	  *   \return The corrected position
	  *   \param position Initial position
	  */
	 virtual Vector3 GetRestPosition(const Vector3& position) const = 0;
	 
	 /// Do collision detection for all the world and send events to the
	 /// corresponding entities.
	 virtual void DetectCollisions () = 0;
	 
	 /** Updates the entire world : calls the update function for each
	  * entities, then do collision detection for all the world and send
	  * events to the corresponding entities.
	  * It also removes entities that are "DEAD" flagged.
	  */
	 virtual void Update (scalar delta = 0.0);
	 
	 //==================================================================

	 /** Add an entity to the world. This function is virtual because the
	  * derived type of world will probably be space-divided, which is
	  * needed to reduce
	  * the costs of collision detection (And the \c World interface can't
	  * know about this division).
	  */
	 virtual void Add (Entity *entity);

	 /**
	  * Add a light to the world. Entity should be modified so that 
	  * it can be used for lights
	  */
	 virtual void Add (const Light &aLight);

	 LightList &GetLights();

	 /**
	  * FIXME: add a Remove (Light);
	  */

	 /// Remove an entity from the world.
	 virtual void Remove (Entity *entity);
	 
	 /// Get the entity list
	 EntityList& GetEntities()
	 {return m_Entities;}
	 
	 /// Find the entity whose id is 'id'
	 Entity *Find (int id) const;

	 /// Find an identity by short name.
	 Entity *FindByName (const String& name) const;

	 /**
	  * Find an identity by class.
	  * \return A list of entities which can be empty
	  */
	 EntityList* FindByClass (const String& className) const;

	 //==================================================================
	 
	 /// Returns the object cache bound to this world
	 Cache *GetCache () const
	 {return m_Cache;}

	 /// Returns the configuration object bound to this world.
	 virtual Config &GetConfig() = 0;
   };

   class ARK_DLL_API WorldFactory : public Factory
   {
      public:
         WorldFactory();
	 virtual ~WorldFactory(){}
	 virtual World *NewWorld(Cache *cache, WorldUpdater *updater = 0) = 0;
	 static World *CreateWorld(const String &impl,
				   Cache *cache, WorldUpdater *updater = 0,
				   FactoryList *faclist = 0);
   };

}

#include <Ark/ArkEntity.h>

#endif
