/* $Id: ArkDataClass.h,v 1.12 2002/10/11 01:10:02 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_DATACLASS_H
#define ARK_DATACLASS_H

#include <Ark/ArkWorld.h>
#include <vector>
#include <map>

namespace Ark
{

   class EntryDef;

 
   /**
    * An Entry contains a value in a specific DataDef instanciation. When a
    * Data object is created from a DataDef one, all data are read from the
    * file, and they are put in Entries (of course the type is checked). If
    * some entries are still missing, defaults values are used.
    */
   class ARK_DLL_API Entry
   {
      public:
	 /// Type of this entry.
	 enum Type
	 {
	    UNKNOWN,
	    STRING,
	    INTEGER,
	    SCALAR,
	    ENTITY,
	    VECTOR2,
	    VECTOR3,
	 };

      public:
	 /// Type of variable contained in this entry.
	 Type m_Type;

	 union
	 {
	       /// String variable
	       String        *d_str;

	       /// Integer or boolean value
	       int            d_integer;
	       
	       /// Scalar (ie float) value
	       scalar         d_scalar;

	       /// 2D vector
	       Vector2       *d_vector2;

	       /// 3D vector
	       Vector3       *d_vector3;

	       /// Pointer value, 
	       void          *d_void;
	 };

	 /**
	  * Eventually destroy the value contained in this entry, depending
	  * on its type (as for yet, only the string will be destroyed).
	  */
	 void Destroy();

      public:

	 /**
	  * Create an entry based on the given definition, with the same
	  * type as the entry def, and a default empty value.
	  * Remark: Entry(def) is the same as Entry(def->m_Type).
	  */
	 Entry (EntryDef *def);

	 /// Copy constructor.
	 Entry (const Entry &src);

	 /// Create an entity of the given type with a default empty value.
	 Entry (Type tp);

	 /**
	  * Create a null entry. Its type is set to \c UNKNOWN, and its
	  * value to null.
	  */
	 Entry () :
	    m_Type (UNKNOWN),
	    d_void (NULL)
	 {}

	 /// Destroy the given entry
	 ~Entry ();

	 /**
	  * Try to read a value to fill this entry from a lexer. If the type
	  * of the read value is not the same as the entity's one, the lexer
	  * remains unchanged.
	  * \return true on success, false on error
	  */
	 bool Read (Lexer &lexer);

	 /**
	  * Converts this entry to a string value. This might be used to dump
	  * the state of an entry list into a file.
	  */
	 String ToString () const;

	 /// Copy the contents of the source entry.
	 Entry &operator = (const Entry &src);
   };


   /**
    * An EntryDef is the definition of an entry, it contains its type, its
    * name, and a short description. It also stores a default value for this
    * entry. It is used by a DataDef to store a list of entries contained
    * in a class.
    */
   class ARK_DLL_API EntryDef
   {
      public:
	 /// Name of the corresponding entry.
	 String m_Name;

	 /// Short description of the entry.
	 String m_Desc;

	 /// Type
	 Entry::Type m_Type;

	 /**
	  * Default value : if a class instance doesn't specify a value for
	  * this entry, then the default value is used.
	  */
	 Entry m_DefVal;

      public:
	 /**
	  * Creates an entry definition with the given name, type and
	  * description. The type will be converted to an integer value
	  * (cf Entry::Type) before it is stored in the entry def.
	  */
	 EntryDef (const String &name,
		   const String &desc,
		   const String &type);

	 /// Creates an empty entry definition.
	 EntryDef () {}

	 /**
	  * Read the default value for this definition. It is normally called
	  * by the DataDef::Read function.
	  */
	 bool ReadDefault (Lexer &lexer);
   };

   /**
    * DataDef: List of entries with their default values, their type, etc..
    * This is basically a wrapper around a vector of entry definitions
    * (EntryDef)
    */
   class ARK_DLL_API DataDef
   {
      protected:
         /**
	  * ClassList can access our private members, to search for a
	  * specific entry in a class hierarchy for example.
	  */
	 friend class ClassList;

	 /// List of entries type.
	 typedef std::vector< EntryDef > EntryDefList;
	 /// List of entries iterator type.
	 typedef EntryDefList::iterator EDLIterator;

	 /// List of entries in this type of data.
	 EntryDefList m_Entries;

      public:
	 /// Creates an empty data definition.
	 DataDef ();

	 /// Destroy the given data def.
	 ~DataDef ();

	 /**
	  * Read a data definition from a lexer, this is a list of entry
	  * definition contained between a pair of brackets. The format of an
	  * entry def is quite straightforward:
	  * \code
	    TYPE name
	    {
	      desc = "a short text describing this entry";
	      default = a default value;
	    }
	    \endcode
	  */
	 bool Read (Lexer &lexer);

	 /**
	  * This function is for debug purposes essentially : it prints a
	  * list of the entries contained in this data type.
	  */
	 void Dump ();

	 /**
	  * Try to retrieve an entry definition with the given name.
	  * \return a pointer to the EntryDef in case of success, 0 if 
	  *         there is no entry with the given name.
	  */
	 EntryDef *GetEntryDef (const String &name)
	 {
	    for (EDLIterator i = m_Entries.begin(); i != m_Entries.end(); ++i)
	    {
	       if (i->m_Name == name)
		  return &(*i);
	    }

	    return NULL;
	 }
   };

   /**
    * ClassDef: DataDef with inheritance informations.
    */
   class ARK_DLL_API ClassDef : public DataDef
   {
      public:
	 /// Parent class of this class.
	 ClassDef *m_Parent;

	 /// Name of the parent of this class.
	 String m_ParentName;

	 /// Name of this class.
	 String m_Name;

      public:
	 /// Create a class with the given name, and superclass name.
	 ClassDef (const String &name, const String &parent = String());

	 /// Initialises an empty class definition.
	 ClassDef () {m_Parent = 0;}

	 /// Destroy this class definition.
	 ~ClassDef ();

	 /**
	  * Read the class definition from a lexer. The format is the same
	  * as the DataDef::Read one.
	  */
	 bool Read (Lexer &lexer);

	 /**
	  * Find an entry definition called \c name in this class and
	  * in its superclasses.
	  * \return 0 if there is no entry called \c name.
	  */
	 EntryDef *FindEntryDef (const String &name);
   };

   typedef std::map<String,Entry> EntryList;

   /**
    * A "ClassList" List of class definitions loaded from a file. This is
    * used by the Engine to store a list of the game classes, to simplify the
    * organization of the world's entities.
    *
    * It also store a list of template "entities" (ie class instances), in
    * order to make it easier to add common entities to a world. A 
    * template is a list of default values, which can be used as default
    * values by entities.
    */
   class ARK_DLL_API ClassList
   {
      public:
	 /// List of classes.
	  std::vector<ClassDef> m_Classes;

	 /// List of templates.
	  std::map<String, EntryList> m_Templates;

      public:

	 /// Creates an empty class list.
	 ClassList ();

	 /// Destroy this lists, and all the classes contained herein.
	 ~ClassList ();

	 /**
	  * Load a list of classes from a file.
	  * \return false on error.
	  */
	 bool Load (const String &file);

	 /**
	  * Load a list of templates from a file.
	  * \return false on error.
	  */
	 bool LoadTemplates (const String &file);

	 /** 
	  * For debug purposes : dump the list of classes and templates
	  * contained in this class list.
	  */
	 void Dump ();

	 /**
	  * Read a class instance from the given lexer, and fills the
	  * given list of values with the read values. Be aware that the
	  * entity declaration can use default or template values. Template
	  * values override default values, and are overriden by entity's
	  * values. 
	  */
	 bool Read (Lexer &lexer, EntryList *values);
	 
	 /// Find a class definition called \c name.
	 ClassDef *Find (const String &name);
   };

}

#endif
