/*
 * Copyright (c) 2000-2004 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * argus - IP Audit Record Generation and Utilization System
 *
 * written by Carter Bullard
 * QoSient LLC
 *
 */


#ifndef Argus
#define Argus
#endif

#include <ArgusModeler.h>
#include <ArgusSource.h>
#include <ArgusOutput.h>
#include <ArgusUtil.h>

void ArgusParseResourceFile (char *);


void
usage(void)
{
   extern char version[];

   fprintf (stderr, "Argus Version %s\n", version);
   fprintf (stderr, "usage: %s [options] [-i interface] [filter-expression] \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options]  -r packetfile [filter-expression] \n\n", ArgusProgramName);

   fprintf (stderr, "options: -b                   dump filter compiler output.\n");
   fprintf (stderr, "         -c                   direct argus to create a pid file.\n");
   fprintf (stderr, "         -d                   run Argus in daemon mode.\n");

#if defined(ARGUSDEBUG)
   fprintf (stderr, "         -D <level>           set debug reporting <level>.\n");
#endif

   fprintf (stderr, "         -e <value>           specify Argus Identifier <value>.\n");
   fprintf (stderr, "         -h                   print help.\n");
   fprintf (stderr, "         -F <conffile>        read configuration from <conffile>.\n");
   fprintf (stderr, "         -I <number>          specify <number> of instances of Argi allowed (1).\n");
   fprintf (stderr, "         -J                   generate packet performance data.\n");
   fprintf (stderr, "         -M <secs>            set MAR Status Report Time Interval (300s).\n");
   fprintf (stderr, "         -m                   turn off MAC Layer Reporting.\n");
   fprintf (stderr, "         -n                   specify pid filename.\n");
   fprintf (stderr, "         -O                   turn off filter optimizer.\n");
   fprintf (stderr, "         -p                   don't go into promiscuous mode.\n");
   fprintf (stderr, "         -P <portnum>         enable remote access on <portnum> (561).\n");
   fprintf (stderr, "         -R                   generate response time data.\n");
   fprintf (stderr, "         -S <secs>            set FAR Status Report Time Interval (60s).\n");
   fprintf (stderr, "         -t                   indicate that packetfile is MOAT Tsh format. \n");
   fprintf (stderr, "         -U <bytes>           specify the number of user bytes to capture.\n");
   fprintf (stderr, "         -w <file [\"filter\"]> write output to <file>, or '-', for stdout,\n");
   fprintf (stderr, "                              against optional filter expression.\n");
   fprintf (stderr, "         -X                   reset argus configuration.\n");
   exit (-1);
}


/*
 *  Argus main routine 
 *
 *  Argus main will:
 *       simply instantiate the source, modeler, and output tasks,
 *       parse out the command line options,
 *       initalize the tasks and then loop.
 *       Afterwards, it will delete all the tasks and exit();
 *
 */

#define ArgusEnvItems      2

char *ArgusResourceEnvStr [] = {
   "ARGUSHOME",
   "HOME",
};


#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>


char *ArgusPidFileName = NULL;
static char ArgusPidFileNameBuf[MAXPATHNAMELEN];
char * ArgusCreatePIDFile (char *, char *);
int ArgusMaxInstances = 1;


char *
ArgusCreatePIDFile (char *homepath, char *appname)
{
   FILE *fd;
   char pidstrbuf[128], *pidstr = pidstrbuf;
   char *retn = NULL, *dev;
   struct stat statbuf;
   int i, pid;

   for (i = 0; i < ArgusMaxInstances; i++) {
      if (ArgusMaxInstances > 1) {
         if ((appname != NULL) && ((dev = getArgusDevice()) != NULL)) {
            snprintf (ArgusPidFileNameBuf, MAXPATHNAMELEN - 1, "%s/%s.%s.%d.pid", homepath, appname, dev, i);
            ArgusPidFileName = ArgusPidFileNameBuf;
         }
      } else {
         if ((homepath != NULL) && (appname != NULL)) {
            snprintf (ArgusPidFileNameBuf, MAXPATHNAMELEN - 1, "%s/%s.pid", homepath, appname);
            ArgusPidFileName = ArgusPidFileNameBuf;
         } else
            if (ArgusPidFileName == NULL)
               ArgusLog (LOG_ERR, "ArgusCreatePIDFile parameter error\n");
      }
      retn = ArgusPidFileName;
      if ((stat (retn, &statbuf)) == 0) {
         if ((fd = fopen (retn, "r")) != NULL) {
            if ((pidstr = fgets (pidstrbuf, 128, fd)) != NULL) {
               if ((pid = strtol(pidstr, (char **)NULL, 10)) > 0) {
                  if (pid < LONG_MAX) {
                     if ((kill (pid, 0)) == 0) {
#ifdef ARGUSDEBUG
                        ArgusDebug (1, "ArgusCreatePIDFile(%s) pid %d is running\n", retn, pid);
#endif 
                        retn = NULL;
                     }
                  }
               }
            }

            fclose (fd);
         }
      }
 
      if (retn != NULL)
         break;
   }

   if (retn && ((fd = fopen (retn, "w+")) != NULL)) {
      pid = getpid();
      fprintf (fd, "%d\n", pid);
      fclose (fd);
   } else
      retn = NULL;

#ifdef ARGUSDEBUG
   if (retn)
      ArgusDebug (1, "ArgusCreatePIDFile(0x%x, 0x%x) returning %s\n", homepath, appname, retn);
   else
      ArgusDebug (1, "ArgusCreatePIDFile(0x%x, 0x%x) returning NULL\n", homepath, appname);
#endif 

   return (retn);
}


char *ArgusPidFile = NULL;
pid_t ArgusSessionId = 0;

int
main (int argc, char *argv[])
{
   int op;
   char *tmparg, *filter;
   static char path[MAXPATHNAMELEN];
   char *homepath = NULL;
   extern char *optarg;
   extern int optind, opterr;
   struct stat statbuf;
   struct hostent *host;
   int commandlinew = 0, doconf = 0;
   int commandlinei = 0;
   int  i, pid = 0;

   ArgusUid = getuid();

   gettimeofday (&ArgusStartTime, 0L);

   if (strchr (argv[0], '/')) {
      strcpy(path, argv[0]);
      argv[0] = strrchr(argv[0], '/') + 1;
   }

   ArgusProgramName = argv[0];
   fclose(stdin);

   setArgusFarReportInterval (ARGUS_FARSTATUSTIMER);
   setArgusMarReportInterval (ARGUS_MARSTATUSTIMER);

   for (i = 1; i < argc; i++) {
      char *ptr = argv[i];
      if (ptr != NULL) {
         if (*ptr == '-') {
            do {
               switch (*++ptr) {
                  case 'F':
                     ArgusParseResourceFile (argv[i++]);
                     doconf++;
                     break;

                  case 'X': clearArgusConfiguration (); break;
                  case 'h': usage (); break;
               }
            } while (isalpha((int)*++ptr));
         }
      }
   }

   if (!doconf) {
      snprintf (path, MAXPATHNAMELEN - 1, "/etc/argus.conf");
      if (stat (path, &statbuf) == 0) {
         ArgusParseResourceFile (path);
      }
   }

   opterr = 0; optind = 1; 
#if defined(__FreeBSD__)
   optreset = 1;
#endif

   while ((op = getopt (argc, argv, "bB:cdD:e:F:I:i:JmM:n:N:OP:pRr:S:tTU:w:Xh")) != EOF) {
      switch (op) {
         case 'b': setArgusbpflag (1); break;
         case 'B': ArgusBindIP = strdup(optarg); break;
         case 'c': setArguspidflag  (1); break;
         case 'd': daemonflag++; break;
         case 'D': setArgusdflag (atoi (optarg)); break;
         case 'e': 
            if (optarg && isalnum((int)*optarg)) {
              if ((host = gethostbyname(optarg)) != NULL) {
                 if ((host->h_addrtype == 2) && (host->h_length == 4)) {
                    unsigned int addr;
                    bcopy ((char *) *host->h_addr_list, (char *)&addr, host->h_length);
                    setArgusID (ntohl(addr));
                 } else
                    ArgusLog (LOG_ERR, "Probe ID %s error %s\n", optarg, strerror(errno));

                 setArgusIDType(ARGUS_ID_IS_IPADDR);

              } else
                  if (optarg && isdigit((int)*optarg)) {
                     setArgusID (atoi (optarg));
                  } else
                     ArgusLog (LOG_ERR, "Probe ID value %s is not appropriate (%s)\n", optarg, strerror(errno));
            } else
               ArgusLog (LOG_ERR, "Probe ID value %s is not appropriate\n", optarg);
            break;

         case 'F': break;
         case 'i': {
            if (!commandlinei++)
               clearArgusDevice();
            setArgusDevice (optarg);
            break;
         }

         case 'I': ArgusMaxInstances = atoi (optarg); break;
         case 'J': setArgusGenerateTime  (1); break;
         case 'm': setArgusmflag (1); break;
         case 'M': setArgusMarReportInterval (optarg); break;
         case 'n': 
            if (ArgusPidFileName != NULL)
               free(ArgusPidFileName);
            ArgusPidFileName = strdup(optarg);
            setArguspidflag  (1);
            ArgusMaxInstances = 1;
            break;

         case 'N': Nflag = (atoi (optarg)); break;
         case 'O': setArgusOflag  (0); break;
         case 'P': ArgusPortNum = atoi (optarg); break;
         case 'p': setArguspflag  (1); break;
         case 'R': setArgusResponseStatus (1); break;
         case 'r': setArgusrfile  (optarg); break;
         case 'S': setArgusFarReportInterval (optarg); break;
         case 't': setArgusMoatTshFile (1); break;
         case 'U': setArgusUserDataLen (atoi (optarg)); break;
 
         case 'w':
            if (!commandlinew++)
               clearArgusWfile();

            if ((tmparg = optarg) != NULL) {
               if ((*tmparg != '-') || ((*tmparg == '-') &&
                                       (!(strcmp (tmparg, "-"))))) {
                  if (argc == optind)
                     filter = NULL;
                  else {
                     filter = argv[optind];
                     if (*filter == '-') {
                        filter = NULL;
                     } else
                        optind++;
                     }
                  setArgusWfile (tmparg, filter);
                  break;
               }
            }

         case 'X': break;

         default:
         case 'h':
            usage ();
      }
   }

   if (!getArgusInfile ()) {
      setArgusArgv   (argv);
      setArgusOptind (optind);
   }

   if (getArgusrfile() != NULL) {
      setArgusPortNum(0, 0);
      daemonflag = 0;

   } else
      setArgusPortNum(ArgusPortNum, ArgusBindIP);

#if defined(HAVE_SOLARIS)
   sigignore(SIGPIPE);
#else
   (void) signal (SIGPIPE, SIG_IGN);
#endif

   (void) signal (SIGHUP,  (void (*)(int)) ArgusScheduleShutDown);
   (void) signal (SIGINT,  (void (*)(int)) ArgusScheduleShutDown);
   (void) signal (SIGTERM, (void (*)(int)) ArgusScheduleShutDown);
   (void) signal (SIGUSR1, (void (*)(int)) ArgusUsr1Sig);
   (void) signal (SIGUSR2, (void (*)(int)) ArgusUsr2Sig);

   if (daemonflag) {
      if (chdir ("/") < 0)
         ArgusLog (LOG_ERR, "Can't chdir to / %s", strerror(errno));

      if ((pid = fork ()) < 0) {
         ArgusLog (LOG_ERR, "Can't fork daemon %s", strerror(errno));
      } else {
         if (pid)
            exit (0);
         else
            ArgusSessionId = setsid();
      }

      ArgusLog(LOG_WARNING, "started");
   }

   if ((ArgusModel = ArgusNewModeler()) == NULL)
      ArgusLog (LOG_ERR, "Error Creating Modeler: Exiting.\n");

   if ((ArgusOutputTask = ArgusNewOutput()) == NULL)
      ArgusLog (LOG_ERR, "Error Creating Output Task: Exiting.\n");

   if ((ArgusSourceTask = ArgusNewSource()) == NULL)
      ArgusLog (LOG_ERR, "Error Creating Source Task: Exiting.\n");

   ArgusInitSource ();

   if (getArguspidflag()) {
      if (ArgusPidFileName != NULL)
         homepath = NULL;
      else
         if (stat ("/var/run", &statbuf) == 0)
            homepath = "/var/run";

      if ((ArgusPidFile = ArgusCreatePIDFile (homepath, argv[0])) == NULL)
         ArgusLog (LOG_ERR, "daemon cannot create new pid file");
   }

   ArgusInitOutput ();
   ArgusInitModeler();

   if (daemonflag) {
      FILE *tmpfile = NULL;

      if ((tmpfile = fopen ("/dev/null", "w")) != NULL) {
         fclose(stdout);
         *stdout = *tmpfile;
         
         if (!getArgusdflag()) {
            fclose(stderr);
            *stderr = *tmpfile;
         }
      } else
         ArgusLog (LOG_ERR, "Cannot map stdout to /dev/null\n");
   }

   ArgusLoop();

#ifdef ARGUSDEBUG
   ArgusDebug (1, "main() shuting down\n");
#endif 

   ArgusShutDown(0);
   exit(0);
}


void
ArgusComplete ()
{
#if defined(ARGUSPERFMETRICS)
   extern struct timeval ArgusStartTime, ArgusEndTime;
   long long ArgusTotalPkts = 0, ArgusTotalIPPkts = 0;
   long long ArgusTotalNonIPPkts = 0;
   struct timeval timediff;
   double totaltime;
   int i, len;
   char buf[256];
 
   bzero(buf, sizeof(buf));
   timediff.tv_sec  = ArgusEndTime.tv_sec  - ArgusStartTime.tv_sec;
   timediff.tv_usec = ArgusEndTime.tv_usec - ArgusStartTime.tv_usec;
 
   if (timediff.tv_usec < 0) {
      timediff.tv_usec += 1000000;
      timediff.tv_sec--;
   }
 
   totaltime = (double) timediff.tv_sec + (((double) timediff.tv_usec)/1000000.0);

   for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
      if (ArgusInterface[i].ArgusDevice != NULL) {
         ArgusTotalPkts      += ArgusInterface[i].ArgusTotalPkts;
         ArgusTotalIPPkts    += ArgusInterface[i].ArgusTotalIPPkts;
         ArgusTotalNonIPPkts += ArgusInterface[i].ArgusTotalNonIPPkts;
      }
   }

   len = strlen(ArgusProgramName);
   for (i = 0; i < len; i++)
      buf[i] = ' ';

   fprintf (stderr, "%s: Time %d.%06d Flows %8Ld  Closed %8Ld  Records %8Ld  Updates %8Ld\n",
                         ArgusProgramName, (int)timediff.tv_sec, (int)timediff.tv_usec,
                         ArgusTotalNewFlows,  ArgusTotalClosedFlows,
                         ArgusTotalRecords, ArgusTotalUpdates);

   fprintf (stderr, "    Total Pkts %8Ld  IP Pkts %8Ld  nonIP Pkts %8Ld   Rate %f\n",
                         ArgusTotalPkts, ArgusTotalIPPkts, ArgusTotalNonIPPkts,
                         ArgusTotalPkts/totaltime);
 
   for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
      if (ArgusInterface[i].ArgusDevice != NULL) {
         fprintf (stderr, "%s\n    Total Pkts %8Ld  IP Pkts %8Ld  nonIP Pkts %8Ld   Rate %f\n",
                         ArgusInterface[i].ArgusDevice, ArgusInterface[i].ArgusTotalPkts,
                         ArgusInterface[i].ArgusTotalIPPkts, ArgusInterface[i].ArgusTotalNonIPPkts,
                         ArgusInterface[i].ArgusTotalPkts/totaltime);
      }
   }

#endif
}


void
ArgusLoop ()
{
   ArgusGetPackets();

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusLoop() returning\n");
#endif 
}


char *ArgusSignalTable [] = { "Normal Shutdown",
"SIGHUP", "SIGINT", "SIGQUIT", "SIGILL", "SIGTRAP",
"SIGABRT", "SIGBUS", "SIGFPE", "SIGKILL", "SIGUSR1",
"SIGSEGV", "SIGUSR2", "SIGPIPE", "SIGALRM", "SIGTERM",
"SIGSTKFLT", "SIGCHLD", "SIGCONT", "SIGSTOP", "SIGTSTP",
"SIGTTIN", "SIGTTOU", "SIGURG", "SIGXCPU", "SIGXFSZ",
"SIGVTALRM", "SIGPROF", "SIGWINCH", "SIGIO",
};

int ArgusShutDownFlag = 0;

void
ArgusScheduleShutDown (int sig)
{
   ArgusShutDownFlag++;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusScheduleShutDown(%d)\n", sig);
#endif 
}

void
ArgusShutDown (int sig)
{
   ArgusShutDownFlag++;

   if (sig < 0) {
#ifdef ARGUSDEBUG
      ArgusDebug (1, "ArgusShutDown(ArgusError)\n");
#endif 
      _exit(0);
   }

#ifdef ARGUSDEBUG
   if (Argusdflag >= 1)
      fprintf(stderr, "\n");

   ArgusDebug (1, "ArgusShutDown(%s)\n\n", ArgusSignalTable[sig]);
#endif 

   if (!(ArgusShutDownStarted++)) {
      ArgusDeleteSource ();
      ArgusDeleteModeler();
      ArgusDeleteOutput ();
      ArgusComplete ();

   } else {
#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusShutDown() returning\n");
#endif 
      return;
   }

   if (ArgusPidFile)
      unlink (ArgusPidFile);

   if (daemonflag)
      ArgusLog(LOG_WARNING, "stopped");

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusShutDown()\n");
#endif 

   _exit(0);
}


void
setArguspidflag  (int value)
{
   pidflag = value;
}

int
getArguspidflag  ()
{
   return (pidflag);
}

#define ARGUS_RCITEMS				21

#define ARGUS_DAEMON				0
#define ARGUS_MONITOR_ID			1
#define ARGUS_ACCESS_PORT			2
#define ARGUS_INTERFACE				3
#define ARGUS_OUTPUT_FILE			4
#define ARGUS_SET_PID 				5
#define ARGUS_GO_PROMISCUOUS			6
#define ARGUS_FLOW_STATUS_INTERVAL		7
#define ARGUS_MAR_STATUS_INTERVAL		8
#define ARGUS_CAPTURE_DATA_LEN			9
#define ARGUS_GENERATE_START_RECORDS		10
#define ARGUS_GENERATE_RESPONSE_TIME_DATA	11
#define ARGUS_GENERATE_JITTER_DATA		12
#define ARGUS_GENERATE_MAC_DATA			13
#define ARGUS_DEBUG_LEVEL			14
#define ARGUS_FILTER_OPTIMIZER			15
#define ARGUS_FILTER				16
#define ARGUS_PACKET_CAPTURE_FILE		17
#define ARGUS_BIND_IP				18
#define ARGUS_MAX_INSTANCES			19
#define ARGUS_PID_FILENAME			20

char *ArgusResourceFileStr [] = {
   "ARGUS_DAEMON=",
   "ARGUS_MONITOR_ID=",
   "ARGUS_ACCESS_PORT=",
   "ARGUS_INTERFACE=",
   "ARGUS_OUTPUT_FILE=",
   "ARGUS_SET_PID=",
   "ARGUS_GO_PROMISCUOUS=",
   "ARGUS_FLOW_STATUS_INTERVAL=",
   "ARGUS_MAR_STATUS_INTERVAL=",
   "ARGUS_CAPTURE_DATA_LEN=",
   "ARGUS_GENERATE_START_RECORDS=",
   "ARGUS_GENERATE_RESPONSE_TIME_DATA=",
   "ARGUS_GENERATE_JITTER_DATA=",
   "ARGUS_GENERATE_MAC_DATA=",
   "ARGUS_DEBUG_LEVEL=",
   "ARGUS_FILTER_OPTIMIZER=",
   "ARGUS_FILTER=",
   "ARGUS_PACKET_CAPTURE_FILE=",
   "ARGUS_BIND_IP=",
   "ARGUS_MAX_INSTANCES=",
   "ARGUS_PID_FILENAME=",
};



extern pcap_dumper_t *ArgusPcapOutFile;
extern char *ArgusWriteOutPacketFile;

#define MAXSTRLEN 2048

void
ArgusParseResourceFile (char *file)
{
   FILE *fd;
   char strbuf[MAXSTRLEN], *str = strbuf, *optarg;
   char result[MAXSTRLEN], *ptr;
   int i, len, done = 0, linenum = 0;
   int interfaces = 0, outputfiles = 0;
   struct hostent *host;

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         while ((fgets(str, MAXSTRLEN, fd)) != NULL)  {
            done = 0;
            linenum++;
            while (*str && isspace((int)*str))
                str++;

            if (*str && (*str != '#') && (*str != '\n') && (*str != '!')) {
               for (i = 0; i < ARGUS_RCITEMS && !done; i++) {
                  len = strlen(ArgusResourceFileStr[i]);
                  if (!(strncmp (str, ArgusResourceFileStr[i], len))) {
                     optarg = &str[len];
                     if (*optarg == '\"') optarg++;
                     if (optarg[strlen(optarg) - 1] == '\n')
                        optarg[strlen(optarg) - 1] = '\0';
                     if (optarg[strlen(optarg) - 1] == '\"')
                        optarg[strlen(optarg) - 1] = '\0';

                     switch (i) {
                        case ARGUS_DAEMON: 
                           if (!(strncasecmp(optarg, "yes", 3)))
                              daemonflag = 1;
                           else
                              daemonflag = 0;
                           break;

                        case ARGUS_MONITOR_ID: 
                           if (optarg && (*optarg == '`')) {
                              if (optarg[strlen(optarg) - 1] == '`') {
                                 FILE *fd;

                                 optarg++;
                                 optarg[strlen(optarg) - 1] = '\0';
                                 if (!(strcmp (optarg, "hostname"))) {
                                    if ((fd = popen("hostname", "r")) != NULL) {
                                       if ((ptr = fgets(result, MAXSTRLEN, fd)) != NULL) {
                                          optarg = ptr;
                                          optarg[strlen(optarg) - 1] = '\0';
                                       } else
                                          ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) `hostname` failed %s.\n", file, strerror(errno));

                                       pclose(fd);
                                    } else
                                       ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) System error: popen() %s\n", file, strerror(errno));
                                 } else
                                    ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) unsupported command `%s` at line %d.\n", file, optarg, linenum);
                              } else
                                 ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) syntax error line %d\n", file, linenum);
                           }
                           if (optarg && isalnum((int)*optarg)) {
                              if ((host = gethostbyname(optarg)) != NULL) {
                                 if ((host->h_addrtype == 2) && (host->h_length == 4)) {
                                    unsigned int addr;
                                    bcopy ((char *) *host->h_addr_list, (char *)&addr, host->h_length);
                                    setArgusID (ntohl(addr));
                                 } else
                                    ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) host '%s' error %s\n", file, optarg, strerror(errno));
               
                                 setArgusIDType(ARGUS_ID_IS_IPADDR);
               
                              } else
                                 if (optarg && isdigit((int)*optarg)) {
                                    setArgusID (atoi (optarg));
                                 } else
                                    ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) syntax error line %d\n", file, linenum);
                           } else
                              ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) syntax error line %d\n", file, linenum);

                           break;
                           
                        case ARGUS_ACCESS_PORT:
                           ArgusPortNum = atoi(optarg);
                           break;

                        case ARGUS_BIND_IP:
                           if (ArgusBindIP != NULL)
                              free(ArgusBindIP);
                           ArgusBindIP = strdup(optarg);
                           break;

                        case ARGUS_MAX_INSTANCES:
                           ArgusMaxInstances = atoi(optarg);
                           break;

                        case ARGUS_PID_FILENAME:
                           if (ArgusPidFileName != NULL)
                              free(ArgusPidFileName);
                           ArgusPidFileName = strdup(optarg);
                           break;

                        case ARGUS_OUTPUT_FILE:
                           ptr = NULL;
                           if ((ptr = strchr (optarg, '"')) != NULL)
                              *ptr = '\0';

                           if ((ptr = strchr (optarg, ' ')) != NULL) {
                              *ptr++ = '\0';
                           }
                   
                           if (outputfiles++ == 0)
                              clearArgusWfile ();
                           setArgusWfile (optarg, ptr);
                           break;

                        case ARGUS_INTERFACE:
                           if (interfaces++ == 0)
                              clearArgusDevice ();
                           setArgusDevice (optarg);
                           break;

                        case ARGUS_SET_PID:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArguspidflag  (1);
                           else
                              setArguspidflag  (0);
                           break;

                        case ARGUS_GO_PROMISCUOUS:
                           if ((strncasecmp(optarg, "yes", 3)))
                              setArguspflag  (1);
                           else
                              setArguspflag  (0);
                           break;

                        case ARGUS_FLOW_STATUS_INTERVAL:
                           setArgusFarReportInterval (optarg);
                           break;

                        case ARGUS_MAR_STATUS_INTERVAL:
                           setArgusMarReportInterval (optarg);
                           break;

                        case ARGUS_CAPTURE_DATA_LEN:
                           setArgusUserDataLen (atoi(optarg));
                           break;

                        case ARGUS_GENERATE_START_RECORDS:
/*
                           if ((strncasecmp(optarg, "yes", 3)))
                              setArguspflag  (1);
*/
                           break;

                        case ARGUS_GENERATE_RESPONSE_TIME_DATA:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusResponseStatus  (1);
                           else
                              setArgusResponseStatus  (0);
                           break;

                        case ARGUS_GENERATE_JITTER_DATA:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusGenerateTime  (1);
                           else
                              setArgusGenerateTime  (0);
                           break;

                        case ARGUS_GENERATE_MAC_DATA:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusmflag (1);
                           else
                              setArgusmflag (0);
                           break;

                        case ARGUS_DEBUG_LEVEL:
                           setArgusdflag (atoi(optarg));
                           break;
                        
                        case ARGUS_FILTER_OPTIMIZER:
                           if ((strncasecmp(optarg, "yes", 3)))
                              setArgusOflag  (1);
                           else
                              setArgusOflag  (0);
                           break;

                        case ARGUS_FILTER:
                           if ((ArgusInputFilter = ArgusCalloc (1, MAXSTRLEN)) != NULL) {
                              ptr = ArgusInputFilter;
                              str = optarg;
                              while (*str) {
                                 if ((*str == '\\') && (str[1] == '\n')) {
                                    fgets(str, MAXSTRLEN, fd);
                                    while (*str && (isspace((int)*str) && (str[1] && isspace((int)str[1]))))
                                       str++;
                                 }
                                 
                                 if ((*str != '\n') && (*str != '"'))
                                    *ptr++ = *str++;
                                 else
                                    str++;
                              }
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusParseResourceFile: ArgusFilter \"%s\" \n", ArgusInputFilter);
#endif 
                           }
                           break;

                        case ARGUS_PACKET_CAPTURE_FILE:
                           if (*optarg != '\0')
                              setArgusWriteOutPacketFile (optarg);
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusParseResourceFile: ArgusPacketCaptureFile \"%s\" \n", ArgusWriteOutPacketFile);
#endif 
                           break;
                     }

                     done = 1;
                     break;
                  }
               }
            }
         }
      } else {
#ifdef ARGUSDEBUG
         ArgusLog (LOG_ERR, "ArgusParseResourceFile: open %s %s\n", file, strerror(errno));
#endif 
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusParseResourceFile (%s) returning\n", file);
#endif 

}

void
clearArgusConfiguration ()
{
   daemonflag = 0;
   setArgusID (0);
   setArgusIDType (0);
   ArgusPortNum = 0;
   if (ArgusBindIP != NULL) {
      free(ArgusBindIP);
      ArgusBindIP = NULL;
   }
   clearArgusWfile ();
   clearArgusDevice ();
   setArguspidflag  (1);
   setArguspflag  (1);
   setArgusFarReportInterval (ARGUS_FARSTATUSTIMER);
   setArgusMarReportInterval (ARGUS_MARSTATUSTIMER);
   setArgusUserDataLen (0);
   setArgusResponseStatus (0);
   setArgusGenerateTime (0);
   setArgusmflag (0);
   setArgusdflag (0);
   setArgusOflag (1);
 
   if (ArgusWriteOutPacketFile) {
      if (ArgusWriteOutPacketFile) {
         if (ArgusPcapOutFile != NULL) {
            pcap_dump_close(ArgusPcapOutFile);
            ArgusPcapOutFile = NULL;
         }
         ArgusWriteOutPacketFile = NULL;
      }
   }

   if (ArgusInputFilter) {
      ArgusFree(ArgusInputFilter);
      ArgusInputFilter = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "clearArgusConfiguration () returning\n");
#endif 

}
