/*
    Copyright (C) 1999-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: session_process.cc,v 1.27 2004/02/29 23:33:56 pauld Exp $
*/

#include <cmath>
#include <cerrno>
#include <unistd.h>

#include <ardour/timestamps.h>

#include <pbd/error.h>
#include <pbd/lockmonitor.h>

#include <ardour/ardour.h>
#include <ardour/session.h>
#include <ardour/diskstream.h>
#include <ardour/audioengine.h>
#include <ardour/slave.h>
#include <ardour/auditioner.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

void
Session::process (jack_nframes_t nframes)
{
	if (slave_source() == JACK) {
		if (jack_slave_sync (nframes)) {
			/* what? */
		}
	}

	if (non_realtime_work_pending()) {
		if (atomic_read (&butler_should_do_transport_work) == 0) {
			post_transport ();
		} 
	} 
	
	(this->*process_function) (nframes);
}

void
Session::prepare_diskstreams ()
{
	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
		(*i)->prepare ();
	}
}

int
Session::no_roll (jack_nframes_t nframes, jack_nframes_t offset)
{
	jack_nframes_t end_frame = _transport_frame + nframes;
	int ret = 0;

	if (_click_io) {
		_click_io->silence (nframes, offset);
	}

	/* XXX we're supposed to have the route_lock while doing this.
	   this is really bad ...
	*/

	if (atomic_read (&processing_prohibited)) {
		for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
			(*i)->silence (nframes, offset);
		}
		return 0;
	}

	for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
		
		if ((*i)->hidden()) {
			continue;
		}
		
		if ((*i)->no_roll (nframes, end_frame, offset, non_realtime_work_pending(), 
				   actively_recording(), get_transport_declick_required())) {
			error << compose(_("Session: error in no roll for %1"), (*i)->name()) << endmsg;
			ret = -1;
			break;
		}
	}

	return ret;
}

int
Session::process_routes (jack_nframes_t nframes, jack_nframes_t offset)
{
	bool record_active = actively_recording();
	int  declick = get_transport_declick_required();
	bool rec_monitors = get_rec_monitors_input();

	if (transport_sub_state & StopPendingCapture) {
		/* force a declick out */
		declick = 1;
	}

	for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {

		int ret;

		if ((*i)->hidden()) {
			continue;
		}

		if ((ret = (*i)->roll (nframes, _transport_frame + nframes, offset, declick, record_active, rec_monitors)) < 0) {

			error << compose(_("Session: error for %1 at frame %2 (%3)"), (*i)->name(), _transport_frame, ret) << endmsg;

			/* we have to do this here. Route::roll() for an AudioTrack will have called DiskStream::process(),
			   and the DS will expect DiskStream::commit() to be called. but we're aborting from that
			   call path, so make sure we release any outstanding locks here before we return failure.
			*/

			for (DiskStreamList::iterator ids = diskstreams.begin(); ids != diskstreams.end(); ++ids) {
				(*ids)->recover ();
			}

			stop_transport ();
			return -1;
		} 
	}

	return 0;
}

void
Session::commit_diskstreams (jack_nframes_t nframes, bool &needs_butler)
{
	int dret;

	for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {

		if ((*i)->hidden()) {
			continue;
		}
		
		/* force all diskstreams not handled by a Route to call do their stuff.
		 */

		if ((dret = (*i)->process (nframes, 0, actively_recording(), get_rec_monitors_input())) == 0) {
			if ((*i)->commit (nframes)) {
				needs_butler = true;
			}

		} else if (dret < 0) {
			(*i)->recover();
		}

	}
	
	if (actively_recording()) {
		set_dirty();
	}
}

void
Session::process_scrub (jack_nframes_t nframes)
{
	bool who_cares_about_the_butler_here_anyway;
	long frames_moved;
	float dss;

	/* if we get here without either of the auditioner or the click port, 
	   we're in deep trouble. it ain't gonna happen. why? because to get
	   into scrub mode requires user intervention, and thus the engine
	   is already running. if its already running, we've created both
	   objects by the time the user gets a chance to ask to scrub.
	   
	   all this to save to conditionals, eh?
	*/

	auditioner->silence (nframes, 0);
	_click_io->silence (nframes, 0);

	{
		TentativeLockMonitor rm (route_lock, __LINE__, __FILE__);

		if (!rm.locked()) {
			no_roll (nframes, 0);
			return;
		}

		dss = _desired_transport_speed;

		if (dss != _transport_speed) {
			if ((_transport_speed >= 0.0 && dss < 0.0) || (_transport_speed < 0.0 && dss >= 0.0)) {

				_transport_speed = dss;

				for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
					if (!(*i)->hidden()) {
						(*i)->reverse_scrub_buffer (_transport_speed >= 0.0);
					}
				}

			} else {
				/* apply hysteresis */
				_transport_speed = ((0.8 * _transport_speed) + (0.2 * dss));

			}

			bool non_rt_required = false;

			for (DiskStreamList::iterator i = diskstreams.begin(); i != diskstreams.end(); ++i) {
				if ((*i)->realtime_set_speed ((*i)->speed())) {
					non_rt_required = true;
				}
			}

			if (non_rt_required) {
				post_transport_work = PostTransportWork (post_transport_work | PostTransportSpeed);
				schedule_butler_transport_work ();
			}
		}

		/* OK, just do it */

		if ((post_transport_work  & PostTransportLocate)|| _transport_speed == 0.0) {

			no_roll (nframes, 0);

		} else {

			frames_moved = (long) floor (_transport_speed * nframes);

			prepare_diskstreams ();

			if (process_routes (nframes, 0)) {
				no_roll (nframes, 0);
				return;
			}
			
			commit_diskstreams (nframes, who_cares_about_the_butler_here_anyway);

			if (frames_moved < 0) {
				decrement_transport_position (-frames_moved);
			} else {
				increment_transport_position (frames_moved);
			}

			if (_transport_frame == max_frames || _transport_frame == 0) {
				stop_transport ();
			}
		}

	} /* implicit release of route lock */
	
	return;
}		

void
Session::process_with_events (jack_nframes_t nframes)
{
	Event* ev;
	jack_nframes_t this_nframes;
	jack_nframes_t end_frame;
	jack_nframes_t offset;
	bool session_needs_butler = false;
	jack_nframes_t stop_limit;
	long           frames_moved;

	if (auditioner) {
		auditioner->silence (nframes, 0);
	}

	while (pending_events.read (&ev, 1) == 1) {
		merge_event (ev);
	}

	/* if we are not in the middle of a state change,
	   and there are immediate events queued up,
	   process them. 
	*/

	while (!non_realtime_work_pending() && !immediate_events.empty()) {
		Event *ev = immediate_events.front ();
		immediate_events.pop_front ();
		process_event (ev);
	}

	if (!process_can_proceed()) {
		no_roll (nframes, 0);
		return;
	}
		
	if (events.empty() || next_event == events.end()) {
		process_without_events (nframes);
		return;
	}

	end_frame = _transport_frame + nframes;

	{
		TentativeLockMonitor rm (route_lock, __LINE__, __FILE__);
		Event* this_event;
		Events::iterator the_next_one;

		if (!rm.locked() || _transport_speed == 0 || (post_transport_work & PostTransportLocate)) {
			no_roll (nframes, 0);
			return;
		}

		if (actively_recording()) {
			stop_limit = max_frames;
		} else {

			if (Config->get_stop_at_session_end()) {
				stop_limit = current_end;
			} else {
				stop_limit = max_frames;
			}
		}

		if (maybe_stop (stop_limit)) {
			no_roll (nframes, 0);
			return;
		} 

		this_event = *next_event;
		the_next_one = next_event;
		++the_next_one;

		offset = 0;

		while (nframes) {

			if (this_event == 0 || this_event->action_frame > end_frame || this_event->action_frame < _transport_frame) {

				this_nframes = nframes;
				
			} else {
				
				/* compute nframes to next event */
				
				if (this_event->action_frame < end_frame) {
					this_nframes = nframes - (end_frame - this_event->action_frame);
				} else {
					this_nframes = nframes;
				}
			}

			if (_slave) {

				if (!_slave->ok()) {
					stop_transport ();
					set_slave_source (None, 0);
					no_roll (nframes, 0);
					return;
				}
				
				if (!_slave->locked()) {
					no_roll (nframes, 0);
					return;
				}
				
				jack_nframes_t slave_transport_frame = _slave->current_audio_frame();
				float delta;
				int dir;
				
				/* OK, control theory time. 
				   
				   If the gap between where we are and the slave is is too
				   large, then non-linear motion is needed to get there.
				   
				   If its small enough, then we aim to catch up to 
				   current position within a certain time. 
				*/
				
				if (slave_transport_frame > _transport_frame) {
					delta = slave_transport_frame - _transport_frame;
					dir = 1;
				} else {
					delta = _transport_frame - slave_transport_frame;
					dir = -1;
				}
				
				cerr << "slave at " << slave_transport_frame
				     << " us at " << _transport_frame
				     << " from delta = " << delta
				     << endl;
				
				if (delta > 2 * nframes) {
					
					cerr << "non-linear slave motion - jump to " << slave_transport_frame;
					/* aim for 50msecs ahead of the current slave position */
					slave_transport_frame += (jack_nframes_t) floor (0.050 * _current_frame_rate);
					cerr << " adjusted to " << slave_transport_frame << endl;
					locate (slave_transport_frame, true, false);
					no_roll (nframes, 0);
					return;
				}
				
				if (dir > 0) {
					set_transport_speed (_transport_speed + 0.0000001 * delta);
				} else {
					set_transport_speed (_transport_speed - 0.0000001 * delta);
				}
				
				if (non_realtime_work_pending()) {
					no_roll (nframes, 0);
					return;
				}
				
				cerr << " slaved adjust speed to " << _transport_speed << endl;
			}
/* */		

			if (this_nframes) {
				
				/* now process frames between now and the first event in this block */
				
				prepare_diskstreams ();
				
				if (process_routes (this_nframes, offset)) {
					no_roll (nframes, 0);
					return;
				}
				
				commit_diskstreams (this_nframes, session_needs_butler);
				
				if (_transport_speed == 1.0f) {
					click (_transport_frame, nframes, offset);
				}
				
				nframes -= this_nframes;
				offset += this_nframes;
				
				frames_moved = (jack_nframes_t) floor (_transport_speed * this_nframes);
			
				if (frames_moved < 0) {
					decrement_transport_position (-frames_moved);
				} else {
					increment_transport_position (frames_moved);
				}

				maybe_stop (stop_limit);
				check_declick_out ();
			}

			/* now handle this event and all others scheduled for the same time */
			
			while (this_event && this_event->action_frame == _transport_frame) {
				
				process_event (this_event);
				
				if (the_next_one == events.end()) {
					this_event = 0;
				} else {
					this_event = *the_next_one;
					++the_next_one;
				}
			} 

			/* if an event left our state changing, do the right thing */

			if (non_realtime_work_pending()) {
				no_roll (nframes, offset);
				break;
			}

		}

		set_next_event ();

	} /* implicit release of route lock */

	if (session_needs_butler) {
		summon_butler ();
	} 
	
	if (!_engine.freewheeling()) {
		send_midi_time_code_in_another_thread ();
	}

	return;
}		

void
Session::process_without_events (jack_nframes_t nframes)
{
	bool session_needs_butler = false;
	jack_nframes_t stop_limit;
	long frames_moved;
	
	{
		TentativeLockMonitor rm (route_lock, __LINE__, __FILE__);

		if (!rm.locked() || (_transport_speed == 0) || (post_transport_work & PostTransportLocate)) {
			no_roll (nframes, 0);
			return;
		}
		
		if (actively_recording()) {
			stop_limit = max_frames;
		} else {
			if (Config->get_stop_at_session_end()) {
				stop_limit = current_end;
			} else {
				stop_limit = max_frames;
			}
		}
		
		if (maybe_stop (stop_limit)) {
			no_roll (nframes, 0);
			return;
		} 

		prepare_diskstreams ();
	
		frames_moved = (long) floor (_transport_speed * nframes);

		if (process_routes (nframes, 0)) {
			no_roll (nframes, 0);
			return;
		}

		commit_diskstreams (nframes, session_needs_butler);

		if (_transport_speed == 1.0f) {
			click (_transport_frame, nframes, 0);
		}

		if (frames_moved < 0) {
			decrement_transport_position (-frames_moved);
		} else {
			increment_transport_position (frames_moved);
		}

		maybe_stop (stop_limit);
		check_declick_out ();

	} /* implicit release of route lock */

	if (session_needs_butler) {
		summon_butler ();
	} 
	
	if (!_engine.freewheeling()) {
		send_midi_time_code_in_another_thread ();
	}

	return;
}		

void
Session::process_audition (jack_nframes_t nframes)
{
	TentativeLockMonitor rm (route_lock, __LINE__, __FILE__);

	if (rm.locked()) {
		for (RouteList::iterator i = routes.begin(); i != routes.end(); ++i) {
			if (!(*i)->hidden()) {
				(*i)->silence (nframes, 0);
			}
		}
	}
	
	if (auditioner->play_audition (nframes) > 0) {
		summon_butler ();
	} 

	if (!auditioner->active()) {
		process_function = &Session::process_with_events;
	}
}

