/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: route.cc,v 1.107 2004/02/29 23:33:56 pauld Exp $
*/

#include <cmath>
#include <fstream>

#include <sigc++/bind.h>
#include <pbd/xml++.h>

#include <ardour/timestamps.h>
#include <ardour/audioengine.h>
#include <ardour/route.h>
#include <ardour/insert.h>
#include <ardour/send.h>
#include <ardour/session.h>
#include <ardour/utils.h>
#include <ardour/configuration.h>
#include <ardour/cycle_timer.h>
#include <ardour/route_group.h>
#include <ardour/port.h>
#include <ardour/ladspa_plugin.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;


unsigned long Route::order_key_cnt = 0;

Route::Route (Session& sess, const string &name, int input_min, int input_max, int output_min, int output_max, Flag flg)
	: IO (sess, name, input_min, input_max, output_min, output_max),
	  _flags (flg)
{
	init ();
}

Route::Route (Session& sess, const XMLNode& node)
	: IO (sess, "route")
{
	init ();
	set_state (node);
}

void
Route::init ()
{
	_muted = false;
	_soloed = false;
	_solo_safe = false;
	_phase_invert = false;
	order_keys[N_("signal")] = order_key_cnt++;
	_active = true;
	_silent = false;
	_meter_pre_fader = false;
	_initial_delay = 0;
	_roll_delay = 0;
	_own_latency = 0;

	_edit_group = 0;
	_mix_group = 0;

	_mute_affects_pre_fader = Config->get_mute_affects_pre_fader();
	_mute_affects_post_fader = Config->get_mute_affects_post_fader();
	_mute_affects_control_outs = Config->get_mute_affects_control_outs();
	_mute_affects_main_outs = Config->get_mute_affects_main_outs();

	solo_gain = 1.0;
	desired_solo_gain = 1.0;
	mute_gain = 1.0;
	desired_mute_gain = 1.0;

	_control_outs = 0;

	input_configuration_changed.connect (slot (this, &Route::input_change_handler));
	output_configuration_changed.connect (slot (this, &Route::output_change_handler));

	reset_midi_control (_session.mmc_port(), _session.get_mmc_control());
}

Route::~Route ()
{
	 GoingAway (); /* EMIT SIGNAL */
	clear_redirects (this);

	if (_control_outs) {
		delete _control_outs;
	}
}

long
Route::order_key (string name) const
{
	OrderKeys::const_iterator i;
	
	if ((i = order_keys.find (name)) == order_keys.end()) {
		return -1;
	}

	return (*i).second;
}

void
Route::set_order_key (string name, long n)
{
	order_keys[name] = n;
	_session.set_dirty ();
}

void
Route::inc_gain (gain_t delta, void *src)
{
	IO::inc_gain (delta, src);
}

void
Route::set_gain (gain_t val, void *src)
{
	if (_mix_group && src != _mix_group && _mix_group->is_active()) {

		if (_mix_group->is_relative()) {
			gain_t delta = val - gain();
			_mix_group->apply (&Route::inc_gain, delta, _mix_group);
		} else {
			_mix_group->apply (&Route::set_gain, val, _mix_group);
		}

		return;
	} 

	if (val == gain()) {
		return;
	}

	IO::set_gain (val, src);
}

void
Route::process_output_buffers (vector<Sample*>& bufs, jack_nframes_t nframes, jack_nframes_t offset, bool with_redirects, int declick)
{
	unsigned long n;
	RedirectList::iterator i;
	bool post_fader_work = false;
	bool mute_declick_applied = false;
	gain_t dmg, dsg, dg;
	vector<Sample*>::iterator bufiter;
	IO *co;
	bool mute_audible;
	bool solo_audible;
	bool no_monitor = (Config->get_use_hardware_monitoring() || Config->get_no_sw_monitoring ());

	{
		TentativeLockMonitor cm (control_outs_lock, __LINE__, __FILE__);
		
		if (cm.locked()) {
			co = _control_outs;
		} else {
			co = 0;
		}
	}
	
	{ 
		TentativeLockMonitor dm (declick_lock, __LINE__, __FILE__);
		
		if (dm.locked()) {
			dmg = desired_mute_gain;
			dsg = desired_solo_gain;
			dg = _desired_gain;
		} else {
			dmg = mute_gain;
			dsg = solo_gain;
			dg = _gain;
		}
	}

	/* ----------------------------------------------------------------------------------------------------
	   PRE-FADER METERING etc.
	   -------------------------------------------------------------------------------------------------- */

	if (_meter_pre_fader || record_enabled()) {

		for (n = 0, bufiter = bufs.begin(); bufiter != bufs.end(); ++bufiter, ++n) {
			compute_peak (*bufiter, n, nframes, _peak_power[n], 1.0);
		}
	}

	if (!_soloed && _mute_affects_pre_fader && (mute_gain != dmg)) {
		apply_declick (bufs, nframes, mute_gain, dmg, _phase_invert);
		mute_gain = dmg;
		mute_declick_applied = true;
	}


	if (_meter_pre_fader && co) {
		
		solo_audible = dsg > 0;
		mute_audible = dmg > 0 || !_mute_affects_pre_fader;
		
		if ( // silent anyway
				
			_gain == 0 || 
			
			// muted by solo of another track
			
			!solo_audible || 
			
			// muted by mute of this track 
			
			!mute_audible ||
			
			// recording but not s/w monitoring 
			
			(_session.actively_recording() && record_enabled() && no_monitor)) {
			
			_control_outs->silence (nframes, offset);
			
		} else {
			
			_control_outs->deliver_output (bufs, nframes, offset);
			
		} 
	} 

	/* ----------------------------------------------------------------------------------------------------
	   PRE-FADER REDIRECTS
	   -------------------------------------------------------------------------------------------------- */

	if (with_redirects) {
		TentativeLockMonitor rm (redirect_lock, __LINE__, __FILE__);
		if (rm.locked()) {
			if (mute_gain > 0 || !_mute_affects_pre_fader) {
				for (i = _redirects.begin(); i != _redirects.end(); ++i) {
					switch ((*i)->placement()) {
					case Redirect::PreFader:
						(*i)->run (bufs, nframes, offset);
						break;
					case Redirect::PostFader:
						post_fader_work = true;
						break;
					}
				}
			} else {
				for (i = _redirects.begin(); i != _redirects.end(); ++i) {
					switch ((*i)->placement()) {
					case Redirect::PreFader:
						(*i)->silence (nframes, offset);
						break;
					case Redirect::PostFader:
						post_fader_work = true;
						break;
					}
				}
			}
		} 
	}


	if (!_soloed && (mute_gain != dmg) && !mute_declick_applied && _mute_affects_post_fader) {
		apply_declick (bufs, nframes, mute_gain, dmg, _phase_invert);
		mute_gain = dmg;
		mute_declick_applied = true;
	}

	/* ----------------------------------------------------------------------------------------------------
	   GAIN STAGE
	   -------------------------------------------------------------------------------------------------- */

	if (!record_enabled()) {

		if (apply_gain_automation) {
			
			if (_phase_invert) {
				for (bufiter = bufs.begin(); bufiter != bufs.end(); ++bufiter) {
					Sample *sp = *bufiter;
					
					for (jack_nframes_t nx = 0; nx < nframes; ++nx) {
						sp[nx] *= -gain_automation_buffer[nx];
					}
				}
			} else {
				for (bufiter = bufs.begin(); bufiter != bufs.end(); ++bufiter) {
					Sample *sp = *bufiter;
					
					for (jack_nframes_t nx = 0; nx < nframes; ++nx) {
						sp[nx] *= gain_automation_buffer[nx];
					}
				}
			}
			
			_effective_gain = gain_automation_buffer[nframes-1];
			
			
		} else {
			
			if (_gain != dg) {
				
				apply_declick (bufs, nframes, _gain, dg, _phase_invert);
				_gain = dg;
				
			} else if (_gain != 1.0 && _gain != 0.0) {
				
				/* no need to interpolate current gain value,
				   but its non-unity, so apply it. if the gain
				   is zero, do nothing because we'll ship silence
				   below.
				*/
				
				gain_t this_gain;
				
				if (_phase_invert) {
					this_gain = -_gain;
				} else {
					this_gain = _gain;
				}
				
				for (bufiter = bufs.begin(); bufiter != bufs.end(); ++bufiter) {
					Sample *sp = *bufiter;
					for (jack_nframes_t nx = 0; nx < nframes; ++nx) {
						sp[nx] *= this_gain;
					}
				}
			}
		}
	}

	/* ----------------------------------------------------------------------------------------------------
	   POST-FADER REDIRECTS
	   -------------------------------------------------------------------------------------------------- */

	if (post_fader_work && with_redirects) {

		TentativeLockMonitor rm (redirect_lock, __LINE__, __FILE__);
		if (rm.locked()) {
			if (mute_gain > 0 || !_mute_affects_post_fader) {
				for (i = _redirects.begin(); i != _redirects.end(); ++i) {
					switch ((*i)->placement()) {
					case Redirect::PreFader:
						break;
					case Redirect::PostFader:
						(*i)->run (bufs, nframes, offset);
						break;
					}
				}
			} else {
				for (i = _redirects.begin(); i != _redirects.end(); ++i) {
					switch ((*i)->placement()) {
					case Redirect::PreFader:
						break;
					case Redirect::PostFader:
						(*i)->silence (nframes, offset);
						break;
					}
				}
			}
		} 
	}

	/* ----------------------------------------------------------------------------------------------------
	   POST-FADER METERING
	   -------------------------------------------------------------------------------------------------- */

	if (!_meter_pre_fader && !record_enabled()) {

		if (_gain == 0 || dmg == 0) {
			for (n = 0; n < n_outputs(); ++n) {
				_peak_power[n] = 0;
			} 
		} else {
			for (n = 0, bufiter = bufs.begin(); bufiter != bufs.end(); ++bufiter, ++n) {
				compute_peak (*bufiter, n, nframes, _peak_power[n], _pans[n]);
			}
		}
	}

	if (!_soloed && (mute_gain != dmg) && !mute_declick_applied && _mute_affects_control_outs) {
		apply_declick (bufs, nframes, mute_gain, dmg, _phase_invert);
		mute_gain = dmg;
		mute_declick_applied = true;
	}
	
	/* ----------------------------------------------------------------------------------------------------
	   GLOBAL DECLICK (for transport changes etc.)
	   -------------------------------------------------------------------------------------------------- */

	if (declick > 0) {
		apply_declick (bufs, nframes, 0.0, 1.0, _phase_invert);
	} else if (declick < 0) {
		apply_declick (bufs, nframes, 1.0, 0.0, _phase_invert);
	} else {

		/* no global declick */

		if (solo_gain != dsg) {
			apply_declick (bufs, nframes, solo_gain, dsg, _phase_invert);
			solo_gain = dsg;
		}
	}

	/* ----------------------------------------------------------------------------------------------------
	   CONTROL OUTPUT STAGE
	   -------------------------------------------------------------------------------------------------- */

	if (!_meter_pre_fader && co) {
		
		solo_audible = solo_gain > 0;
		mute_audible = dmg > 0 || !_mute_affects_control_outs;
		
		if ( // silent anyway

		    _gain == 0 || 

                     // muted by solo of another track

		    !solo_audible || 
		    
                     // muted by mute of this track 

		    !mute_audible ||

		    // recording but not s/w monitoring 

		    (_session.actively_recording() && record_enabled() && no_monitor)) {

			_control_outs->silence (nframes, offset);

		} else {

			_control_outs->deliver_output (bufs, nframes, offset);
		} 
	} 

	/* ----------------------------------------------------------------------
	   GLOBAL MUTE 
	   ----------------------------------------------------------------------*/

	if (!_soloed && (mute_gain != dmg) && !mute_declick_applied && _mute_affects_main_outs) {
		apply_declick (bufs, nframes, mute_gain, dmg, _phase_invert);
		mute_gain = dmg;
		mute_declick_applied = true;
	}
	
	/* ----------------------------------------------------------------------------------------------------
	   MAIN OUTPUT STAGE
	   -------------------------------------------------------------------------------------------------- */

	solo_audible = dsg > 0;
	mute_audible = dmg > 0 || !_mute_affects_main_outs;
	
	if (n_outputs() == 0) {
	    
	    /* relax */

	} else if (record_enabled() && no_monitor) {
		
		IO::silence (nframes, offset);
		
	} else {
		
		if ( // silent anyway

		    _gain == 0 ||
		    
		    // muted by solo of another track, but not using control outs for solo

		    (!solo_audible && (_session.solo_model() != Session::SoloBus)) ||
		    
		    // muted by mute of this track

		    !mute_audible) {

			/* don't use Route::silence() here, because that causes
			   all outputs (sends, port inserts, etc. to be silent).
			*/
			
			if (!_meter_pre_fader) {
				reset_peak_meters ();
			}

			IO::silence (nframes, offset);
		    
		    } else {
			    

			if (apply_pan_automation) {
				pan (bufs, nframes, offset, pan_automation_buffer);
				_effective_stereo_pan = pan_automation_buffer[nframes-1];
			} else {
				pan (bufs, nframes, offset, 0);
				_effective_stereo_pan = pans().front();
			}
		} 
	}
}

void
Route::passthru (jack_nframes_t nframes, jack_nframes_t offset, int declick)
{
	vector<Sample*> bufs;

	_silent = false;
	
	get_process_buffers (bufs, nframes, offset);
	collect_input (bufs, nframes, offset);
	process_output_buffers (bufs, nframes, offset, true, declick);
}

void
Route::set_phase_invert (bool yn, void *src)
{
	if (_phase_invert != yn) {
		_phase_invert = yn;
	}
	//  phase_invert_changed (src); /* EMIT SIGNAL */
}

void
Route::set_solo (bool yn, void *src)
{
	if (_solo_safe) {
		return;
	}

	if (_mix_group && src != _mix_group && _mix_group->is_active()) {
		_mix_group->apply (&Route::set_solo, yn, _mix_group);
		return;
	}

	if (_soloed != yn) {
		_soloed = yn;
		 solo_changed (src); /* EMIT SIGNAL */
	}
}

void
Route::set_solo_mute (bool yn)
{
	LockMonitor lm (declick_lock, __LINE__, __FILE__);

	/* Called by Session in response to another Route being soloed.
	 */
	   
	desired_solo_gain = (yn?0.0:1.0);
}

void
Route::set_solo_safe (bool yn, void *src)
{
	if (_solo_safe != yn) {
		_solo_safe = yn;
		 solo_safe_changed (src); /* EMIT SIGNAL */
	}
}

void
Route::set_mute (bool yn, void *src)

{
	if (_mix_group && src != _mix_group && _mix_group->is_active()) {
		_mix_group->apply (&Route::set_mute, yn, _mix_group);
		return;
	}

	if (_muted != yn) {
		_muted = yn;
		 mute_changed (src); /* EMIT SIGNAL */

		LockMonitor lm (declick_lock, __LINE__, __FILE__);
		desired_mute_gain = (yn?0.0:1.0);
	}
}

void
Route::add_redirect (Redirect *redirect, void *src)

{
	{
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		_redirects.push_back (redirect);
		redirect->activate ();
	}

	 redirects_changed (src); /* EMIT SIGNAL */
}

void
Route::copy_redirects (const Route& other, Redirect::Placement placement)
{
	{
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		RedirectList::iterator tmp;

		/* remove all relevant redirects */

		for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ) {
			tmp = i;
			++tmp;
			if ((*i)->placement() == placement) {
				delete *i;
				_redirects.erase (i);
				i = tmp;
			}
		}

		/* now copy the relevant ones from "other" */
		
		for (RedirectList::const_iterator i = other._redirects.begin(); i != other._redirects.end(); ++i) {
			if ((*i)->placement() == placement) {
				_redirects.push_back (Redirect::clone (**i));
			}
		}
	}

	 redirects_changed (this); /* EMIT SIGNAL */
}

void
Route::all_redirects_flip ()
{
	LockMonitor lm (redirect_lock, __LINE__, __FILE__);

	if (_redirects.empty()) {
		return;
	}

	bool first_is_on = _redirects.front()->active();
	
	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
		(*i)->set_active (!first_is_on, this);
	}
}

void
Route::all_redirects_active (bool state)
{
	LockMonitor lm (redirect_lock, __LINE__, __FILE__);

	if (_redirects.empty()) {
		return;
	}

	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
		(*i)->set_active (state, this);
	}
}

void
Route::clear_redirects (void *src)

{
	{
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);

		for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
			delete *i;
		}

		_redirects.clear ();
	}

	 redirects_changed (src); /* EMIT SIGNAL */
}

void
Route::remove_redirect (Redirect *redirect, void *src)

{
	{
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		
		for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
			if (*i == redirect) {
				_redirects.erase (i);
				break;
			}
		}
	}

	 redirects_changed (src); /* EMIT SIGNAL */
}

void
Route::set_block_size (jack_nframes_t nframes)
{
	if (_automation_playback_bits & GainAutomation) {
		if (gain_automation_buffer) {
			delete [] gain_automation_buffer;
		}
		gain_automation_buffer = new gain_t[nframes];
	}

	if (_automation_playback_bits & PanAutomation) {
		if (pan_automation_buffer) {
			delete [] pan_automation_buffer;
		}
		pan_automation_buffer = new pan_t[nframes];
	}
}

XMLNode&
Route::get_state()
{
	return state(true);
}

XMLNode&
Route::get_template()
{
	return state(false);
}

XMLNode&
Route::state(bool full_state)
{
	XMLNode *node = new XMLNode("Route");
	XMLNode *aevents;
	RedirectList:: iterator i;
	char buf[32];

	if (_flags) {
		snprintf (buf, sizeof (buf), "0x%x", _flags);
		node->add_property("flags", buf);
	}
	node->add_property("muted", _muted?"yes":"no");
	node->add_property("soloed", _soloed?"yes":"no");
	node->add_property("phase-invert", _phase_invert?"yes":"no");
	node->add_property("mute-affects-pre-fader", _mute_affects_pre_fader?"yes":"no"); 
	node->add_property("mute-affects-post-fader", _mute_affects_post_fader?"yes":"no"); 
	node->add_property("mute-affects-control-outs", _mute_affects_control_outs?"yes":"no"); 
	node->add_property("mute-affects-main-outs", _mute_affects_main_outs?"yes":"no"); 

	if (_edit_group) {
		node->add_property("edit-group", _edit_group->name());
	}
	if (_mix_group) {
		node->add_property("mix-group", _mix_group->name());
	}

	string order_string;
	OrderKeys::iterator x = order_keys.begin(); 

	while (x != order_keys.end()) {
		order_string += (*x).first;
		order_string += '=';
		snprintf (buf, sizeof(buf), "%ld", (*x).second);
		order_string += buf;
		
		++x;

		if (x == order_keys.end()) {
			break;
		}

		order_string += ':';
	}
	node->add_property ("order-keys", order_string);

	node->add_child_nocopy (IO::get_state());

	if (_control_outs) {
		XMLNode* cnode = new XMLNode (X_("ControlOuts"));
		cnode->add_child_nocopy (_control_outs->get_state());
		node->add_child_nocopy (*cnode);
	}

	if (_comment.length()) {
		XMLNode *cmt = node->add_child ("Comment");
		cmt->add_content (_comment);
	}

	if(full_state){
		string path;

		path += legalize_for_path (_name);
		path += ".automation";

		/* XXX we didn't ask for a state save, we asked for the current state.
		   FIX ME!
		*/

		if (save_automation (path)) {
			error << _("Could not get state of route.  Problem with save_automation") << endmsg;
		}

		aevents = node->add_child ("Automation");
		aevents->add_property ("path", path);
	}	

	for (i = _redirects.begin(); i != _redirects.end(); ++i) {
		node->add_child_nocopy((*i)->get_state());
	}

	if (_extra_xml){
		node->add_child_copy (*_extra_xml);
	}
	
	return *node;
}

int
Route::set_state (const XMLNode& node)
{
	XMLNodeList nlist;
	XMLNodeConstIterator niter;
	XMLNode *child;
	XMLPropertyList plist;
	const XMLProperty *prop;
	Insert *insert = 0;
	Send *send;

	if (node.name() != "Route"){
		error << compose(_("Bad node sent to Route::set_state() [%1]"), node.name()) << endmsg;
		return -1;
	}

	if ((prop = node.property ("flags")) != 0) {
		int x;
		sscanf (prop->value().c_str(), "0x%x", &x);
		_flags = Flag (x);
	} else {
		_flags = Flag (0);
	}

	if ((prop = node.property ("phase-invert")) != 0) {
		set_phase_invert(prop->value()=="yes"?true:false, this);
	}

	if ((prop = node.property ("muted")) != 0) {
		bool yn = prop->value()=="yes"?true:false; 

		/* force reset of mute status */

		_muted = !yn;
		set_mute(yn, this);
		mute_gain = desired_mute_gain;
	}

	if ((prop = node.property ("soloed")) != 0) {
		bool yn = prop->value()=="yes"?true:false; 

		/* force reset of solo status */

		_soloed = !yn;
		set_solo (yn, this);
		solo_gain = desired_solo_gain;
	}

	if ((prop = node.property ("mute-affects-pre-fader")) != 0) {
		_mute_affects_pre_fader = (prop->value()=="yes")?true:false;
	}

	if ((prop = node.property ("mute-affects-post-fader")) != 0) {
		_mute_affects_post_fader = (prop->value()=="yes")?true:false;
	}

	if ((prop = node.property ("mute-affects-control-outs")) != 0) {
		_mute_affects_control_outs = (prop->value()=="yes")?true:false;
	}

	if ((prop = node.property ("mute-affects-main-outs")) != 0) {
		_mute_affects_main_outs = (prop->value()=="yes")?true:false;
	}

	if ((prop = node.property ("edit-group")) != 0) {
		RouteGroup* edit_group = _session.edit_group_by_name(prop->value());
		if(edit_group == 0) {
			error << compose(_("Route %1: unknown edit group \"%2 in saved state (ignored)"), _name, prop->value()) << endmsg;
		} else {
			set_edit_group(edit_group, this);
		}
	}

	if ((prop = node.property ("order-keys")) != 0) {

		long n;

		string::size_type colon, equal;
		string remaining = prop->value();

		while (remaining.length()) {

			if ((equal = remaining.find_first_of ('=')) == string::npos || equal == remaining.length()) {
				error << compose (_("badly formed order key string in state file! [%1] ... ignored."), remaining)
				      << endmsg;
			} else {
				if (sscanf (remaining.substr (equal+1).c_str(), "%ld", &n) != 1) {
					error << compose (_("badly formed order key string in state file! [%1] ... ignored."), remaining)
					      << endmsg;
				} else {
					set_order_key (remaining.substr (0, equal), n);
				}
			}

			colon = remaining.find_first_of (':');

			if (colon != string::npos) {
				remaining = remaining.substr (colon+1);
			} else {
				break;
			}
		}
	}

	nlist = node.children();
	
	for (niter = nlist.begin(); niter != nlist.end(); ++niter){

		child = *niter;
			
		if (child->name() == "Send") {

			try {
				send = new Send (_session, *child);
			} 
			
			catch (failed_constructor &err) {
				error << _("Send construction failed") << endmsg;
				continue;
			}
			
			add_redirect (send, this);

		} else if (child->name() == "Insert") {
			
			XMLProperty *prop;

			try {
				if ((prop = child->property ("type")) != 0) {
					if (prop->value() == "ladspa") {
						insert = new PluginInsert(_session, *child);
					} else if (prop->value() == "port") {
						insert = new PortInsert (_session, *child);
					} else {
						error << compose(_("unknown Insert type \"%1\"; ignored"), prop->value()) << endmsg;
					}
					
					add_redirect (insert, this);
					
				} else {
					error << _("Insert XML node has no type property") << endmsg;
				}
			}
			
			catch (failed_constructor &err) {
				warning << _("insert could not be created. Ignored.") << endmsg;
				continue;
			}
			

		} else if (child->name() == "Automation-Events" || child->name() == "Automation") {

			XMLPropertyList plist;
			XMLPropertyConstIterator piter;
			XMLProperty *prop;
			
			plist = child->properties();
			for (piter = plist.begin(); piter != plist.end(); ++piter) {
				prop = *piter;
				if (prop->name() == "path") {
					load_automation (prop->value());
				}
			}

		} else if (child->name() == IO::state_node_name) {

			IO::set_state (*child);

		} else if (child->name() == "ControlOuts") {
			
			string coutname = _name;
			coutname += _("[control]");

			_control_outs = new IO (_session, coutname);
			_control_outs->set_state (**(child->children().begin()));

		} else if (child->name() == "Comment") {

			/* XXX this is a terrible API design in libxml++ */

			XMLNode *cmt = *(child->children().begin());
			_comment = cmt->content();

		} else if (child->name() == "extra") {
			_extra_xml = new XMLNode (*child);
		}
	}

	if ((prop = node.property ("mix-group")) != 0) {
		RouteGroup* mix_group = _session.mix_group_by_name(prop->value());
		if (mix_group == 0) {
			error << compose(_("Route %1: unknown mix group \"%2 in saved state (ignored)"), _name, prop->value()) << endmsg;
		}  else {
			set_mix_group(mix_group, this);
		}
	}

	return 0;
}

void
Route::curve_reallocate ()
{
//	_gain_automation_curve.finish_resize ();
//	_pan_automation_curve.finish_resize ();
}

void
Route::silence (jack_nframes_t nframes, jack_nframes_t offset)
{

	if (!_silent || offset != 0) {

		// reset_peak_meters ();
		
		IO::silence (nframes, offset);

		if (_control_outs) {
			_control_outs->silence (nframes, offset);
		}

		{ 
			TentativeLockMonitor lm (redirect_lock, __LINE__, __FILE__);
			
			if (lm.locked()) {
				for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
					(*i)->silence (nframes, offset);
				}
				_silent = true;
			}
		}
		
	}
}	

int
Route::set_control_outs (const vector<string>& ports)
{
	LockMonitor lm (control_outs_lock, __LINE__, __FILE__);
 	unsigned long n;
	vector<string>::const_iterator i;

 	if (_control_outs) {
 		delete _control_outs;
 		_control_outs = 0;
 	}
 	
 	if (ports.empty()) {
 		return 0;
 	}
 
 	string coutname = _name;
 	coutname += _("[control]");
 	
 	_control_outs = new IO (_session, coutname);
 	
	/* create as many control output ports as we have regular outputs,
	   and connect them to the specified ports.

	   XXX what happens when our regular output port count changes?
	*/

 	for (i = ports.begin(), n = 0; n < n_outputs() && i != ports.end(); ++i, ++n) {
 		_control_outs->add_output_port (*i, this);
 	}
 
 	return 0;
}	

void
Route::set_edit_group (RouteGroup *eg, void *src)

{
	if (_edit_group) {
		_edit_group->remove (this);
	}

	if ((_edit_group = eg)) {
		_edit_group->add (this);
	}

	 edit_group_changed (src); /* EMIT SIGNAL */
}

void
Route::set_mix_group (RouteGroup *mg, void *src)

{
	if (_mix_group) {
		_mix_group->remove (this);
	}

	if ((_mix_group = mg)) {
		_mix_group->add (this);
	}

	 mix_group_changed (src); /* EMIT SIGNAL */
}

void
Route::set_comment (const string& cmt, void *src)
{
	_comment = cmt;
	comment_changed (src);
}

bool
Route::feeds (Route *o)
{
	unsigned long i, j;

	/* input and output IO objects are virtualized */

	IO& other = o->input_io();
	IO& self = output_io();

	for (i = 0; i < self.n_outputs(); ++i) {
		for (j = 0; j < other.n_inputs(); ++j) {
			if (self.output(i)->connected_to (other.input(j)->name())) {
				return true;
			}
		}
	}

	/* check Redirects which may also interconnect Routes */

	for (RedirectList::iterator r = _redirects.begin(); r != _redirects.end(); r++) {
		for (i = 0; i < (*r)->n_outputs(); ++i) {
			for (j = 0; j < other.n_inputs(); ++j) {
				if ((*r)->output(i)->connected_to (other.input (j)->name())) {
					return true;
				}
			}
		}
	}

	/* check for control room outputs which may also interconnect Routes */

	if (_control_outs) {
		for (i = 0; i < _control_outs->n_outputs(); ++i) {
			for (j = 0; j < other.n_inputs(); ++j) {
				if (_control_outs->output(i)->connected_to (other.input (j)->name())) {
					return true;
				}
			}
		}
	}

	return false;
}

void
Route::set_mute_config (mute_type t, bool onoff, void *src)
{
	switch (t) {
	case PRE_FADER:
		_mute_affects_pre_fader = onoff;
		 pre_fader_changed(src); /* EMIT SIGNAL */
		break;

	case POST_FADER:
		_mute_affects_post_fader = onoff;
		 post_fader_changed(src); /* EMIT SIGNAL */
		break;

	case CONTROL_OUTS:
		_mute_affects_control_outs = onoff;
		 control_outs_changed(src); /* EMIT SIGNAL */
		break;

	case MAIN_OUTS:
		_mute_affects_main_outs = onoff;
		 main_outs_changed(src); /* EMIT SIGNAL */
		break;
	}
}

bool
Route::get_mute_config (mute_type t)
{
	bool onoff = false;
	
	switch (t){
	case PRE_FADER:
		onoff = _mute_affects_pre_fader; 
		break;
	case POST_FADER:
		onoff = _mute_affects_post_fader;
		break;
	case CONTROL_OUTS:
		onoff = _mute_affects_control_outs;
		break;
	case MAIN_OUTS:
		onoff = _mute_affects_main_outs;
		break;
	}
	
	return onoff;
}

void
Route::set_active (bool yn)
{
	_active = yn; 
	 active_changed(); /* EMIT SIGNAL */
}

void
Route::transport_stopped (bool abort_ignored, bool did_locate, bool can_flush_redirects)
{
	if (Config->get_plugins_stop_with_transport() && can_flush_redirects) {

		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		
		for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
			(*i)->deactivate ();
			(*i)->activate ();
		}
	}

	_roll_delay = _initial_delay;

	if (!did_locate) {
		automation_snapshot (_session.transport_frame());
	}

	_gain_automation_curve.reposition_for_rt_add (_session.transport_frame());
	_pan_automation_curve.reposition_for_rt_add (_session.transport_frame());
}

UndoAction
Route::get_memento() const
{
	void (Route::*pmf)(state_id_t) = &Route::set_state;
	return bind (slot (*(const_cast<Route *>(this)), pmf), _current_state_id);
}

void
Route::set_state (state_id_t id)
{
	return;
}

struct RedirectSorter {
    bool operator() (const Redirect *a, const Redirect *b) {
	    return a->sort_key() < b->sort_key();
    }
};

void
Route::sort_redirects ()
{
	{
		RedirectSorter comparator;
		LockMonitor lm (redirect_lock, __LINE__, __FILE__);
		_redirects.sort (comparator);
	}

	 redirects_changed (this); /* EMIT SIGNAL */
}

void
Route::reset_plugin_counts ()
{
	PluginInsert *pi;
	
	/* reset the count on any plugins we have */
	LockMonitor lm (redirect_lock, __LINE__, __FILE__);
	
	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {

		if ((pi  = dynamic_cast<PluginInsert*>(*i)) != 0) {
			unsigned int cnt;
			unsigned int rmaxports = max(input_io().n_inputs(), output_io().n_outputs());
			unsigned int pmaxports = max(pi->plugin().get_info().n_inputs, pi->plugin().get_info().n_outputs);
			
			if (rmaxports <= pmaxports) {
				/* always use at least 1 */
				cnt = 1;
			}
			else {
				/* only replicate plugin such that all ports may be used */
				cnt = rmaxports / pmaxports;
			}

			if (cnt != pi->get_count()) {

				pi->set_count(cnt);
			}
		}
	}
}

void
Route::input_change_handler (void *ignored)
{
	reset_plugin_counts();
}

void
Route::output_change_handler (void *ignored)
{
	reset_plugin_counts();
}

int 
Route::no_roll (jack_nframes_t nframes, jack_nframes_t end_frame, jack_nframes_t offset, 
		   bool session_state_changing, bool can_record, bool rec_monitors_input)
{
	if (n_outputs() == 0) {
		return 0;
	}

	if (session_state_changing || !_active)  {
		silence (nframes, offset);
		return 0;
	}
	
	if (n_inputs()) {
		passthru (nframes, offset, 0);
	} else {
		silence (nframes, offset);
	}

	return 0;
}

jack_nframes_t
Route::check_initial_delay (jack_nframes_t nframes, jack_nframes_t& offset)
{
	if (_roll_delay > nframes) {

		_roll_delay -= nframes;
		silence (nframes, offset);
		return 0;

	} else if (_roll_delay > 0) {

		nframes -= _roll_delay;
		offset += _roll_delay;

		silence (_roll_delay, offset);

		_roll_delay = 0;

		/* do the rest as usual*/
	} 

	return nframes;
}

int
Route::roll (jack_nframes_t nframes, jack_nframes_t end_frame, jack_nframes_t offset, int declick,
		bool can_record, bool rec_monitors_input)
{
	automation_snapshot (_session.transport_frame());

	if ((n_outputs() == 0 && _redirects.empty()) || n_inputs() == 0 || !_active) {
		silence (nframes, offset);
		return 0;
	}
	
	if ((nframes = check_initial_delay (nframes, offset)) == 0) {
		return 0;
	}

	_silent = false;
	apply_gain_automation = false;
	apply_pan_automation = false;
	
	{ 
		TentativeLockMonitor am (automation_lock, __LINE__, __FILE__);
		
		if (am.locked()) {
			
			jack_nframes_t start_frame = end_frame - nframes;
			
			if (_automation_playback && (_automation_playback_bits & GainAutomation)) {
				apply_gain_automation = _gain_automation_curve.rt_safe_get_vector (start_frame, end_frame, gain_automation_buffer, nframes);
			}
			
			if (_automation_playback && (_automation_playback_bits & PanAutomation)) {
				apply_pan_automation = _pan_automation_curve.rt_safe_get_vector (start_frame, end_frame, pan_automation_buffer, nframes);
			}
		}
	}

	passthru (nframes, offset, declick);

	return 0;
}

void
Route::toggle_monitor_input ()
{
	for (vector<Port *>::iterator i = _inputs.begin(); i != _inputs.end(); ++i) {
		(*i)->request_monitor_input (!(*i)->monitoring_input());
	}
}

bool
Route::has_external_redirects () const
{
	const PortInsert* pi;
	
	for (RedirectList::const_iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
		if ((pi = dynamic_cast<const PortInsert*>(*i)) != 0) {

			for (unsigned long n = 0; n < pi->n_outputs(); ++n) {
				
				string port_name = pi->output(n)->name();
				string client_name = port_name.substr (0, port_name.find(':'));

				/* only say "yes" if the redirect is actually in use */
				
				if (client_name != "ardour" && pi->active()) {
					return true;
				}
			}
		}
	}

	return false;
}

void
Route::get_process_buffers (vector<Sample*>& bufs, jack_nframes_t nframes, jack_nframes_t offset)
{
	vector<Port*>::iterator p;
	unsigned long limit = max (n_inputs(), n_outputs());
	unsigned long i = 0;

	/* if we have some outputs, then we just use the buffers of the output ports
	   themselves. don't try this if we have sends, since they might
	   try to scribble all over the buffers when they do a gain click
	   adjustment. XXX use a more precise test for sends.
	*/

	if (_redirects.empty()) { 
		for (p = _outputs.begin(); p != _outputs.end(); ++p, ++i) {
			bufs.push_back ((*p)->get_buffer (nframes) + offset);
			(*p)->mark_silence (false);
		}
	}
	
	for (unsigned long n = 0; i < limit; ++i, ++n) {
		bufs.push_back (_session.get_passthru_buffers()[n]);
	}
}

void
Route::reset_midi_control (MIDI::Port* port, bool on)
{
	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
			(*i)->reset_midi_control (port, on);
	}
	
	IO::reset_midi_control (port, on);
}

void
Route::flush_redirects ()
{
	/* XXX shouldn't really try to take this lock, since
	   this is called from the RT audio thread.
	*/

	LockMonitor lm (redirect_lock, __LINE__, __FILE__);

	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
		(*i)->deactivate ();
		(*i)->activate ();
	}
}

void
Route::set_meter_pre_fader (bool yn, void *src)
{
	if (_meter_pre_fader != yn) {
		_meter_pre_fader = yn; 
		 meter_change (src); /* EMIT SIGNAL */
	}
}

jack_nframes_t
Route::update_total_latency ()
{
	_own_latency = 0;

	for (RedirectList::iterator i = _redirects.begin(); i != _redirects.end(); ++i) {
		if ((*i)->active ()) {
			_own_latency += (*i)->latency ();
		}
	}

	set_port_latency (_own_latency);

	/* this (virtual) function is used for pure Routes,
	   not derived classes like AudioTrack.  this means
	   that the data processed here comes from an input
	   port, not prerecorded material, and therefore we
	   have to take into account any input latency.
	*/

	_own_latency += input_latency ();

	return _own_latency;
}

void
Route::set_latency_delay (jack_nframes_t longest_session_latency)
{
	_initial_delay = longest_session_latency - _own_latency;
}
