/*
    Copyright (C) 2000-2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: audioregion.cc,v 1.15 2004/02/29 23:33:56 pauld Exp $
*/

#include <cmath>
#include <climits>

#include <set>

#include <sigc++/bind.h>
#include <sigc++/class_slot.h>

#include <pbd/basename.h>
#include <pbd/lockmonitor.h>
#include <pbd/xml++.h>

#include <ardour/audioregion.h>
#include <ardour/session.h>
#include <ardour/gain.h>
#include <ardour/dB.h>
#include <ardour/playlist.h>
#include <ardour/audiofilter.h>

#include "i18n.h"
#include <locale.h>

using namespace ARDOUR;

/* a Session will reset these to its chosen defaults by calling AudioRegion::set_default_fade() */

gain_t  *AudioRegion::default_fade_in_data = 0;
gain_t  *AudioRegion::default_fade_out_data = 0;
LogCurve AudioRegion::default_fade_in;
LogCurve AudioRegion::default_fade_out;

Change AudioRegion::FadeInChanged = ARDOUR::new_change();
Change AudioRegion::FadeOutChanged = ARDOUR::new_change();
Change AudioRegion::FadeInActiveChanged = ARDOUR::new_change();
Change AudioRegion::FadeOutActiveChanged = ARDOUR::new_change();
Change AudioRegion::EnvelopeActiveChanged = ARDOUR::new_change();

AudioRegion::AudioRegion (Source& src, jack_nframes_t start, jack_nframes_t length, bool announce)
	: Region (start, length, basename_nosuffix(src.name()), 0,  Region::Flag(Region::DefaultFlags|Region::External))
{
	/* basic AudioRegion constructor */

	sources.push_back (&src);
	master_sources.push_back (&src);
	src.GoingAway.connect (slot (*this, &AudioRegion::source_deleted));

	_fade_in_data = 0;
	_fade_out_data = 0;

	set_default_envelope ();
	save_state ("initial state");

	if (announce) {
		 RegionCreated (this); /* EMIT SIGNAL */
	}
}

AudioRegion::AudioRegion (Source& src, jack_nframes_t start, jack_nframes_t length, const string& name, layer_t layer, Flag flags, bool announce)
	: Region (start, length, name, layer, flags)
{
	/* basic AudioRegion constructor */

	sources.push_back (&src);
	master_sources.push_back (&src);
	src.GoingAway.connect (slot (*this, &AudioRegion::source_deleted));

	_fade_in_data = 0;
	_fade_out_data = 0;

	set_default_envelope ();
	save_state ("initial state");

	if (announce) {
		 RegionCreated (this); /* EMIT SIGNAL */
	}
}

AudioRegion::AudioRegion (SourceList& srcs, jack_nframes_t start, jack_nframes_t length, const string& name, layer_t layer, Flag flags, bool announce)
	: Region (start, length, name, layer, flags)
{
	/* basic AudioRegion constructor */

	for (SourceList::iterator i=srcs.begin(); i != srcs.end(); ++i) {
		sources.push_back (*i);
		master_sources.push_back (*i);

		(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
	}

	_fade_in_data = 0;
	_fade_out_data = 0;

	set_default_envelope ();
	save_state ("initial state");
	if (announce) {
		 RegionCreated (this); /* EMIT SIGNAL */
	}
}


AudioRegion::AudioRegion (const AudioRegion& other, jack_nframes_t offset, jack_nframes_t length, const string& name, layer_t layer, Flag flags, bool announce)
	: Region (other, offset, length, name, layer, flags)
{
	/* create a new AudioRegion, that is part of an existing one */
	
	set<Source*> unique_srcs;

	for (SourceList::const_iterator i= other.sources.begin(); i != other.sources.end(); ++i) {
		sources.push_back (*i);
		(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		unique_srcs.insert (*i);
	}

	for (SourceList::const_iterator i = other.master_sources.begin(); i != other.master_sources.end(); ++i) {
		if (unique_srcs.find (*i) == unique_srcs.end()) {
			(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		}
		master_sources.push_back (*i);
	}

	_fade_in_data = 0;
	_fade_out_data = 0;
	_fade_in = other._fade_in;
	_fade_out = other._fade_out;
	build_fade (&_fade_in_data, _fade_in, true);
	build_fade (&_fade_out_data, _fade_out, false);

	set_default_envelope (); /* XXX should really take part of the curve of `other' */
	save_state ("initial state");

	if (announce) {
		RegionCreated (this); /* EMIT SIGNAL */
	}
}

AudioRegion::AudioRegion (const AudioRegion &other, const string& name)
	: Region (other, name)
{
	/* create a new AudioRegion, with a new name, by copying */

	set<Source*> unique_srcs;

	for (SourceList::const_iterator i = other.sources.begin(); i != other.sources.end(); ++i) {
		sources.push_back (*i);
		(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		unique_srcs.insert (*i);
	}

	for (SourceList::const_iterator i = other.master_sources.begin(); i != other.master_sources.end(); ++i) {
		master_sources.push_back (*i);
		if (unique_srcs.find (*i) == unique_srcs.end()) {
			(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		}
	}

	_fade_in_data = 0;
	_fade_out_data = 0;
	_fade_in = other._fade_in;
	_fade_out = other._fade_out;
	build_fade (&_fade_in_data, _fade_in, true);
	build_fade (&_fade_out_data, _fade_out, false); 

	_envelope = other._envelope;

	save_state ("initial state");
	 RegionCreated (this); /* EMIT SIGNAL */
}

AudioRegion::AudioRegion (const AudioRegion &other)
	: Region (other)
{
	/* Pure copy constructor */

	set<Source*> unique_srcs;

	for (SourceList::const_iterator i = other.sources.begin(); i != other.sources.end(); ++i) {
		sources.push_back (*i);
		(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		unique_srcs.insert (*i);
	}

	for (SourceList::const_iterator i = other.master_sources.begin(); i != other.master_sources.end(); ++i) {
		master_sources.push_back (*i);
		if (unique_srcs.find (*i) == unique_srcs.end()) {
			(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		}
	}

	_fade_in_data = 0;
	_fade_out_data = 0;
	_fade_in = other._fade_in;
	_fade_out = other._fade_out;
	build_fade (&_fade_in_data, _fade_in, true);
	build_fade (&_fade_out_data, _fade_out, false);

	_envelope = other._envelope;

	save_state ("initial state");
	/* NOTE: no RegionCreated signal emitted here. This is the copy constructor */
}

AudioRegion::AudioRegion (Source& src, const XMLNode& node)
	: Region (node)
{
	sources.push_back (&src);
	master_sources.push_back (&src);
	src.GoingAway.connect (slot (*this, &AudioRegion::source_deleted));

	build_fade (&_fade_in_data, _fade_in, true);
	build_fade (&_fade_out_data, _fade_out, false);

	if (set_state (node)) {
		throw failed_constructor();
	}
	
	save_state ("initial state");
	 RegionCreated (this); /* EMIT SIGNAL */
}

AudioRegion::AudioRegion (SourceList& srcs, const XMLNode& node)
	: Region (node)
{
	/* basic AudioRegion constructor */

	set<Source*> unique_srcs;

	for (SourceList::iterator i=srcs.begin(); i != srcs.end(); ++i) {
		sources.push_back (*i);
		(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		unique_srcs.insert (*i);
	}

	for (SourceList::iterator i = srcs.begin(); i != srcs.end(); ++i) {
		master_sources.push_back (*i);
		if (unique_srcs.find (*i) == unique_srcs.end()) {
			(*i)->GoingAway.connect (slot (*this, &AudioRegion::source_deleted));
		}
	}

	_fade_in_data = 0;
	_fade_out_data = 0;
	build_fade (&_fade_in_data, _fade_in, true);
	build_fade (&_fade_out_data, _fade_out, false);

	if (set_state (node)) {
		throw failed_constructor();
	}

	save_state ("initial state");
	RegionCreated (this); /* EMIT SIGNAL */
}

AudioRegion::~AudioRegion ()
{
	GoingAway (this);
}

StateManager::State*
AudioRegion::state_factory (std::string why) const
{
	AudioRegionState* state = new AudioRegionState (why);

	Region::store_state (*state);

	state->_fade_in = _fade_in;
	state->_fade_out = _fade_out;
	state->_envelope = _envelope;

	return state;
}	

Change
AudioRegion::restore_state (StateManager::State& sstate) 
{
	AudioRegionState* state = dynamic_cast<AudioRegionState*> (&sstate);

	Change what_changed = Region::restore_and_return_flags (*state);
	
	if (_flags != Flag (state->_flags)) {
		
		unsigned int old_flags = _flags;
		
		_flags = Flag (state->_flags);
		
		if ((old_flags ^ state->_flags) & EnvelopeActive) {
			what_changed = Change (what_changed|EnvelopeActiveChanged);
		}
	}
		
	if (_fade_in != state->_fade_in) {
		_fade_in = state->_fade_in;
		build_fade (&_fade_in_data, _fade_in, true);
		what_changed = Change (what_changed|FadeInChanged);
	}

	if (_fade_out != state->_fade_out) {
		_fade_out = state->_fade_out;
		build_fade (&_fade_out_data, _fade_out, false);
		what_changed = Change (what_changed|FadeOutChanged);
	}

	/* XXX need a way to test stored state versus current for envelopes */

	_envelope = state->_envelope;
	what_changed = Change (what_changed);

	return what_changed;
}

UndoAction
AudioRegion::get_memento() const
{
	return bind (slot (*(const_cast<AudioRegion *> (this)), &StateManager::use_state), _current_state_id);
}

bool
AudioRegion::verify_length (jack_nframes_t len)
{
	for (unsigned int n=0; n < sources.size(); ++n) {
		if (_start > sources[n]->length() - len) {
			return false;
		}
	}
	return true;
}

bool
AudioRegion::verify_start_and_length (jack_nframes_t new_start, jack_nframes_t new_length)
{
	for (unsigned int n=0; n < sources.size(); ++n) {
		if (new_length > sources[n]->length() - new_start) {
			return false;
		}
	}
	return true;
}
bool
AudioRegion::verify_start (jack_nframes_t pos)
{
	for (unsigned int n=0; n < sources.size(); ++n) {
		if (pos > sources[n]->length() - _length) {
			return false;
		}
	}
	return true;
}

bool
AudioRegion::verify_start_mutable (jack_nframes_t& new_start)
{
	for (unsigned int n=0; n < sources.size(); ++n) {
		if (new_start > sources[n]->length() - _length) {
			new_start = sources[n]->length() - _length;
		}
	}
	return true;
}
void
AudioRegion::set_envelope_active (bool yn)
{
	if (envelope_active() != yn) {
		char buf[64];
		if (yn) {
			sprintf (buf, "envelope active");
			_flags = Flag (_flags|EnvelopeActive);
		} else {
			sprintf (buf, "envelope off");
			_flags = Flag (_flags & ~EnvelopeActive);
		}
		if (!_frozen) {
			save_state (buf);
		}
		send_change (EnvelopeActiveChanged);
	}
}

void
AudioRegion::use_default_fade_out ()
{
	if (_flags & DefaultFadeOut) {
		return;
	}
	
	_flags = Flag (_flags|DefaultFadeOut);

	/* reset the non-default one to match the default, but
	   be lazy by not constructing the data set. we won't
	   need it till someone resets one of the fade params,
	   when we'll recalculate it anyway.
	*/

	_fade_out.set_steepness (default_fade_out.steepness());
	_fade_out.set_length (default_fade_out.length());
	build_fade (&_fade_out_data, _fade_out, false);

	send_change (FadeOutChanged);
}

void
AudioRegion::use_default_fade_in ()
{
	if (_flags & DefaultFadeIn) {
		return;
	}
	
	_flags = Flag (_flags|DefaultFadeIn);

	/* reset the non-default one to match the default, but
	   be lazy by not constructing the data set. we won't
	   need it till someone resets one of the fade params,
	   when we'll recalculate it anyway.
	*/

	_fade_in.set_steepness(default_fade_in.steepness());
	_fade_in.set_length (default_fade_in.length());

	send_change (FadeInChanged);
}

void
AudioRegion::set_fade_in (float steepness, jack_nframes_t length)
{
	if (steepness == _fade_in.steepness() && length == _fade_in.length()) {
		return;
	}

	_fade_in.set_steepness(steepness);
	_fade_in.set_length (length);
	
	_flags = Flag (_flags & ~DefaultFadeIn);

	if (!_frozen) {
		char buf[64];
		sprintf (buf, "fade in set to %lu/%f", _fade_in.length(), _fade_in.steepness());
		save_state (buf);
	}

	build_fade (&_fade_in_data, _fade_in, true);
	send_change (FadeInChanged);
}

void
AudioRegion::set_fade_out (float steepness, jack_nframes_t length)
{
	if (steepness == _fade_out.steepness() && length == _fade_out.length()) {
		return;
	}

	_fade_out.set_steepness(steepness);
	_fade_out.set_length (length);

	_flags = Flag (_flags & ~DefaultFadeOut);

	if (!_frozen) {
		char buf[64];
		sprintf (buf, "fade out set to %lu/%f", _fade_out.length(), _fade_out.steepness());
		save_state (buf);
	}

	build_fade (&_fade_out_data, _fade_out, false);
		
	send_change (FadeOutChanged);
}

jack_nframes_t
AudioRegion::read_peaks (PeakData *buf, jack_nframes_t npeaks, jack_nframes_t offset, jack_nframes_t cnt, unsigned int chan_n) const
{
	if (chan_n >= sources.size()) {
		return 0;
	}
	
	if (sources[chan_n]->read_peaks (buf, npeaks, _start + offset, cnt)) {
		return 0;
	} else {
		return cnt;
	}
}

jack_nframes_t
AudioRegion::read_at (Sample *buf, Sample *mixdown_buffer, float *gain_buffer, jack_nframes_t position, 
		      jack_nframes_t cnt, 
		      unsigned int chan_n, jack_nframes_t read_frames, jack_nframes_t skip_frames) const
{
	return _read_at (sources, buf, mixdown_buffer, gain_buffer, position, cnt, chan_n, read_frames, skip_frames);
}

jack_nframes_t
AudioRegion::master_read_at (Sample *buf, Sample *mixdown_buffer, float *gain_buffer, jack_nframes_t position, 
			     jack_nframes_t cnt, unsigned int chan_n) const
{
	return _read_at (master_sources, buf, mixdown_buffer, gain_buffer, position, cnt, chan_n, 0, 0);
}

jack_nframes_t
AudioRegion::_read_at (const SourceList& srcs, Sample *buf, Sample *mixdown_buffer, float *gain_buffer, 
		       jack_nframes_t position, jack_nframes_t cnt, 
		       unsigned int chan_n, jack_nframes_t read_frames, jack_nframes_t skip_frames) const
{
	jack_nframes_t internal_offset;
	jack_nframes_t buf_offset;
	jack_nframes_t to_read;
	
	/* precondition: caller has verified that we cover the desired section */

	if (chan_n >= sources.size()) {
		return 0; /* read nothing */
	}
	
	if (position < _position) {
		internal_offset = 0;
		buf_offset = _position - position;
		cnt -= buf_offset;
	} else {
		internal_offset = position - _position;
		buf_offset = 0;
	}

	if (internal_offset >= _length) {
		return 0; /* read nothing */
	}
	

	if ((to_read = min (cnt, _length - internal_offset)) == 0) {
		return 0; /* read nothing */
	}

	if (opaque()) {
		/* overwrite whatever is there */
		mixdown_buffer = buf + buf_offset;
	} else {
		mixdown_buffer += buf_offset;
	}

	if (muted()) {
		if (opaque()) {
			/* write silence */
			for (jack_nframes_t n = 0; n < to_read; ++n) {
				*mixdown_buffer++ = 0.0;;
			}
			return to_read;
		}
		else 
			return 0; /* read nothing */
	}

	 _read_data_count = 0;

#if 0
	 if (read_frames) {

		 cerr << _name << " read " << cnt << " (scaled to " << to_read << ") @ " << position << " ioffset = " << internal_offset << endl;

		/* SKIP READ MODE: no fades, no gain curves */

		jack_nframes_t left = to_read;
		jack_nframes_t pos = _start + internal_offset;
		jack_nframes_t offset = 0;
		jack_nframes_t orig = to_read;
		jack_nframes_t xx = 0;

		while (left) {

			to_read = min (left, read_frames);

			cerr << _name << " read " << to_read << " @ " << pos << " w/offset = " << offset
			     << " left = " << left 
			     << endl;
			
			if (srcs[chan_n]->read (mixdown_buffer+offset, pos, to_read) != to_read) {
				return 0; /* "read nothing" */
			}

			xx += to_read;
			
			cerr << "\t data transferred to buffer = " << xx << endl;

			_read_data_count += to_read;
			pos += skip_frames;
			offset += to_read;

			if (left > to_read + skip_frames) {
				left -= to_read + skip_frames;
			} else {
				break;
			}
		}

		return orig;

	} 
#endif

	if (srcs[chan_n]->read (mixdown_buffer, _start + internal_offset, to_read) != to_read) {
		return 0; /* "read nothing" */
	}

	_read_data_count += srcs[chan_n]->read_data_count();

	/* Apply fades if appropriate */

	const LogCurve *fin = 0;
	const LogCurve *fout = 0;
	gain_t   *din = 0;
	gain_t   *dout = 0;

	TentativeLockMonitor lmfi (_fade_in.lock, __LINE__, __FILE__);
	TentativeLockMonitor lmfo (_fade_out.lock, __LINE__, __FILE__);

	if (!lmfi.locked()) {
		goto fo;
	}

	if (_flags & DefaultFadeIn) {
		fin = &default_fade_in;
		din = default_fade_in_data;
	} else {
		fin = &_fade_in;
		din = _fade_in_data;
	}

  fo:
	if (!lmfo.locked()) {
		goto nf;
	}

	if (_flags & DefaultFadeOut) {
		fout = &default_fade_out;
		dout = default_fade_out_data;
	} else {
		fout = &_fade_out;
		dout = _fade_out_data;
	}

  nf:

	/* fade in */

	if (din && (_flags & FadeIn) && internal_offset < fin->length()) {

		jack_nframes_t gain = internal_offset;
		jack_nframes_t limit = min ((jack_nframes_t)fin->length() - gain, to_read);

		for (jack_nframes_t n = 0; n < limit; ++n, ++gain) {
			mixdown_buffer[n] *= din[gain];
		}
	}
	
	/* fade out */

	if (_flags & FadeOut) {
		jack_nframes_t fade_out_start = _length - fout->length();
		
		if (dout && internal_offset+to_read >= fade_out_start) {
			
			jack_nframes_t limit;
			jack_nframes_t frame;
			jack_nframes_t gain;

	
			if (internal_offset < fade_out_start) {
				jack_nframes_t fade_out_offset = fade_out_start - internal_offset;
				limit = min ((jack_nframes_t)fout->length(), to_read - fade_out_offset);
				frame = fade_out_offset;
				gain = 0;
			} else {
				jack_nframes_t fade_out_offset = internal_offset - fade_out_start;
				limit = min ((jack_nframes_t)fout->length() - fade_out_offset, to_read);
				frame = 0;
				gain = fade_out_offset;
			}

			for (jack_nframes_t n = 0; n < limit; ++n, ++frame, ++gain) {
				mixdown_buffer[frame] *= dout[gain];
			}
		}
	}

	/* Regular gain curves */

	if (envelope_active())  {
		_envelope.get_vector (internal_offset, internal_offset + to_read, gain_buffer, to_read);
		
		for (jack_nframes_t n = 0; n < to_read; ++n) {
			mixdown_buffer[n] *= gain_buffer[n];
		}
	}

	if (!opaque()) {

		/* gack. the things we do for users.
		 */

		buf += buf_offset;

		for (jack_nframes_t n = 0; n < to_read; ++n) {
			buf[n] = mixdown_buffer[n];
		}
	} 
	
	return to_read;
}
	
XMLNode&
AudioRegion::get_state ()
{
	XMLNode& node (Region::get_state());
	XMLNode *child;
	char buf[64];
	char buf2[64];

	setlocale (LC_NUMERIC, "POSIX");
	
	sprintf (buf, "0x%x", (int) _flags);
	node.add_property ("flags", buf);

	/* backward compatibility */
	sprintf (buf, "%Lu", sources[0]->id());
	node.add_property ("source", buf);

	for (unsigned int n=0; n < sources.size(); ++n) {
		snprintf (buf2, sizeof(buf2), "source-%d", n);
		snprintf (buf, sizeof(buf), "%Lu", sources[n]->id());
		node.add_property (buf2, buf);
	}

	sprintf (buf, "%u", (unsigned int) sources.size());
	node.add_property ("channels", buf);
	
	if ((_flags & DefaultFadeIn) == 0) {
		child = node.add_child ("FadeIn");
		sprintf (buf, "%f", _fade_in.steepness());
		child->add_property ("steepness", buf);
		sprintf (buf, "%lu", _fade_in.length());
		child->add_property ("length", buf);
	} 

	if ((_flags & DefaultFadeOut) == 0) {
		child = node.add_child ("FadeOut");
		sprintf (buf, "%f", _fade_out.steepness());
		child->add_property ("steepness", buf);
		sprintf (buf, "%lu", _fade_out.length());
		child->add_property ("length", buf);
	}

	child = node.add_child ("Envelope");

	bool default_env = false;

	if (_envelope.size() == 2) {
		if ((*(--_envelope.end()))->when == _length) {
			default_env = true;
			child->add_property ("default", "yes");
			
		}
	} 

	if (default_env) {
		child->add_property ("default", "yes");
	} else {
		for (AutomationList::iterator i = _envelope.begin(); i != _envelope.end(); ++i) {

			XMLNode *pointnode = new XMLNode ("point");
			
			sprintf (buf, "%lu", (jack_nframes_t) floor ((*i)->when));
			pointnode->add_property ("x", buf);
			sprintf (buf, "%f", (*i)->value);
			pointnode->add_property ("y", buf);
			child->add_child_nocopy (*pointnode);
		}
	}

	if (_extra_xml) {
		node.add_child_copy (*_extra_xml);
	}

	setlocale (LC_NUMERIC, "");
	
	return node;
}

XMLNode&
AudioRegion::get_short_state ()
{
	XMLNode& node (Region::get_short_state());
	XMLNode *child;
	char buf[64];
	char buf2[64];

	/* Region doesn't store this, allowing derived classes
	   to add/remove bits from it.
	*/
	
	sprintf (buf, "0x%x", (int) _flags);
	node.add_property ("flags", buf);

	/* backward compatibility */
	sprintf (buf, "%Lu", sources[0]->id());
	node.add_property ("source", buf);

	for (unsigned int n=0; n < sources.size(); ++n) {
		snprintf (buf2, sizeof(buf2), "source-%d", n);
		snprintf (buf, sizeof(buf), "%Lu", sources[n]->id());
		node.add_property (buf2, buf);
	}

	sprintf (buf, "%u", (unsigned int) sources.size());
	node.add_property ("channels", buf);

	if ((_flags & DefaultFadeIn) == 0) {
		child = node.add_child ("FadeIn");
		sprintf (buf, "%f", _fade_in.steepness());
		child->add_property ("steepness", buf);
		sprintf (buf, "%lu", _fade_in.length());
		child->add_property ("length", buf);
	} 

	if ((_flags & DefaultFadeOut) == 0) {
		child = node.add_child ("FadeOut");
		sprintf (buf, "%f", _fade_out.steepness());
		child->add_property ("steepness", buf);
		sprintf (buf, "%lu", _fade_out.length());
		child->add_property ("length", buf);
	}

	child = node.add_child ("Envelope");
	child->add_property ("default", "yes");

	return node;
}

int
AudioRegion::set_state (const XMLNode& node)
{
	const XMLNodeList& nlist = node.children();
	const XMLProperty *prop;

	Region::set_state (node);

	if ((prop = node.property ("flags")) != 0) {
		_flags = Flag (strtol (prop->value().c_str(), (char **) 0, 16));
	}
	
	if ((_flags & DefaultFadeIn) == 0) {

		/* Check for children that describe various curves */
		
		for (XMLNodeConstIterator niter = nlist.begin(); niter != nlist.end(); ++niter) {
			
			XMLNode *child;
			XMLProperty *childprop;
			
			child = (*niter);
			
			if (child->name() == "FadeIn") {
				
				if ((childprop = child->property ("steepness")) == 0) {
					error << _("AudioRegion XML node contains FadeIn, but no steepness") << endmsg;
					return -1;
				}
				_fade_in.set_steepness(atof (childprop->value().c_str()));
				
				if ((childprop = child->property ("length")) == 0) {
					error << _("AudioRegion XML node contains FadeIn, but no length") << endmsg;
					return -1;
				}
				_fade_in.set_length((jack_nframes_t) atoi (childprop->value().c_str()));
				
				build_fade (&_fade_in_data, _fade_in, true);
			}
		}
	}
	
	if ((_flags & DefaultFadeOut) == 0) {

		for (XMLNodeConstIterator niter = nlist.begin(); niter != nlist.end(); ++niter) {
			
			XMLNode *child;
			XMLProperty *childprop;
			
			child = (*niter);
			
			if (child->name() == "FadeOut") {
				
				if ((childprop = child->property ("steepness")) == 0) {
					error << _("AudioRegion XML node contains FadeOut, but no steepness") << endmsg;
					return -1;
				}
				setlocale (LC_NUMERIC, "POSIX");
				_fade_out.set_steepness(atof (childprop->value().c_str()));
				setlocale (LC_NUMERIC, "");
				
				if ((childprop = child->property ("length")) == 0) {
					error << _("AudioRegion XML node contains FadeOut, but no length") << endmsg;
					return -1;
				}
				_fade_out.set_length((jack_nframes_t) atoi (childprop->value().c_str()));

				build_fade (&_fade_out_data, _fade_out, false);
			}
		}
	}

	/* Now find envelope description and other misc child items */
	setlocale (LC_NUMERIC, "POSIX");
				
	for (XMLNodeConstIterator niter = nlist.begin(); niter != nlist.end(); ++niter) {
		
		XMLNode *child;
		
		child = (*niter);
		
		if (child->name() == "Envelope") {
			
			XMLProperty *prop;
			jack_nframes_t x;
			float y;

			_envelope.clear ();

			if ((prop = child->property ("default")) != 0) {
				set_default_envelope ();
			} else {
			
				const XMLNodeList& elist = child->children();
				XMLNodeConstIterator i;
				
				

				for (i = elist.begin(); i != elist.end(); ++i) {
					
					if ((prop = (*i)->property ("x")) == 0) {
						error << compose(_("%1: no x-coordinate stored for control point (point ignored)"), _name) << endmsg;
						continue;
					}
					x = atoi (prop->value().c_str());
					
					if ((prop = (*i)->property ("y")) == 0) {
						error << compose(_("%1: no y-coordinate stored for control point (point ignored)"), _name) << endmsg;
						continue;
					}
					y = atof (prop->value().c_str());

					_envelope.add (x, y);
				}

			}

			_envelope.set_max_xval (_length);
			
		} 
	}

	setlocale (LC_NUMERIC, "");
	
	return 0;
}

void
AudioRegion::build_fade (gain_t **gain, LogCurve& curve, bool in)
{
	if (*gain) {
		delete [] *gain;
		*gain = 0;
	}

	if (curve.length() == 0 || curve.steepness() == 0) {
		return;
	}

	*gain = new gain_t[curve.length()];
	curve.fill (*gain, curve.length(), !in);
}

void
AudioRegion::set_default_fade (float steepness, jack_nframes_t len)
{
	default_fade_in.set_steepness (steepness);
	default_fade_in.set_length (len);
	default_fade_out.set_steepness (steepness);
	default_fade_out.set_length (len);

	build_fade (&default_fade_in_data, default_fade_in, true);
	build_fade (&default_fade_out_data, default_fade_out, false);
}

void
AudioRegion::set_default_envelope ()
{
	_envelope.freeze ();
	_envelope.clear ();
	_envelope.add (0, 1.0f);
	_envelope.add (_length, 1.0f);
	_envelope.thaw ();
}

void
AudioRegion::recompute_at_end ()
{
	/* our length has changed. recompute a new final point by interpolating 
	   based on the the existing curve.
	*/
	
	_envelope.freeze ();
	_envelope.truncate_end (_length);
	_envelope.set_max_xval (_length);
	_envelope.thaw ();
}	

void
AudioRegion::recompute_at_start ()
{
	/* as above, but the shift was from the front */

	_envelope.truncate_start (_length);
}

void
AudioRegion::set_fade_in_length (jack_nframes_t len)
{
	{
		LockMonitor lm (_fade_in.lock, __LINE__, __FILE__);
		
		if (len == _fade_in.length()) {
			return;
		}
		
		_fade_in.set_length (len);
		if (_flags & DefaultFadeIn) {
			_fade_in.set_steepness (default_fade_in.steepness());
		}
		
		build_fade (&_fade_in_data, _fade_in, true);
		
		_flags = Flag (_flags & ~DefaultFadeIn);
		
		if (!_frozen) {
			char buf[64];
			sprintf (buf, "fade in length changed to %u", len);
			save_state (buf);
		}
	}

	send_change (FadeInChanged);
}

void
AudioRegion::set_fade_out_length (jack_nframes_t len)
{
	{ 
		LockMonitor lm (_fade_out.lock, __LINE__, __FILE__);
		
		if (len == _fade_out.length()) {
			return;
		}
		
		_fade_out.set_length (len);
		if (_flags & DefaultFadeOut) {
			_fade_out.set_steepness (default_fade_in.steepness());
		}
		build_fade (&_fade_out_data, _fade_out, false);
		
		_flags = Flag (_flags & ~DefaultFadeOut);
		
		if (!_frozen) {
			char buf[64];
			sprintf (buf, "fade out length changed to %u", len);
			save_state (buf);
		}
	}

	send_change (FadeOutChanged);
}

void
AudioRegion::set_fade_in_steepness (float steep)
{
	{
		LockMonitor lm (_fade_in.lock, __LINE__, __FILE__);
		
		if (steep == _fade_in.steepness()) {
			return;
		}
		
		_fade_in.set_steepness (steep);
		if (_flags & DefaultFadeIn) {
			_fade_in.set_length (default_fade_in.length());
		}
		build_fade (&_fade_in_data, _fade_in, true);
		
		_flags = Flag (_flags & ~DefaultFadeIn);
		
		if (!_frozen) {
			char buf[64];
			sprintf (buf, "fade in steepness changed to %f", steep);
			save_state (buf);
		}
	}

	send_change (FadeInChanged);
}

void
AudioRegion::set_fade_out_steepness (float steep)
{
	{
		LockMonitor lm (_fade_out.lock, __LINE__, __FILE__);
		
		if (steep == _fade_out.steepness()) {
			return;
		}
		
		_fade_out.set_steepness (steep);
		if (_flags & DefaultFadeOut) {
			_fade_out.set_length (default_fade_out.length());
		}
		build_fade (&_fade_out_data, _fade_out, false);
		
		_flags = Flag (_flags & ~DefaultFadeOut);
		
		if (!_frozen) {
			char buf[64];
			sprintf (buf, "fade out steepness changed to %f", steep);
			save_state (buf);
		}
	}

	send_change (FadeOutChanged);
}

const LogCurve& 
AudioRegion::fade_in() const
{
	if (_flags & DefaultFadeIn) {
		return default_fade_in;
	} else {
		return _fade_in;
	}
}

const LogCurve& 
AudioRegion::fade_out() const
{
	if (_flags & DefaultFadeOut) {
		return default_fade_out;
	} else {
		return _fade_out;
	}
}

void
AudioRegion::set_fade_in_active (bool yn)
{
	if (yn == (_flags & FadeIn)) {
		return;
	}
	if (yn) {
		_flags = Flag (_flags|FadeIn);
	} else {
		_flags = Flag (_flags & ~FadeIn);
	}

	send_change (FadeInActiveChanged);
}

void
AudioRegion::set_fade_out_active (bool yn)
{
	if (yn == (_flags & FadeOut)) {
		return;
	}
	if (yn) {
		_flags = Flag (_flags | FadeOut);
	} else {
		_flags = Flag (_flags & ~FadeOut);
	}

	send_change (FadeOutActiveChanged);
}

int
AudioRegion::separate_by_channel (Session& session, vector<AudioRegion*>& v) const
{
	SourceList s, m;
	AudioRegion r (*this); // invoke pure copy constructor
	string new_name;

	/* now mess with its source list */
	
	s = r.sources;
	m = r.master_sources;

	for (SourceList::iterator i = s.begin(); i != s.end(); ++i) {

		r.sources.clear ();
		r.master_sources.clear();
		r.sources.push_back (*i);
		r.master_sources.push_back (*i);

		/* generate a new name */
		
		if (session.region_name (new_name, _name)) {
			return -1;
		}

		/* create a copy with just one source */

		v.push_back (new AudioRegion (r, new_name));
	}

	return 0;
}

void
AudioRegion::source_deleted (Source* ignored)
{
	delete this;
}

void
AudioRegion::lock_sources ()
{
	SourceList::iterator i;
	set<Source*> unique_srcs;

	for (i = sources.begin(); i != sources.end(); ++i) {
		unique_srcs.insert (*i);
		(*i)->use ();
	}

	for (i = master_sources.begin(); i != master_sources.end(); ++i) {
		if (unique_srcs.find (*i) == unique_srcs.end()) {
			(*i)->use ();
		}
	}
}

void
AudioRegion::unlock_sources ()
{
	SourceList::iterator i;
	set<Source*> unique_srcs;

	for (i = sources.begin(); i != sources.end(); ++i) {
		unique_srcs.insert (*i);
		(*i)->release ();
	}

	for (i = master_sources.begin(); i != master_sources.end(); ++i) {
		if (unique_srcs.find (*i) == unique_srcs.end()) {
			(*i)->release ();
		}
	}
}

vector<string>
AudioRegion::master_source_names ()
{
	SourceList::iterator i;

	vector<string> names;
	for (i = master_sources.begin(); i != master_sources.end(); ++i) {
		names.push_back((*i)->name());
	}

	return names;
}

bool
AudioRegion::source_equivalent (const AudioRegion& other)
{
	SourceList::iterator i;
	SourceList::const_iterator io;

	for (i = sources.begin(), io = other.sources.begin(); i != sources.end() && io != other.sources.end(); ++i, ++io) {
		if ((*i)->id() != (*i)->id()) {
			return false;
		}
	}

	for (i = master_sources.begin(), io = other.master_sources.begin(); i != master_sources.end() && io != other.master_sources.end(); ++i, ++io) {
		if ((*i)->id() != (*i)->id()) {
			return false;
		}
	}

	return true;
}

bool
AudioRegion::equivalent (const AudioRegion& other)
{
	return _start == other._start &&
		_position == other._position &&
		_length == other._length;
}

int
AudioRegion::apply (AudioFilter& filter)
{
	return filter.run (*this);
}

int
AudioRegion::exportme (Session& session, AudioExportSpecification& spec)
{
	const jack_nframes_t blocksize = 4096;
	jack_nframes_t to_read;
	int status = -1;

	spec.channels = sources.size();

	if (spec.prepare (blocksize, session.frame_rate())) {
		goto out;
	}

	spec.pos = 0;
	spec.total_frames = _length;

	while (spec.pos < _length && !spec.stop) {
		
		
		/* step 1: interleave */
		
		to_read = min (_length - spec.pos, blocksize);
		
		if (spec.channels == 1) {

			if (sources.front()->read (spec.dataF, _start + spec.pos, to_read) != to_read) {
				goto out;
			}

		} else {

			Sample buf[blocksize];

			for (unsigned int chan = 0; chan < spec.channels; ++chan) {
				
				if (sources[chan]->read (buf, _start + spec.pos, to_read) != to_read) {
					goto out;
				}
				
				for (jack_nframes_t x = 0; x < to_read; ++x) {
					spec.dataF[chan+(x*spec.channels)] = buf[x];
				}
			}
		}
		
		if (spec.process (to_read)) {
			goto out;
		}
		
		spec.pos += to_read;
		spec.progress = 1.0 - ((double) spec.pos /_length);
		
	}
	
	status = 0;

  out:	
	spec.running = false;
	spec.status = status;
	
	return status;
}

extern "C" {

int region_read_peaks_from_c (void *arg, gulong npeaks, gulong start, gulong cnt, gpointer data, unsigned int n_chan) 
{
	return ((AudioRegion *) arg)->read_peaks ((PeakData *) data, (jack_nframes_t) npeaks, (jack_nframes_t) start, (jack_nframes_t) cnt, n_chan);
}

gulong region_length_from_c (void *arg)
{

	return ((AudioRegion *) arg)->length();
}

} /* extern "C" */
