/*
    Copyright (C) 2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: slave.h,v 1.3 2002/07/20 02:54:46 pbd Exp $
*/

#ifndef __ardour_slave_h__
#define __ardour_slave_h__

#include <pthread.h>
#include <sigc++/signal_system.h>
#include <ardour/ardour.h>
#include <midi++/parser.h>

namespace MIDI {
	class Port;
}

namespace ARDOUR {
class Session;

class Slave {
  public:
	Slave() {}
	virtual ~Slave() {}

	virtual jack_nframes_t current_audio_frame() const = 0;
	virtual bool locked() const = 0;
	virtual bool ok() const = 0;
};


class MTC_Slave : public Slave, public SigC::Object {
  public:
	MTC_Slave (Session&, MIDI::Port&);
	~MTC_Slave ();

	jack_nframes_t current_audio_frame() const;
	bool      locked() const;
	bool      ok() const;

  private:
	pthread_t thread;
	Session&    session;
	MIDI::Port& port;

	struct SafeTime {

	    int guard1;
	    SMPTE_Time  mtc;
	    jack_nframes_t   timestamp;
	    int guard2;

	    SafeTime() {
		    guard1 = 0;
		    guard2 = 0;
		    timestamp = 0;
	    }
	};

	SigC::Connection mtc_connection;
	SafeTime   current;
	jack_nframes_t  last_inbound_mtc_frame;   
	static const int mtc_accumulator_size = 16;
	float     mtc_speed_accumulator[mtc_accumulator_size];
	int       mtc_speed_counter;
	float     mtc_speed;
	int       thread_ok;
	pthread_mutex_t thread_lock;
	pthread_cond_t  thread_running;

	static void *_thread_work (void *);
	void *thread_work ();
	void update_mtc_time (const MIDI::byte *, bool);
	void update_mtc_status (MIDI::Parser::MTC_Status);
	void read_current (SafeTime *) const;
};

class ADAT_Slave : public Slave 
{
  public:
	ADAT_Slave () {}
	~ADAT_Slave () {}
	
	jack_nframes_t current_audio_frame() const { return 0; }
	bool locked() const { return false; }
	bool ok() const { return false; }
};

class JACK_Slave : public Slave 
{
  public:
	JACK_Slave () {}
	~JACK_Slave () {}
	
	jack_nframes_t current_audio_frame() const { return 0; }
	bool locked() const { return false; }
	bool ok() const { return false; }
};

class Scrub_Slave : public Slave 
{
  public:
	Scrub_Slave () { _current = 0; }
	~Scrub_Slave () {}
	
	jack_nframes_t current_audio_frame() const { return _current; }
	bool locked() const { return true; }
	bool ok() const { return true; }

	void set_current_frame (jack_nframes_t f) { _current = f; }

  private:
	jack_nframes_t _current;
};

} /* namespace */

#endif /* __ardour_slave_h__ */
