/*
 *  Copyright (C) 2002 Steve Harris <steve@plugin.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Id: dither.h,v 1.1 2003/01/07 16:44:16 pbd Exp $
 */

#ifndef DITHER_H
#define DITHER_H

#define DITHER_SH_BUF_SIZE 8
#define DITHER_SH_BUF_MASK 7

typedef enum {
    DitherNone,
    DitherRect,
    DitherTri,
    DitherShaped
} DitherType;

typedef struct {
    unsigned int phase;
    float buffer[DITHER_SH_BUF_SIZE];
} DitherShapedState;

typedef struct {
    DitherType type;
    unsigned int channels;
    unsigned int bit_depth;
    float scale;
    float bias;
    int   clamp_u;
    int   clamp_l;
    float *tri_state;
    DitherShapedState *shaped_state;
} Dither;

/* Create and initialise a state structure, takes a dither type, a number of
 * channels and a bit depth as input
 *
 * The Dither type is one of
 *
 *   DitherNone - straight nearest neighbour rounding. Theres no good reason to
 *   do this at 8 or 16 bit, but you might want to at 24, for some reason.
 *
 *   DitherRect - mathematically most accurate, lowest noise floor, but not
 *   that good for audio. It is the fastest though.
 *
 *   DitherTri - a happy medium between Rectangular and Shaped, reasonable
 *   noise floor, not too obvious, quite fast.
 *
 *   DitherShaped - should have the least audible impact, but has the highest
 *   noise floor, fairly CPU intensive.
 *
 * bit depth can be one of
 *
 *   8 - 8 bit unsiged
 *
 *   16 - 16 bit signed
 *
 *   24 - 24 in upper 24 bits of a 32 bit word
 */

Dither *dither_new(DitherType type, unsigned int channels, unsigned int bit_depth);
void    dither_free(Dither * s);

/* Applies dithering to the supplied signal.
 *
 * channel is the channle number you are processing (0 - channles-1), length is
 * the length of the input, in samples, x is the input samples (float), y is
 * where the output samples will be written, it should have the approaprate
 * type for the chosen bit depth
 */
void dither_run(Dither * s, unsigned int channel, unsigned int length, float *x, void *y);

#endif /* DITHER_H */
