/*
    Copyright (C) 2000-2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: audioregion.h,v 1.11 2004/02/29 23:33:57 pauld Exp $
*/

#ifndef __ardour_audio_region_h__
#define __ardour_audio_region_h__

#include <vector>

#include <pbd/fastlog.h>
#include <pbd/undo.h>

#include <ardour/ardour.h>
#include <ardour/source.h>
#include <ardour/gain.h>
#include <ardour/logcurve.h>
#include <ardour/region.h>
#include <ardour/export.h>

class XMLNode;

namespace ARDOUR {

class Route;
class Playlist;
class Session;
class AudioFilter;

struct AudioRegionState : public RegionState {

    AudioRegionState (std::string why) : RegionState (why) {}

    LogCurve  _fade_in;
    LogCurve  _fade_out;
    Gain      _envelope;
};

class AudioRegion : public Region
{
  public:
	typedef vector<Source *> SourceList;

	static Change FadeInChanged;
	static Change FadeOutChanged;
	static Change FadeInActiveChanged;
	static Change FadeOutActiveChanged;
	static Change EnvelopeActiveChanged;

	AudioRegion (Source&, jack_nframes_t start, jack_nframes_t length, bool announce = true);
	AudioRegion (Source&, jack_nframes_t start, jack_nframes_t length, const string& name, layer_t = 0, Region::Flag flags = Region::DefaultFlags, bool announce = true);
	AudioRegion (SourceList &, jack_nframes_t start, jack_nframes_t length, const string& name, layer_t = 0, Region::Flag flags = Region::DefaultFlags, bool announce = true);
	AudioRegion (const AudioRegion&, jack_nframes_t start, jack_nframes_t length, const string& name, layer_t = 0, Region::Flag flags = Region::DefaultFlags, bool announce = true);
	AudioRegion (const AudioRegion&, const string& name);
	AudioRegion (const AudioRegion&);
	AudioRegion (Source&, const XMLNode&);
	AudioRegion (SourceList &, const XMLNode&);
	~AudioRegion();

	bool source_equivalent (const AudioRegion&);
	bool equivalent (const AudioRegion&);

	void lock_sources ();
	void unlock_sources ();
	Source& source (unsigned int n=0) const { if (n < sources.size()) return *sources[n]; else return *sources[0]; } 

	Gain&   envelope() { return _envelope; }

	unsigned int n_channels() { return sources.size(); }
	vector<string> master_source_names();
	
	bool fade_in_active () const { return _flags & Region::FadeIn; }
	bool fade_out_active () const { return _flags & Region::FadeOut; }
	bool captured() const { return !(_flags & (Region::Flag (Region::Import|Region::External))); }

	const LogCurve& fade_in() const;
	const LogCurve& fade_out() const;

	virtual bool is_crossfade() const { return false; }

	// jack_nframes_t read (Sample *buf, jack_nframes_t offset, jack_nframes_t cnt) const;

	jack_nframes_t read_peaks (PeakData *buf, jack_nframes_t npeaks, jack_nframes_t offset, jack_nframes_t cnt, unsigned int chan_n=0) const;

	virtual jack_nframes_t read_at (Sample *buf, Sample *mixdown_buffer, 
					float *gain_buffer, jack_nframes_t position, jack_nframes_t cnt, 
					unsigned int chan_n = 0,
					jack_nframes_t read_frames = 0,
					jack_nframes_t skip_frames = 0) const;

	jack_nframes_t master_read_at (Sample *buf, Sample *mixdown_buffer, 
				       float *gain_buffer, jack_nframes_t position, jack_nframes_t cnt, unsigned int chan_n=0) const;


	XMLNode& get_state ();
	int      set_state (const XMLNode&);
	XMLNode& get_short_state (); /* used only by Session */

	static void set_default_fade (float steepness, jack_nframes_t len);

	void use_default_fade_in ();
	void use_default_fade_out ();

	void set_fade_in_active (bool yn);
	void set_fade_in (float steepness, jack_nframes_t len);
	void set_fade_in_length (jack_nframes_t);
	void set_fade_in_steepness (float);

	void set_fade_out_active (bool yn);
	void set_fade_out (float steepness, jack_nframes_t len);
	void set_fade_out_length (jack_nframes_t);
	void set_fade_out_steepness (float);

	void set_envelope_active (bool yn);

	int separate_by_channel (ARDOUR::Session&, vector<AudioRegion*>&) const;

	unsigned long read_data_count() const { return _read_data_count; }

	ARDOUR::Playlist* playlist() const { return _playlist; }

	UndoAction get_memento() const;

	/* filter */

	int apply (AudioFilter&);

	/* export */

	int exportme (ARDOUR::Session&, ARDOUR::AudioExportSpecification&);

  private:
	friend class Playlist;

  private:
	SourceList        sources;
	SourceList        master_sources; /* used when timefx are applied, so 
					     we can always use the original
					     source.
					  */

	LogCurve       	  _fade_in;
	LogCurve       	  _fade_out;
	mutable Gain   	  _envelope;
	gain_t*           _fade_in_data;
	gain_t*           _fade_out_data;

	void set_default_envelope ();

	static gain_t    *default_fade_in_data;
	static gain_t    *default_fade_out_data;
	static LogCurve   default_fade_in;
	static LogCurve   default_fade_out;

	static void build_fade (gain_t **gain, LogCurve&, bool in);

	StateManager::State* state_factory (std::string why) const;
	Change restore_state (StateManager::State&);

	void recompute_gain_at_end ();
	void recompute_gain_at_start ();

	bool copied() const { return _flags & Copied; }
	void maybe_uncopy ();
	void rename_after_first_edit ();

	jack_nframes_t _read_at (const SourceList&, Sample *buf, Sample *mixdown_buffer, 
				 float *gain_buffer, jack_nframes_t position, jack_nframes_t cnt, 
				 unsigned int chan_n = 0,
				 jack_nframes_t read_frames = 0,
				 jack_nframes_t skip_frames = 0) const;

	bool verify_start (jack_nframes_t position);
	bool verify_length (jack_nframes_t position);
	bool verify_start_mutable (jack_nframes_t& start);
	bool verify_start_and_length (jack_nframes_t start, jack_nframes_t length);
	void recompute_at_start ();
	void recompute_at_end ();

	void source_deleted (Source*);
};

} /* namespace ARDOUR */

/* access from C objects */

extern "C" {
	int    region_read_peaks_from_c   (void *arg, gulong npeaks, gulong start, gulong length, gpointer data, unsigned int n_chan);
	gulong region_length_from_c (void *arg);
}

#endif /* __ardour_audio_region_h__ */
