/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: route_params_ui.cc,v 1.34 2004/01/15 01:56:10 pbd Exp $
*/

#include <algorithm>

#include <pbd/lockmonitor.h>
#include <gtkmmext/utils.h>
#include <gtkmmext/stop_signal.h>

#include <ardour/session.h>
#include <ardour/session_route.h>
#include <ardour/diskstream.h>
#include <ardour/ladspa_plugin.h>
#include <ardour/ardour.h>
#include <ardour/session.h>
#include <ardour/route.h>
#include <ardour/audio_track.h>
#include <ardour/send.h>
#include <ardour/insert.h>
#include <ardour/connection.h>
#include <ardour/session_connection.h>

#include "route_params_ui.h"
#include "keyboard.h"
#include "mixer_strip.h"
#include "plugin_manager.h"
#include "ardour_ui.h"
#include "plugin_ui.h"
#include "io_selector.h"
#include "send_ui.h"
#include "utils.h"

#include "i18n.h"

using namespace ARDOUR;
using namespace Gtk;
using namespace SigC;

static const gchar *route_display_titles[] = { N_("Tracks/Buses"), 0 };
static const gchar *pre_display_titles[] = { N_("Pre Redirects"), 0 };
static const gchar *post_display_titles[] = { N_("Post Redirects"), 0 };

RouteParams_UI::RouteParams_UI (AudioEngine& eng)
	: KeyboardTarget (*this, "routeparams"),
	  engine (eng),
	  route_select_list (internationalize(route_display_titles)),
	  pre_redirect_list (internationalize(pre_display_titles)),
	  post_redirect_list (internationalize(post_display_titles)),
	  input_button(_("INPUTs")), 
	  output_button(_("OUTPUTs")),
	  _route(0), 
	  _redirect(0), 
	  redirect_menu(0),
	  track_menu(0)
{
	redirect_drag_in_progress = false;

	route_select_list.column_titles_active();
	route_select_list.set_name ("RouteParamsListDisplay");
	route_select_list.set_shadow_type (GTK_SHADOW_IN);
	route_select_list.set_selection_mode (GTK_SELECTION_SINGLE);
	route_select_list.set_reorderable (false);
	route_select_list.set_usize (75, -1);
	route_select_scroller.add (route_select_list);
	route_select_scroller.set_policy (GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

	route_select_frame.set_name("RouteSelectBaseFrame");
	route_select_frame.set_shadow_type (GTK_SHADOW_IN);
	route_select_frame.add(route_select_scroller);

	list_vpacker.pack_start (route_select_frame, true, true);

	// i/o pre/post choices
	route_choice_frame.set_name("RouteChoiceBaseFrame");
	//route_choice_frame.set_shadow_type (GTK_SHADOW_IN);

	//track_input_label.set_name ("RouteParamsTitleButton");
	//track_input_label.set_text (_("NO TRACK"));
	//choice_vpacker.pack_start(track_input_label, false, false);
	
	input_button.set_name ("RouteParamsTitleButton");
	input_button.unset_flags (GTK_CAN_FOCUS);
	input_button.set_usize (-1, 30);

	output_button.set_name ("RouteParamsTitleButton");
	output_button.unset_flags (GTK_CAN_FOCUS);
	output_button.set_usize (-1, 30);
	
	choice_vpacker.pack_start(input_button, false, false);
	choice_vpacker.pack_start (output_button, false, false);

	
	pre_redirect_list.set_name ("RouteParamsPreListDisplay");
	pre_redirect_list.set_shadow_type (GTK_SHADOW_IN);
	pre_redirect_list.set_selection_mode (GTK_SELECTION_SINGLE);
	pre_redirect_list.column_titles_active ();
	pre_redirect_list.set_reorderable (true);
	pre_redirect_list.set_button_actions (0, GTK_BUTTON_DRAGS);
	pre_redirect_list.set_button_actions (1, 0);
	pre_redirect_list.set_button_actions (2, 0);
	pre_redirect_list.set_usize (60, 60);
	pre_redirect_list.drag_begin.connect (slot (*this, &RouteParams_UI::redirect_drag_begin));
	pre_redirect_list.drag_end.connect (slot (*this, &RouteParams_UI::redirect_drag_end));
	pre_redirect_list.row_move.connect (slot (*this, &RouteParams_UI::redirects_reordered));
	pre_redirect_scroller.add (pre_redirect_list);
	pre_redirect_scroller.set_policy (GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);


	post_redirect_list.set_name ("RouteParamsPostListDisplay");
	post_redirect_list.set_shadow_type (GTK_SHADOW_IN);
	post_redirect_list.set_selection_mode (GTK_SELECTION_SINGLE);
	post_redirect_list.set_reorderable (true);
	post_redirect_list.set_button_actions (0, GTK_BUTTON_DRAGS);
	post_redirect_list.set_button_actions (1, 0);
	post_redirect_list.set_button_actions (2, 0);
	post_redirect_list.column_titles_active ();
	post_redirect_list.set_usize (60, 60);
	post_redirect_list.drag_begin.connect (slot (*this, &RouteParams_UI::redirect_drag_begin));
	post_redirect_list.drag_end.connect (slot (*this, &RouteParams_UI::redirect_drag_end));
	post_redirect_list.row_move.connect (slot (*this, &RouteParams_UI::redirects_reordered));
	post_redirect_scroller.add (post_redirect_list);
	post_redirect_scroller.set_policy (GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

	choice_vpacker.pack_start (pre_redirect_scroller, true, true);
	choice_vpacker.pack_start (post_redirect_scroller, true, true);


	route_choice_frame.add (choice_vpacker);

	_active_view = 0;
	_current_view = NO_CONFIG_VIEW;

	title_label.set_name ("RouteParamsTitleLabel");
	update_title();
	
	// changeable area
	route_param_frame.set_name("RouteParamsBaseFrame");
	route_param_frame.set_shadow_type (GTK_SHADOW_IN);
	
	right_hpane.add1 (route_choice_frame);
	right_hpane.add2 (route_param_frame);

	right_hpane.set_position(110);
	
	route_hpacker.pack_start (right_hpane, true, true);
	
	
	route_vpacker.pack_start (title_label, false, false);
	route_vpacker.pack_start (route_hpacker, true, true);

	
	list_hpane.add1 (list_vpacker);
	list_hpane.add2 (route_vpacker);

	list_hpane.set_position(110);
	
	global_vpacker.pack_start (list_hpane, true, true);
	
	add (global_vpacker);
	set_name ("RouteParamsWindow");
	set_default_size (620,370);
	set_title (_("ardour: route parameters"));
	set_wmclass (_("ardour_route_parameters"), "Ardour");

	// events
	route_select_list.select_row.connect (slot (*this, &RouteParams_UI::route_selected));
	route_select_list.unselect_row.connect (slot (*this, &RouteParams_UI::route_unselected));
	route_select_list.click_column.connect (slot (*this, &RouteParams_UI::show_track_menu));

	pre_redirect_list.click_column.connect (bind (slot (*this, &RouteParams_UI::show_redirect_menu), Redirect::PreFader));
	post_redirect_list.click_column.connect (bind (slot (*this, &RouteParams_UI::show_redirect_menu), Redirect::PostFader));

	pre_redirect_list.button_press_event.connect (bind (slot (*this, &RouteParams_UI::redirect_click), Redirect::PreFader));
	pre_redirect_list.button_release_event.connect (bind (slot (*this, &RouteParams_UI::redirect_click), Redirect::PreFader));
	post_redirect_list.button_press_event.connect (bind (slot (*this, &RouteParams_UI::redirect_click), Redirect::PostFader));
	post_redirect_list.button_release_event.connect (bind (slot (*this, &RouteParams_UI::redirect_click), Redirect::PostFader));

	
	pre_redirect_list.button_release_event.connect_after (slot (do_not_propagate));
	post_redirect_list.button_release_event.connect_after (slot (do_not_propagate));
	
	input_button.button_press_event.connect (slot (*this, &RouteParams_UI::edit_input_configuration));
	input_button.button_release_event.connect (slot (*this, &RouteParams_UI::edit_input_configuration));
	output_button.button_press_event.connect (slot (*this, &RouteParams_UI::edit_output_configuration));
	output_button.button_release_event.connect (slot (*this, &RouteParams_UI::edit_output_configuration));

	
	// only sensitive when a route is selected
	input_button.set_sensitive(false);
	output_button.set_sensitive(false);
	pre_redirect_list.set_sensitive(false);
	post_redirect_list.set_sensitive(false);

	add_events (GDK_KEY_PRESS_MASK|GDK_KEY_RELEASE_MASK|GDK_BUTTON_RELEASE_MASK);
	
	_plugin_manager = new PluginManager (LADSPA::manager);
	_plugin_manager->delete_event.connect (bind (slot (just_hide_it), 
						     static_cast<Window *> (_plugin_manager)));

	_plugin_manager->hide.connect(slot(*this,&RouteParams_UI::disconnect_newplug));

	delete_event.connect (bind (slot (just_hide_it), static_cast<Gtk::Window*> (this)));
}

RouteParams_UI::~RouteParams_UI ()
{
}

void
RouteParams_UI::add_route (Route& route)
{
	
	if (route.hidden()) {
		return;
	}

	const gchar *rowdata[1];
	rowdata[0] = route.name().c_str();
	route_select_list.rows().push_back (rowdata);
	route_select_list.rows().back().set_data (&route);
	//route_select_list.rows().back().select ();
	
	route.name_changed.connect (bind (slot (*this, &RouteParams_UI::route_name_changed), &route));
	route.GoingAway.connect (bind (slot (*this, &RouteParams_UI::route_removed), &route));
}


void
RouteParams_UI::route_name_changed (void *src, Route *route)
{
	CList_Helpers::RowList::iterator i;

	if ((i = route_select_list.rows().find_data (route)) == route_select_list.rows().end()) {
		error << _("route display list item for renamed route not found!") << endmsg;
		return;
	}

	route_select_list.cell ((*i)->get_row_num(), 0).set_text (route->name());

	if (route == _route) {
		track_input_label.set_text (route->name());
		update_title();
	}
}

void
RouteParams_UI::route_removed (Route *route)
{
	/*
	route_select_list.freeze ();
	route_select_list.clear ();
	session->foreach_route (this, &RouteParams_UI::add_route);
	route_select_list.thaw ();
	*/
	
	CList_Helpers::RowList::iterator i;
	
	if ((i = route_select_list.rows().find_data (route)) == route_select_list.rows().end()) {
		// couldn't find route to be deleted
		return;
	}

	if (route == _route)
	{
		route_param_frame.remove();
		if (_active_view) {
			delete _active_view;
			_active_view = 0;
		}
		pre_redirect_list.clear ();
		post_redirect_list.clear ();
		_route = 0;
		_redirect = 0;
		input_button.set_sensitive(false);
		output_button.set_sensitive(false);
		pre_redirect_list.set_sensitive(false);
		post_redirect_list.set_sensitive(false);
		update_title();
	}

	route_select_list.rows().erase(i);
	
}

void
RouteParams_UI::set_session (Session *sess)
{
	ArdourDialog::set_session (sess);

	route_select_list.freeze ();
	route_select_list.clear ();

	if (session) {
		session->foreach_route (this, &RouteParams_UI::add_route);
		session->going_away.connect (slot (*this, &ArdourDialog::session_gone));
		session->RouteAdded.connect (slot (*this, &RouteParams_UI::add_route));
		start_updating ();
	} else {
		stop_updating ();
	}

	route_select_list.thaw ();

	_plugin_manager->set_session (session);
}	

void
RouteParams_UI::session_gone ()
{

	route_select_list.clear ();
	route_param_frame.remove();
	if (_active_view) {
		delete _active_view;
		_active_view = 0;
	}
	pre_redirect_list.clear ();
	post_redirect_list.clear ();
	_route = 0;
	_redirect = 0;
	input_button.set_sensitive(false);
	output_button.set_sensitive(false);
	pre_redirect_list.set_sensitive(false);
	post_redirect_list.set_sensitive(false);
	update_title();

	ArdourDialog::session_gone();

}

void
RouteParams_UI::route_selected (gint row, gint col, GdkEvent *ev)
{
	Route *route;

	if ((route = (Route *) route_select_list.get_row_data (row)) != 0) {

		if (_route == route) {
			// do nothing
			return;
		}
		
		// remove event binding from previously selected
		if (_route) {
			_route_conn.disconnect();
			_route_ds_conn.disconnect();
			finish_current_view();
		}
	
		// update the other panes with the correct info
		_route = route;
		update_routeinfo (route);

		if (_current_view == INPUT_CONFIG_VIEW) {
			// show for new route
			IOSelector *iosel = new IOSelector (*session, _route->input_io(), true);
			iosel->redisplay ();
			set_current_view (iosel, _current_view);
		}
		else if (_current_view == OUTPUT_CONFIG_VIEW) {
			// show for new route
			IOSelector *iosel = new IOSelector (*session, _route->output_io(), false);
			iosel->redisplay ();
			set_current_view (iosel, _current_view);
		}

		// bind to redirects changed event for this route
		_route_conn = route->redirects_changed.connect (slot (*this, &RouteParams_UI::redirects_changed));

		input_button.set_sensitive(true);
		output_button.set_sensitive(true);
		pre_redirect_list.set_sensitive(true);
		post_redirect_list.set_sensitive(true);

		track_input_label.set_text (_route->name());
		
		update_title();
	}
}

void
RouteParams_UI::route_unselected (gint row, gint col, GdkEvent *ev)
{
	if (_route) {
		_route_conn.disconnect();

		// remove from view
		finish_current_view();
		pre_redirect_list.clear ();
		post_redirect_list.clear ();
		_route = 0;
		_redirect = 0;
		input_button.set_sensitive(false);
		output_button.set_sensitive(false);
		track_input_label.set_text(_("NO TRACK"));
		pre_redirect_list.set_sensitive(false);
		post_redirect_list.set_sensitive(false);
		update_title();
	}
}

void
RouteParams_UI::update_routeinfo (Route *route)
{
	// update redirects
	redirects_changed (this);

	// unselect all
	pre_redirect_list.selection().clear();
	post_redirect_list.selection().clear();
	
}

void
RouteParams_UI::redirects_changed (void *src)

{
	pre_redirect_list.freeze ();
	pre_redirect_list.clear ();
	post_redirect_list.freeze ();
	post_redirect_list.clear ();
	if (_route) {
		_route->foreach_redirect (this, &RouteParams_UI::add_redirect_to_display);
	}
	pre_redirect_list.thaw ();
	post_redirect_list.thaw ();

	_redirect = 0;
	update_title();
}


void
RouteParams_UI::add_redirect_to_display (Redirect *redirect)

{
	const gchar *rowdata[1];
	gint row;
	CList *clist = 0;

	switch (redirect->placement()) {
	case Redirect::PreFader:
		clist = &pre_redirect_list;
		break;
	case Redirect::PostFader:
		clist = &post_redirect_list;
		break;
	}

	rowdata[0] = redirect_name (*redirect).c_str();
	clist->rows().push_back (rowdata);
	row = clist->rows().size() - 1;
	clist->row (row).set_data (redirect);

	if (redirect == _redirect) {
		clist->row (row).select ();
	} 

	redirect->active_changed.connect (slot (*this, &RouteParams_UI::show_redirect_active));
}

string
RouteParams_UI::redirect_name (Redirect& redirect)
{
	Send *send;
	string name_display;

	if (!redirect.active()) {
		name_display = '(';
	}

	if ((send = dynamic_cast<Send *> (&redirect)) != 0) {
		name_display += '>';

		/* grab the send name out of its overall name */

		string::size_type lbracket, rbracket;
		lbracket = send->name().find ('[');
		rbracket = send->name().find (']');

		name_display += send->name().substr (lbracket+1, lbracket-rbracket-1);
		
	} else {
		name_display += redirect.name();
	}

	if (!redirect.active()) {
		name_display += ')';
	}

	return name_display;
}


void
RouteParams_UI::show_redirect_active (Redirect *redirect, void *src)

{
	CList_Helpers::RowIterator ri;
	CList *clist;

	if ((ri = pre_redirect_list.rows().find_data (redirect)) == pre_redirect_list.rows().end()) {
		if ((ri = post_redirect_list.rows().find_data (redirect)) == post_redirect_list.rows().end()) {
			return;
		} else {
			clist = &post_redirect_list;
		}
	} else {
		clist = &pre_redirect_list;
	}

	clist->cell(ri->get_row_num(), 0).set_text (redirect_name (*redirect));

// 	if (redirect->active()) {
// 		ri->select ();
// 	} else {
// 		ri->unselect ();
// 	}

}


gint
RouteParams_UI::redirect_click (GdkEventButton *ev, Redirect::Placement which)

{
	gint row, col;
	Redirect *redirect;
	CList *clist = 0;

	switch (which) {
	case Redirect::PreFader:
		clist = &pre_redirect_list;
		break;
	case Redirect::PostFader:
		clist = &post_redirect_list;
		break;
	}

	selected_redirect_placement = which;

	if (ev->type == GDK_BUTTON_PRESS) {
		//return stop_signal (*clist, "button-press-event");
		return FALSE;
		//return TRUE;
		//goto stop_emission;
	} else if (redirect_drag_in_progress) {
		// drag-n-drop reordering 
		return stop_signal (*clist, "button-release-event");
	}

	/* NOTE: everything below here is handling a button release, not button press */

	if (Keyboard::is_context_menu_event (ev)) {
		show_redirect_menu(0, selected_redirect_placement);

		return stop_signal (*clist, "button-release-event");
	}

	if (clist->get_selection_info ((int)ev->x, (int)ev->y, &row, &col) != 1) {
		return FALSE;
	}

	redirect = (Redirect *) clist->row (row).get_data ();
	

	if (Keyboard::is_delete_event (ev)) {

		Gtk::Main::idle.connect (bind (slot (*this, &RouteParams_UI::remove_redirect), redirect));
		return TRUE;
			
	} else if (Keyboard::is_edit_event(ev)) {

		Insert *insert;

		if ((insert = dynamic_cast<Insert *> (redirect)) == 0) {

			/* its a send */
			Send *send = dynamic_cast<Send*> (redirect);

			SendUIWindow *send_ui;
			
			if (send->get_gui() == 0) {
				
				string title;
				title = compose(_("ardour: %1"), send->name());	
					
				send_ui = new SendUIWindow (*send, *session);
				send_ui->set_title (title);
				send->set_gui (send_ui);
					
			} else {
				send_ui = reinterpret_cast<SendUIWindow *> (send->get_gui());
			}
			
			if (send_ui->is_visible()) {
				send_ui->get_window().raise ();
			} else {
				send_ui->show_all ();
			}

		} else {
				
			/* its an insert */
				
			PluginInsert *plugin_insert;
			PortInsert *port_insert;

			if ((plugin_insert = dynamic_cast<PluginInsert *> (insert)) != 0) {
					
				PluginUIWindow *plugin_ui;
					
				if (plugin_insert->get_gui() == 0) {

					string title;
						
					title = compose(_("ardour: %1: %2"), _route->name(), plugin_insert->name());
						
					plugin_ui = new PluginUIWindow (session->engine(), *plugin_insert);
					plugin_ui->set_title (title);
					plugin_insert->set_gui (plugin_ui);
				} else {
					plugin_ui = reinterpret_cast<PluginUIWindow *> (plugin_insert->get_gui());
				}

				if (plugin_ui->is_visible()) {
					plugin_ui->get_window().raise ();
				} else {
					plugin_ui->show_all ();
				}
					
			} else if ((port_insert = dynamic_cast<PortInsert *> (insert)) != 0) {

					
				PortInsertWindow *io_selector;

				if (port_insert->get_gui() == 0) {
					io_selector = new PortInsertWindow (*session, *port_insert);
					port_insert->set_gui (io_selector);
				} else {
					io_selector = reinterpret_cast<PortInsertWindow *> (port_insert->get_gui());
				}

				if (io_selector->is_visible()) {
					io_selector->get_window().raise ();
				} else {
					io_selector->show_all ();
				}
					
			}
		}

	}
	else if (ev->button == 2)
	{
		// middle click, toggle active
		redirect->set_active (!redirect->active(), this);
	}
	else {
		// regular left click, display redirect ui in view frame
			
		Insert *insert;

		if ((insert = dynamic_cast<Insert *> (redirect)) == 0) {

			Send *send;

			if ((send = dynamic_cast<Send *> (redirect)) != 0) {

				/* its a send */

				finish_current_view ();
				SendUI *send_ui = new SendUI (*send, *session);
				_plugin_conn.disconnect();
				_plugin_conn = send->GoingAway.connect (slot (*this, &RouteParams_UI::redirect_going_away));
				set_current_view (send_ui, SEND_CONFIG_VIEW);
			}

		} else {
			/* its an insert, though we don't know what kind yet. */

			PluginInsert *plugin_insert;
			PortInsert *port_insert;
				
			if ((plugin_insert = dynamic_cast<PluginInsert *> (insert)) != 0) {				

				finish_current_view();
				PluginUI *plugin_ui = new PluginUI (session->engine(), *plugin_insert, true);

				_plugin_conn.disconnect();
				_plugin_conn = plugin_insert->plugin().GoingAway.connect (slot (*this, &RouteParams_UI::plugin_going_away));
				plugin_ui->start_updating (0);
				set_current_view(plugin_ui, PLUGIN_CONFIG_VIEW);

			} else if ((port_insert = dynamic_cast<PortInsert *> (insert)) != 0) {

				finish_current_view();
				PortInsertUI *portinsert_ui = new PortInsertUI (*session, *port_insert);
					
				_plugin_conn.disconnect();
				_plugin_conn = port_insert->GoingAway.connect (slot (*this, &RouteParams_UI::redirect_going_away));
				set_current_view(portinsert_ui, PORTINSERT_CONFIG_VIEW);				
				portinsert_ui->redisplay();
			}
				
			// let emission through
			//return TRUE;
		}

		// reselect this row only
		unselect_all_redirects ();
		clist->row (row)->select ();
		_redirect = redirect;
			
		update_title();

	}

	return TRUE;
}

void
RouteParams_UI::build_redirect_menu ()
{
	using namespace Menu_Helpers;
	
	redirect_menu = new Menu;
	MenuList& items = redirect_menu->items();
	
	items.push_back (MenuElem (_("New Plugin ..."), slot (*this, &RouteParams_UI::choose_plugin)));
	items.push_back (MenuElem (_("New Insert"), slot (*this, &RouteParams_UI::choose_insert)));
	items.push_back (MenuElem (_("New Send ..."), slot (*this, &RouteParams_UI::choose_send)));
	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Activate All"), bind (slot (*this, &RouteParams_UI::all_redirects_active), true)));
	items.push_back (MenuElem (_("Deactivate All"), bind (slot (*this, &RouteParams_UI::all_redirects_active), false)));
	items.push_back (SeparatorElem());
	items.push_back (MenuElem (_("Clear"), slot (*this, &RouteParams_UI::clear_redirects)));

}

void
RouteParams_UI::show_redirect_menu (gint arg, ARDOUR::Redirect::Placement dir)
{
	selected_redirect_placement = dir;
	
	if (redirect_menu == 0) {
		build_redirect_menu ();
	}

	redirect_menu->popup (1, 0);
}

void
RouteParams_UI::show_track_menu (gint arg)
{
	using namespace Menu_Helpers;
	
	if (track_menu == 0) {
		track_menu = new Menu;
		track_menu->items().push_back 
				(MenuElem (_("Add Track/Bus"), 
					   slot (*(ARDOUR_UI::instance()), &ARDOUR_UI::add_route)));
	}
	track_menu->popup (1, 0);
}

void
RouteParams_UI::choose_plugin ()
{
	show_plugin_selector();
}

void
RouteParams_UI::insert_plugin_chosen (LADSPA::Plugin *plugin)
{
	if (plugin) {
		unsigned int count;
		unsigned int rmaxports = max(_route->input_io().n_inputs(), _route->output_io().n_outputs());
		unsigned int pmaxports = max(plugin->get_info().n_inputs, plugin->get_info().n_outputs);
		
		if (rmaxports <= pmaxports) {
			/* always use at least 1 */
			count = 1;
		}
		else {
			/* only replicate plugin such that all ports may be used */
			count = rmaxports / pmaxports;
		}

		
		Redirect *redirect = new PluginInsert (*session, *plugin, selected_redirect_placement, count);
		redirect->active_changed.connect (slot (*this, &RouteParams_UI::show_redirect_active));
		_route->add_redirect (redirect, this);
	}
}

void
RouteParams_UI::choose_insert ()
{
	Redirect *redirect = new PortInsert (*session, selected_redirect_placement);
	
	redirect->active_changed.connect (slot (*this, &RouteParams_UI::show_redirect_active));
	_route->add_redirect (redirect, this);
}

void
RouteParams_UI::choose_send ()
{
	Send *send = new Send (*session, selected_redirect_placement);

	/* start with a basic configuration of N outs, where N is route.n_outputs()
	   no inputs are needed */

	for (unsigned int n=0; n < _route->n_outputs(); n++)
	{
		send->add_output_port ("", this);
	}
		
 	send->active_changed.connect (slot (*this, &RouteParams_UI::show_redirect_active));
 	_route->add_redirect (send, this);
}

void
RouteParams_UI::all_redirects_active (bool state)
{
	_route->all_redirects_active (state);
}

void
RouteParams_UI::clear_redirects()
{
	_route->clear_redirects (this);
}


void 
RouteParams_UI::disconnect_newplug ()
{
    newplug_connection.disconnect();
}

void
RouteParams_UI::show_plugin_selector ()
{
	newplug_connection = _plugin_manager->PluginCreated.connect (slot (*this,&RouteParams_UI::insert_plugin_chosen));
	_plugin_manager->show_all ();
}


void
RouteParams_UI::set_current_view (Gtk::Container *view, RouteParams_UI::ConfigView cfgv)
{
	if (view) {
		_active_view = view;
		route_param_frame.add (*_active_view);
		_active_view->show_all ();
		_current_view = cfgv;

		switch (_current_view) {
		case OUTPUT_CONFIG_VIEW:
			input_button.set_active (false);
			break;
		case INPUT_CONFIG_VIEW:
			output_button.set_active (false);
			break;

		default:
			input_button.set_active (false);
			output_button.set_active (false);
			break;
		}
		
		update_title();
	}
}

void
RouteParams_UI::finish_current_view ()
{
	if (_active_view) {
		switch (_current_view) {
		case INPUT_CONFIG_VIEW:
		case OUTPUT_CONFIG_VIEW:
			dynamic_cast<IOSelector *>(_active_view)->Finished (IOSelector::Cancelled);
			// leave _current_view alone to save some clicks in route_selected
			break;
		case PLUGIN_CONFIG_VIEW:
			dynamic_cast<PluginUI*>(_active_view)->stop_updating (0);
			/* fallthru */
		case SEND_CONFIG_VIEW:
		default:
		   _current_view = NO_CONFIG_VIEW;
		   break;
		}

		route_param_frame.remove ();
		delete _active_view;
		_active_view = 0;
	}
}

void
RouteParams_UI::plugin_going_away (LADSPA::Plugin *plugin)

{
	// delete the current view without calling finish
	if (_active_view) {
		route_param_frame.remove();
		delete _active_view;
		_active_view = 0;
		_redirect = 0;
	}
}

void
RouteParams_UI::redirect_going_away (ARDOUR::Redirect *plugin)

{
	printf ("redirect going away\n");
	// delete the current view without calling finish
	if (_active_view) {
		Gtk::Container *av = _active_view;
		_active_view = 0;
		_current_view = NO_CONFIG_VIEW;
		_redirect = 0;
		route_param_frame.remove();
		delete av;
	}
}

gint
RouteParams_UI::edit_output_configuration (GdkEventButton *ev)
{
	if (ev->type == GDK_BUTTON_PRESS)
	{
		finish_current_view ();
		
		IOSelector *iosel = new IOSelector (*session, _route->output_io(), false);
		iosel->redisplay ();
		
		set_current_view (iosel, OUTPUT_CONFIG_VIEW);
		unselect_all_redirects ();
	}
	else {
		// force to be active (this looks weird but works)
		output_button.set_active (false);
	}
	
	return 1;
}

gint
RouteParams_UI::edit_input_configuration (GdkEventButton *ev)
{
	if (ev->type == GDK_BUTTON_PRESS)
	{
		finish_current_view ();
		
		IOSelector * iosel = new IOSelector (*session, _route->input_io(), true);
		iosel->redisplay ();
		
		set_current_view (iosel, INPUT_CONFIG_VIEW);
		unselect_all_redirects ();
	}
	else {
		// force to be active (this looks weird but works)
		input_button.set_active (false);
	}
	return 1;
}


void
RouteParams_UI::update_title ()
{
     	if (_route) {
		string title;
		title += _route->name();
		title += ": ";

		if (_redirect && (_current_view == PLUGIN_CONFIG_VIEW || _current_view == SEND_CONFIG_VIEW)) {
			title += _redirect->name();
		}
		else if (_current_view == INPUT_CONFIG_VIEW) {
			title += _("INPUT");
		}
		else if (_current_view == OUTPUT_CONFIG_VIEW) {
			title += _("OUTPUT");
		}
		
		title_label.set_text(title);

		title = _("ardour: route parameters: ") + title;
		set_title(title);
	}
	else {
		title_label.set_text(_("No Route Selected"));
		set_title(_("ardour: route parameters: no route selected"));
	}	
}

gint
RouteParams_UI::remove_redirect (Redirect *redirect)
{
	_route->remove_redirect (redirect, this);
	delete redirect;
	return FALSE;
}

void
RouteParams_UI::redirect_drag_begin (GdkDragContext *context)
{
	redirect_drag_in_progress = true;
}

void
RouteParams_UI::redirect_drag_end (GdkDragContext *context)
{
	redirect_drag_in_progress = false;
}


void
RouteParams_UI::redirects_reordered (gint src, gint dst)
{
	/* this is called before the reorder has been done, so just queue
	   something for idle time.
	*/

	Gtk::Main::idle.connect (slot (*this, &RouteParams_UI::compute_redirect_sort_keys));
}

gint
RouteParams_UI::compute_redirect_sort_keys ()
{
	using Gtk::CList_Helpers;
	CList_Helpers::RowList::iterator i;
	unsigned long sort_key;

	sort_key = 0;

	for (i = pre_redirect_list.rows().begin(); i != pre_redirect_list.rows().end(); ++i) {
		Redirect *redirect = reinterpret_cast<Redirect*> (i->get_data());
		redirect->set_sort_key (sort_key, this);
		sort_key++;
	}

	for (i = post_redirect_list.rows().begin(); i != post_redirect_list.rows().end(); ++i) {
		Redirect *redirect = reinterpret_cast<Redirect*> (i->get_data());
		redirect->set_sort_key (sort_key, this);
		sort_key++;
	}

	_route->sort_redirects ();

	return FALSE;
}


void
RouteParams_UI::unselect_all_redirects ()
{
	pre_redirect_list.selection().clear();
	post_redirect_list.selection().clear();
}

void
RouteParams_UI::start_updating ()
{
	update_connection = ARDOUR_UI::instance()->RapidScreenUpdate.connect 
		(slot (*this, &RouteParams_UI::update_views));
}

void
RouteParams_UI::stop_updating ()
{
	update_connection.disconnect();
}

void
RouteParams_UI::update_views ()
{
	SendUI *sui;

	switch (_current_view) {
	case SEND_CONFIG_VIEW:
		if ((sui = dynamic_cast<SendUI*> (_active_view)) != 0) {
			sui->update ();
		}
		break;
	default:
		break;
	}
}
