/*
    Copyright (C) 2000-2004 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: editor_ops.cc,v 1.175 2004/02/29 23:33:55 pauld Exp $
*/

#include <unistd.h>

#include <cstdlib>
#include <cmath>

#include <string>
#include <map>

#include <sndfile.h>

#include <pbd/error.h>
#include <pbd/basename.h>
#include <pbd/pthread_utils.h>

#include <gtkmmext/utils.h>
#include <gtkmmext/choice.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>
#include <ardour/audioplaylist.h>
#include <ardour/audioregion.h>
#include <ardour/diskstream.h>
#include <ardour/filesource.h>
#include <ardour/sndfilesource.h>
#include <ardour/utils.h>
#include <ardour/location.h>
#include <ardour/named_selection.h>
#include <ardour/audio_track.h>
#include <ardour/region_factory.h>
#include <ardour/reverse.h>
#include <ardour/normalize.h>

#include "ardour_ui.h"
#include "editor.h"
#include "time_axis_view.h"
#include "audio_time_axis.h"
#include "automation_time_axis.h"
#include "streamview.h"
#include "regionview.h"
#include "rgb_macros.h"
#include "extra_bind.h"
#include "selection_templates.h"
#include "selection.h"
#include "library_ui.h"
#include "editing.h"
#include "gtk-custom-hruler.h"

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;
using namespace Gtk;
using namespace Editing;

/***********************************************************************
  Editor operations
 ***********************************************************************/

void
Editor::undo (unsigned int n)
{
	if (session) {
		session->undo (n);
	}
}

void
Editor::redo (unsigned int n)
{
	if (session) {
		session->redo (n);
	}
}

int
Editor::ensure_cursor (jack_nframes_t *pos)
{
	*pos = edit_cursor->current_frame;
	return 0;
}

void
Editor::split_region ()
{
	begin_reversible_command ("split");
	
	for (AudioRegionSelection::iterator a = selection->audio_regions.begin(); a != selection->audio_regions.end(); ++a) {

		Playlist* pl = (*a)->region.playlist();
		
		if (pl) {
			session->add_undo (pl->get_memento());
			pl->split_region ((*a)->region, edit_cursor->current_frame);
			session->add_redo_no_execute (pl->get_memento());
		}
		
        }

	commit_reversible_command ();
}

void
Editor::remove_some_regions ()
{
	if (selection->audio_regions.empty()) {
		return;
	}
	
	begin_reversible_command ("remove region");

	for (AudioRegionSelection::iterator i = selection->audio_regions.begin(); i != selection->audio_regions.end(); ) {
		AudioPlaylist* playlist = dynamic_cast<AudioPlaylist*> ((*i)->region.playlist());
		AudioRegionSelection::iterator tmp;

		tmp = i;
		++tmp;

		session->add_undo (playlist->get_memento());
		playlist->remove_region (&((*i)->region));
		session->add_redo_no_execute (playlist->get_memento());

		i = tmp;
	}

	commit_reversible_command ();
}

void
Editor::remove_clicked_region ()
{
	if (clicked_audio_trackview == 0 || clicked_regionview == 0) {
		return;
	}

	Playlist* playlist = clicked_audio_trackview->playlist();
	
	begin_reversible_command ("remove region");
	session->add_undo (playlist->get_memento());
	playlist->remove_region (&clicked_regionview->region);
	session->add_redo_no_execute (playlist->get_memento());
	commit_reversible_command ();
}

void
Editor::destroy_clicked_region ()
{
	if (clicked_regionview == 0) {
		return;
	}

	session->destroy_region (&clicked_regionview->region);
}

AudioRegion *
Editor::select_region_for_operation (int dir, TimeAxisView **tv)
{
	AudioRegionView* rv;
	AudioRegion *region;
	jack_nframes_t start = 0;

	if (selection->time.start () == selection->time.end_frame ()) {
		
		/* no current selection-> is there a selected regionview? */

		if (selection->audio_regions.empty()) {
			return 0;
		}

	} 

	region = 0;

	if (!selection->audio_regions.empty()) {

		rv = *(selection->audio_regions.begin());
		(*tv) = &rv->get_time_axis_view();
		region = &rv->region;

	} else if (!selection->tracks.empty()) {

		(*tv) = selection->tracks.front();

		AudioTimeAxisView* atv;

		if ((atv = dynamic_cast<AudioTimeAxisView*> (*tv)) != 0) {
			Playlist *pl;
			
			if ((pl = atv->playlist()) == 0) {
				return 0;
			}
			
			region = dynamic_cast<AudioRegion*> (pl->top_region_at (start));
		}
	} 
	
	return region;
}
	
void
Editor::extend_selection_to_end_of_region (bool next)
{
	TimeAxisView *tv;
	Region *region;
	jack_nframes_t start;

	if ((region = select_region_for_operation (next ? 1 : 0, &tv)) == 0) {
		return;
	}

	if (region && selection->time.start () == selection->time.end_frame ()) {
		start = region->position();
	} else {
		start = selection->time.start ();
	}

	/* Try to leave the selection with the same route if possible */

	if ((tv = selection->time.track) == 0) {
		return;
	}

	begin_reversible_command ("extend selection");
	selection->set (tv, start, region->position() + region->length());
	commit_reversible_command ();
}

void
Editor::extend_selection_to_start_of_region (bool previous)
{
	TimeAxisView *tv;
	Region *region;
	jack_nframes_t end;

	if ((region = select_region_for_operation (previous ? -1 : 0, &tv)) == 0) {
		return;
	}

	if (region && selection->time.start () == selection->time.end_frame ()) {
		end = region->position() + region->length();
	} else {
		end = selection->time.end_frame ();
	}

	/* Try to leave the selection with the same route if possible */
	
	if ((tv = selection->time.track) == 0) {
		return;
	}

	begin_reversible_command ("extend selection");
	selection->set (tv, region->position(), end);
	commit_reversible_command ();
}


void
Editor::nudge_forward (bool next)
{
	jack_nframes_t distance;
	jack_nframes_t next_distance;

	if (!session) return;
	
	if (!selection->audio_regions.empty()) {

		begin_reversible_command ("nudge forward");

		for (AudioRegionSelection::iterator i = selection->audio_regions.begin(); i != selection->audio_regions.end(); ++i) {
			AudioRegion& r ((*i)->region);
			
			distance = get_nudge_distance (r.position(), next_distance);

			if (next) {
				distance = next_distance;
			}

			session->add_undo (r.playlist()->get_memento());
			r.set_position (r.position() + distance, this);
			session->add_redo_no_execute (r.playlist()->get_memento());
		}

		commit_reversible_command ();

	} else {
		distance = get_nudge_distance (playhead_cursor->current_frame, next_distance);
		session->request_locate (playhead_cursor->current_frame + distance);
	}
}
		
void
Editor::nudge_backward (bool next)
{
	jack_nframes_t distance;
	jack_nframes_t next_distance;

	if (!session) return;
	
	if (!selection->audio_regions.empty()) {

		begin_reversible_command ("nudge forward");

		for (AudioRegionSelection::iterator i = selection->audio_regions.begin(); i != selection->audio_regions.end(); ++i) {
			AudioRegion& r ((*i)->region);

			distance = get_nudge_distance (r.position(), next_distance);
			
			if (next) {
				distance = next_distance;
			}

			session->add_undo (r.playlist()->get_memento());
			
			if (r.position() > distance) {
				r.set_position (r.position() - distance, this);
			} else {
				r.set_position (0, this);
			}
			session->add_redo_no_execute (r.playlist()->get_memento());
		}

		commit_reversible_command ();

	} else {

		distance = get_nudge_distance (playhead_cursor->current_frame, next_distance);

		if (playhead_cursor->current_frame > distance) {
			session->request_locate (playhead_cursor->current_frame - distance);
		} else {
			session->request_locate (0);
		}
	}
}

/* DISPLAY MOTION */

void
Editor::move_to_start ()
{
	session->request_locate (0);
}

void
Editor::move_to_end ()
{

	session->request_locate (session->current_end_frame());
}

void
Editor::build_region_boundary_cache ()
{
	jack_nframes_t pos = 0;
	RegionPoint point;
	Region *r;

	region_boundary_cache.clear ();

	if (session == 0) {
		return;
	}

	switch (snap_type) {
	case SnapToRegionStart:
		point = Start;
		break;
	case SnapToRegionEnd:
		point = End;
		break;	
	case SnapToRegionSync:
		point = SyncPoint;
		break;	
	case SnapToRegionBoundary:
		point = Start;
		break;	
	default:
		fatal << compose (_("build_region_boundary_cache called with snap_type = %1"), snap_type) << endmsg;
		/*NOTREACHED*/
		return;
	}

	while (pos < session->current_end_frame()) {

		if ((r = find_next_region (pos, point, 1)) == 0) {
			break;
		}

		jack_nframes_t rpos;
		
		switch (snap_type) {
		case SnapToRegionStart:
			rpos = r->first_frame();
			break;
		case SnapToRegionEnd:
			rpos = r->last_frame();
			break;	
		case SnapToRegionSync:
			rpos = r->first_frame() + r->sync_offset();
			break;	
		case SnapToRegionBoundary:
			rpos = r->last_frame();
			break;	
		default:
			break;
		}

		if (region_boundary_cache.empty() || rpos != region_boundary_cache.back()) {
			if (snap_type == SnapToRegionBoundary) {
				region_boundary_cache.push_back (r->first_frame());
			}
			region_boundary_cache.push_back (rpos);
		}

		pos = rpos + 1;
	}
}

Region*
Editor::find_next_region (jack_nframes_t frame, RegionPoint point, int dir)
{
	TrackViewList::iterator i;
	jack_nframes_t closest = max_frames;
	Region* ret = 0;
	jack_nframes_t rpos = 0;

	for (i = track_views.begin(); i != track_views.end(); ++i) {

		jack_nframes_t distance;
		Region* r;

		if ((r = (*i)->find_next_region (frame, point, dir)) == 0) {
			continue;
		}

		switch (point) {
		case Start:
			rpos = r->first_frame ();
			break;

		case End:
			rpos = r->last_frame ();
			break;

		case SyncPoint:
			rpos = r->first_frame() + r->sync_offset();
			break;
		}

		if (rpos > frame) {
			distance = rpos - frame;
		} else {
			distance = frame - rpos;
		}

		if (distance < closest) {
			closest = distance;
			ret = r;
		}
	}

	return ret;
}

void
Editor::cursor_to_region_point (Cursor* cursor, RegionPoint point, int dir)
{
	Region* r;
	jack_nframes_t pos = 0;

	if (!session) {
		return;
	}

	if ((r = find_next_region (cursor->current_frame, point, dir)) == 0) {
		return;
	}
	
	switch (point){
	case Start:
		pos = r->first_frame ();
		break;

	case End:
		pos = r->last_frame ();
		break;

	case SyncPoint:
		pos = r->first_frame() + r->sync_offset();
		break;
	}
	
	if (cursor == playhead_cursor) {
		session->request_locate (pos);
	} else {
		cursor->set_position (pos);
	}
}

void
Editor::cursor_to_next_region_point (Cursor* cursor, RegionPoint point)
{
	cursor_to_region_point (cursor, point, 1);
}

void
Editor::cursor_to_previous_region_point (Cursor* cursor, RegionPoint point)
{
	cursor_to_region_point (cursor, point, -1);
}

void
Editor::cursor_to_selection_start (Cursor *cursor)
{
	jack_nframes_t pos = 0;
	switch (mouse_mode) {
	case MouseObject:
		if (!selection->audio_regions.empty()) {
			pos = selection->audio_regions.start();
		}
		break;

	case MouseRange:
		if (!selection->time.empty()) {
			pos = selection->time.start ();
		}

	default:
		return;
	}

	if (cursor == playhead_cursor) {
		session->request_locate (pos);
	} else {
		cursor->set_position (pos);
	}
}

void
Editor::cursor_to_selection_end (Cursor *cursor)
{
	jack_nframes_t pos = 0;

	switch (mouse_mode) {
	case MouseObject:
		if (!selection->audio_regions.empty()) {
			pos = selection->audio_regions.end_frame();
		}
		break;

	case MouseRange:
		if (!selection->time.empty()) {
			pos = selection->time.end_frame ();
		}

	default:
		return;
	}

	if (cursor == playhead_cursor) {
		session->request_locate (pos);
	} else {
		cursor->set_position (pos);
	}
}

void
Editor::playhead_backward ()
{
	jack_nframes_t pos;
	jack_nframes_t cnt;
	float prefix;
	bool was_floating;

	if (get_prefix (prefix, was_floating)) {
		cnt = 1;
	} else {
		if (was_floating) {
			cnt = (jack_nframes_t) floor (prefix * session->frame_rate ());
		} else {
			cnt = (jack_nframes_t) prefix;
		}
	}

	pos = playhead_cursor->current_frame;

	if ((jack_nframes_t) pos < cnt) {
		pos = 0;
	} else {
		pos -= cnt;
	}
	
	/* XXX this is completely insane. with the current buffering
	   design, we'll force a complete track buffer flush and
	   reload, just to move 1 sample !!!
	*/

	session->request_locate (pos);
}

void
Editor::playhead_forward ()
{
	jack_nframes_t pos;
	jack_nframes_t cnt;
	bool was_floating;
	float prefix;

	if (get_prefix (prefix, was_floating)) {
		cnt = 1;
	} else {
		if (was_floating) {
			cnt = (jack_nframes_t) floor (prefix * session->frame_rate ());
		} else {
			cnt = (jack_nframes_t) floor (prefix);
		}
	}

	pos = playhead_cursor->current_frame;
	
	/* XXX this is completely insane. with the current buffering
	   design, we'll force a complete track buffer flush and
	   reload, just to move 1 sample !!!
	*/

	session->request_locate (pos+cnt);
}

void
Editor::cursor_align (bool playhead_to_edit)
{
	if (playhead_to_edit) {
		if (session) {
			session->request_locate (edit_cursor->current_frame);
		}
	} else {
		edit_cursor->set_position (playhead_cursor->current_frame);
	}
}

void
Editor::edit_cursor_backward ()
{
	jack_nframes_t pos;
	jack_nframes_t cnt;
	float prefix;
	bool was_floating;

	if (get_prefix (prefix, was_floating)) {
		cnt = 1;
	} else {
		if (was_floating) {
			cnt = (jack_nframes_t) floor (prefix * session->frame_rate ());
		} else {
			cnt = (jack_nframes_t) prefix;
		}
	}

	pos = edit_cursor->current_frame;

	if ((jack_nframes_t) pos < cnt) {
		pos = 0;
	} else {
		pos -= cnt;
	}
	
	edit_cursor->set_position (pos);
}

void
Editor::edit_cursor_forward ()
{
	jack_nframes_t pos;
	jack_nframes_t cnt;
	bool was_floating;
	float prefix;

	if (get_prefix (prefix, was_floating)) {
		cnt = 1;
	} else {
		if (was_floating) {
			cnt = (jack_nframes_t) floor (prefix * session->frame_rate ());
		} else {
			cnt = (jack_nframes_t) floor (prefix);
		}
	}

	pos = edit_cursor->current_frame;
	edit_cursor->set_position (pos+cnt);
}

void
Editor::goto_frame ()
{
	float prefix;
	bool was_floating;
	jack_nframes_t frame;

	if (get_prefix (prefix, was_floating)) {
		return;
	}

	if (was_floating) {
		frame = (jack_nframes_t) floor (prefix * session->frame_rate());
	} else {
		frame = (jack_nframes_t) floor (prefix);
	}

	session->request_locate (frame);
}

void
Editor::scroll_backward ()
{
	jack_nframes_t frame;
	jack_nframes_t one_page = (jack_nframes_t) rint (canvas_width * frames_per_unit);
	bool was_floating;
	float prefix;
	jack_nframes_t cnt;
	
	if (get_prefix (prefix, was_floating)) {
		cnt = (jack_nframes_t) floor (0.8 * one_page);
	} else {
		if (was_floating) {
			cnt = (jack_nframes_t) floor (prefix * session->frame_rate());
		} else {
			cnt = (jack_nframes_t) floor (prefix * one_page);
		}
	}

	if (leftmost_frame < cnt) {
		frame = 0;
	} else {
		frame = leftmost_frame - cnt;
	}

	reposition_x_origin (frame);
}

void
Editor::scroll_forward ()
{
	jack_nframes_t frame;
	jack_nframes_t one_page = (jack_nframes_t) rint (canvas_width * frames_per_unit);
	bool was_floating;
	float prefix;
	jack_nframes_t cnt;
	
	if (get_prefix (prefix, was_floating)) {
		cnt = one_page;
	} else {
		if (was_floating) {
			cnt = (jack_nframes_t) floor (prefix * session->frame_rate());
		} else {
			cnt = (jack_nframes_t) floor (prefix * one_page);
		}
	}

	if (ULONG_MAX - cnt < leftmost_frame) {
		frame = ULONG_MAX - cnt;
	} else {
		frame = leftmost_frame + cnt;
	}

	reposition_x_origin (frame);
}

void
Editor::scroll_tracks_down ()
{
	float prefix;
	bool was_floating;
	int cnt;

	if (get_prefix (prefix, was_floating)) {
		cnt = 1;
	} else {
		cnt = (int) floor (prefix);
	}

	Gtk::Adjustment *adj = track_canvas_scroller.get_vadjustment();
	adj->set_value (adj->get_value() + (cnt * adj->get_page_size()));
}

void
Editor::scroll_tracks_up ()
{
	float prefix;
	bool was_floating;
	int cnt;

	if (get_prefix (prefix, was_floating)) {
		cnt = 1;
	} else {
		cnt = (int) floor (prefix);
	}

	Gtk::Adjustment *adj = track_canvas_scroller.get_vadjustment();
	adj->set_value (adj->get_value() - (cnt * adj->get_page_size()));
}

/* ZOOM */

void
Editor::temporal_zoom_step (bool coarser)
{
	double nfpu;

	nfpu = frames_per_unit;
	
	if (coarser) { 
		nfpu *= 2.0;
	} else { 
		nfpu = max(1.0,(nfpu/2.0));
	}

	temporal_zoom (nfpu);
}	

void
Editor::temporal_zoom (gdouble fpu)
{
	if (!session) return;
	
	jack_nframes_t current_page = current_page_frames();
	jack_nframes_t current_leftmost = leftmost_frame;
	jack_nframes_t current_rightmost;
	jack_nframes_t current_center;
	jack_nframes_t new_page;
	jack_nframes_t leftmost_after_zoom = 0;
	double nfpu;

	nfpu = fpu;
	
	new_page = (jack_nframes_t) floor (canvas_width * nfpu);

	switch (zoom_focus) {
	case ZoomFocusLeft:
		leftmost_after_zoom = current_leftmost;
		break;
		
	case ZoomFocusRight:
		current_rightmost = leftmost_frame + current_page;
		if (current_rightmost > new_page) {
			leftmost_after_zoom = current_rightmost - new_page;
		} else {
			leftmost_after_zoom = 0;
		}
		break;
		
	case ZoomFocusCenter:
		current_center = current_leftmost + (current_page/2); 
		if (current_center > (new_page/2)) {
			leftmost_after_zoom = current_center - (new_page / 2);
		} else {
			leftmost_after_zoom = 0;
		}
		break;
		
	case ZoomFocusPlayhead:
		/* try to keep the playhead in the center */
		if (playhead_cursor->current_frame > new_page/2) {
			leftmost_after_zoom = playhead_cursor->current_frame - (new_page/2);
		} else {
			leftmost_after_zoom = 0;
		}
		break;

	case ZoomFocusEdit:
		/* try to keep the edit cursor in the center */
		if (edit_cursor->current_frame > leftmost_frame + (new_page/2)) {
			leftmost_after_zoom = edit_cursor->current_frame - (new_page/2);
		} else {
			leftmost_after_zoom = 0;
		}
		break;
		
	}
 
	// leftmost_after_zoom = min (leftmost_after_zoom, session->current_end_frame());

	begin_reversible_command ("zoom");
	session->add_undo (bind (slot (*this, &Editor::reposition_and_zoom), current_leftmost, frames_per_unit));
	session->add_redo (bind (slot (*this, &Editor::reposition_and_zoom), leftmost_after_zoom, nfpu));
	commit_reversible_command ();
}	

void
Editor::temporal_zoom_selection ()
{
	if (!selection) return;
	
	if (selection->time.empty()) {
		return;
	}

	jack_nframes_t start = selection->time[clicked_selection].start;
	jack_nframes_t end = selection->time[clicked_selection].end;

	temporal_zoom_by_frame (start, end, "zoom to selection");
}

void
Editor::temporal_zoom_session ()
{
	if (session) {
		temporal_zoom_by_frame (0, session->current_end_frame(), "zoom to session");
	}
}

void
Editor::temporal_zoom_by_frame (jack_nframes_t start, jack_nframes_t end, string op)
{
	if (!session) return;

	if ((start == 0 && end == 0) || end < start) {
		return;
	}

	jack_nframes_t range = end - start;

	double new_fpu = range / canvas_width;
// 	double p2 = 1.0;

// 	while (p2 < new_fpu) {
// 		p2 *= 2.0;
// 	}
// 	new_fpu = p2;
	
	jack_nframes_t new_page = (jack_nframes_t) floor (canvas_width * new_fpu);
	jack_nframes_t middle = start + (range / 2 );
	jack_nframes_t new_leftmost = middle - (new_page/2);

	if (new_leftmost > middle) new_leftmost = 0;

	begin_reversible_command (op);
	session->add_undo (bind (slot (*this, &Editor::reposition_and_zoom), leftmost_frame, frames_per_unit));
	session->add_redo (bind (slot (*this, &Editor::reposition_and_zoom), new_leftmost, new_fpu));
	commit_reversible_command ();
}

void 
Editor::temporal_zoom_to_frame (bool coarser, jack_nframes_t frame)
{
	if (!session) return;
	
	jack_nframes_t range_before = frame - leftmost_frame;
	double new_fpu;
	
	new_fpu = frames_per_unit;
	
	if (coarser) { 
		new_fpu *= 2.0;
		range_before *= 2;
	} else { 
		new_fpu = max(1.0,(new_fpu/2.0));
		range_before /= 2;
	}

	if (new_fpu == frames_per_unit) return;

	jack_nframes_t new_leftmost = frame - range_before;

	if (new_leftmost > frame) new_leftmost = 0;

	begin_reversible_command ("zoom to frame");
	session->add_undo (bind (slot (*this, &Editor::reposition_and_zoom), leftmost_frame, frames_per_unit));
	session->add_redo (bind (slot (*this, &Editor::reposition_and_zoom), new_leftmost, new_fpu));
	commit_reversible_command ();

}

void
Editor::select_all_within (jack_nframes_t start, jack_nframes_t end, double top, double bot, bool add)
{
	list<Selectable *> touched;
	
	for (TrackViewList::iterator iter = track_views.begin(); iter != track_views.end(); ++iter) {
		(*iter)->get_selectables (start, end, top, bot, touched);
	}

	if (add) {
		selection->add (touched);
	} else {
		selection->set (touched);
	}

	flush_track_canvas ();
}

void
Editor::amplitude_zoom_step (bool in)
{
	gdouble zoom = 1.0;

	if (in) {
		zoom *= 2.0;
	} else {
		if (zoom > 2.0) {
			zoom /= 2.0;
		} else {
			zoom = 1.0;
		}
	}

#ifdef FIX_FOR_CANVAS
	/* XXX DO SOMETHING */
#endif
}	


/* DELETION */


void
Editor::delete_sample_forward ()
{
}

void
Editor::delete_sample_backward ()
{
}

void
Editor::delete_screen ()
{
}

/* SEARCH */

void
Editor::search_backwards ()
{
	/* what ? */
}

void
Editor::search_forwards ()
{
	/* what ? */
}

/* MARKS */

void
Editor::jump_forward_to_mark ()
{
	if (!session) {
		return;
	}
	
	Location *location = session->locations()->first_location_after (playhead_cursor->current_frame);

	if (location) {
		session->request_locate (location->start(), false);
	} else {
		session->request_locate (session->current_end_frame());
	}
}

void
Editor::jump_backward_to_mark ()
{
	if (!session) {
		return;
	}

	Location *location = session->locations()->first_location_before (playhead_cursor->current_frame);
	
	if (location) {
		session->request_locate (location->start());
	} else {
		session->request_locate (0);
	}
}

void
Editor::set_mark ()
{
	jack_nframes_t pos;
	float prefix;
	bool was_floating;

	if (get_prefix (prefix, was_floating)) {
		pos = session->audible_frame ();
	} else {
		if (was_floating) {
			pos = (jack_nframes_t) floor (prefix * session->frame_rate ());
		} else {
			pos = (jack_nframes_t) floor (prefix);
		}
	}

	session->locations()->add (new Location (pos, 0, "mark", Location::IsMark), true);
}

void
Editor::clear_markers ()
{
	session->begin_reversible_command (_("clear markers"));
	session->add_undo (session->locations()->get_memento());
	session->locations()->clear_markers ();
	session->add_redo_no_execute (session->locations()->get_memento());
	session->commit_reversible_command ();
}

void
Editor::clear_ranges ()
{
	session->begin_reversible_command (_("clear ranges"));
	session->add_undo (session->locations()->get_memento());

	Location * looploc = session->locations()->auto_loop_location();
	Location * punchloc = session->locations()->auto_punch_location();
	
	session->locations()->clear_ranges ();
	// re-add these
	if (looploc) session->locations()->add (looploc);
	if (punchloc) session->locations()->add (punchloc);

	session->add_redo_no_execute (session->locations()->get_memento());
	session->commit_reversible_command ();
}

void
Editor::clear_locations ()
{
	session->begin_reversible_command (_("clear locations"));
	session->add_undo (session->locations()->get_memento());
	session->locations()->clear ();
	session->add_redo_no_execute (session->locations()->get_memento());
	session->commit_reversible_command ();
	session->locations()->clear ();
}

/* INSERT/REPLACE */

void
Editor::insert_region_list_drag (AudioRegion& region)
{
	gint x, y;
	double wx, wy;
	double cx, cy;
	TimeAxisView *tv;
	jack_nframes_t where;
	AudioTimeAxisView *atv = 0;
	Playlist *playlist;
	
	track_canvas->get_pointer (x, y);

	gtk_canvas_window_to_world (GTK_CANVAS(track_gtk_canvas), x, y, &wx, &wy);
	
	GdkEvent event;
	event.type = GDK_BUTTON_RELEASE;
	event.button.x = wx;
	event.button.y = wy;
	
	where = event_frame (&event, &cx, &cy);

	if (where < leftmost_frame || where > leftmost_frame + current_page_frames()) {
		/* clearly outside canvas area */
		return;
	}
	
	if ((tv = trackview_by_y_position (cy)) == 0) {
		return;
	}
	
	if ((atv = dynamic_cast<AudioTimeAxisView*>(tv)) == 0) {
		return;
	}

	if ((playlist = atv->playlist()) == 0) {
		return;
	}
	
	snap_to (where);
	
	begin_reversible_command ("insert dragged region");
	session->add_undo (playlist->get_memento());
	playlist->add_region (*(new AudioRegion (region)), where, 1.0);
	session->add_redo_no_execute (playlist->get_memento());
	commit_reversible_command ();
}

void
Editor::insert_region_list_selection (float times)
{
	AudioTimeAxisView *tv = 0;
	Playlist *playlist;

	if (clicked_audio_trackview != 0) {
		tv = clicked_audio_trackview;
	} else if (!selection->tracks.empty()) {
		if ((tv = dynamic_cast<AudioTimeAxisView*>(selection->tracks.front())) == 0) {
			return;
		}
	} else {
		return;
	}

	if ((playlist = tv->playlist()) == 0) {
		return;
	}
	
	Gtk::CTree_Helpers::SelectionList& selected = region_list_display.selection();
	
	if (selected.empty()) {
		return;
	}
	
	Region* region = reinterpret_cast<Region *> (selected.front().get_data ());

	begin_reversible_command ("insert region");
	session->add_undo (playlist->get_memento());
	playlist->add_region (*(createRegion (*region)), edit_cursor->current_frame, times);
	session->add_redo_no_execute (playlist->get_memento());
	commit_reversible_command ();
}


/* BUILT-IN EFFECTS */

void
Editor::reverse_selection ()
{

}

/* GAIN ENVELOPE EDITING */

void
Editor::edit_envelope ()
{
}

/* PLAYBACK */

void
Editor::toggle_playback (bool with_abort)
{
	if (session) {
		if (session->transport_rolling()) {
			session->request_stop (with_abort);
		} else {
			session->request_roll ();
		}
	}
}

void
Editor::play_from_start ()
{
	session->request_locate (0, true);
}

void
Editor::play_selection ()
{
	if (selection->time.empty()) {
		return;
	}

	session->request_locate (selection->time[0].start, false);
	session->request_play_range (true);
	session->request_roll ();
}

void 
Editor::play_clicked_regionview ()
{
	if (clicked_regionview == 0) {
		return;
	}

	session->request_bounded_roll (clicked_regionview->region.position(), clicked_regionview->region.last_frame());
}

void
Editor::play_selected_region ()
{
	if (!selection->audio_regions.empty()) {
		AudioRegionView *rv = *(selection->audio_regions.begin());

		session->request_bounded_roll (rv->region.position(), rv->region.last_frame());	
	}
}

void 
Editor::loop_clicked_regionview ()
{
	if (clicked_regionview == 0) {
		return;
	}

	if (transport_loop_location) {
		transport_loop_location->set (clicked_regionview->region.position(), clicked_regionview->region.last_frame());

		// enable looping, reposition and start rolling
		session->request_auto_loop (true);
		session->request_locate (transport_loop_location->start(), false);
		session->request_roll ();
	}
}

void
Editor::loop_selected_region ()
{
	if (!selection->audio_regions.empty()) {
		AudioRegionView *rv = *(selection->audio_regions.begin());

		if (transport_loop_location) {
			transport_loop_location->set (rv->region.position(), rv->region.last_frame());
			
			// enable looping, reposition and start rolling
			session->request_auto_loop (true);
			session->request_locate (transport_loop_location->start(), false);
			session->request_roll ();
		}
	}
}

void
Editor::play_location ()
{
	Location *location;

	if ((location = session->locations()->current()) == 0)  {
		return;
	}

	session->request_bounded_roll (location->start(), location->end());
}

void
Editor::loop_location ()
{
	Location *location;

	if ((location = session->locations()->current()) == 0)  {
		return;
	}

	if (transport_loop_location) {
		transport_loop_location->set (location->start(), location->end());

		// enable looping, reposition and start rolling
		session->request_auto_loop(true);
		session->request_locate (transport_loop_location->start(), true);
	}
	
}

void 
Editor::toggle_region_mute ()
{
	if (clicked_regionview) {
		clicked_regionview->region.set_muted (!clicked_regionview->region.muted());
	} else if (!selection->audio_regions.empty()) {
		bool yn = ! (*selection->audio_regions.begin())->region.muted();
		selection->foreach_audio_region (&AudioRegion::set_muted, yn);
	}
}

void
Editor::toggle_region_opaque ()
{
	if (clicked_regionview) {
		clicked_regionview->region.set_opaque (!clicked_regionview->region.opaque());
	} else if (!selection->audio_regions.empty()) {
		bool yn = ! (*selection->audio_regions.begin())->region.opaque();
		selection->foreach_audio_region (&Region::set_opaque, yn);
	}
}

void
Editor::raise_region ()
{
	selection->foreach_audio_region (&Region::raise);
}

void
Editor::raise_region_to_top ()
{
	selection->foreach_audio_region (&Region::raise_to_top);
}

void
Editor::lower_region ()
{
	selection->foreach_audio_region (&Region::lower);
}

void
Editor::lower_region_to_bottom ()
{
	selection->foreach_audio_region (&Region::lower_to_bottom);
}

void
Editor::edit_region ()
{
	if (clicked_regionview == 0) {
		return;
	}
	
	clicked_regionview->show_region_editor ();
}

void
Editor::rename_region ()
{
	Dialog dialog;
	Entry  entry;
	Button ok_button (_("OK"));
	Button cancel_button (_("Cancel"));

	if (selection->audio_regions.empty()) {
		return;
	}

	dialog.set_title (_("ardour: rename region"));
	dialog.set_name ("RegionRenameWindow");
	dialog.set_usize (300, -1);
	dialog.set_position (GTK_WIN_POS_MOUSE);
	dialog.set_modal (true);

	dialog.get_vbox()->set_border_width (10);
	dialog.get_vbox()->pack_start (entry);
	dialog.get_action_area()->pack_start (ok_button);
	dialog.get_action_area()->pack_start (cancel_button);

	entry.set_name ("RegionNameDisplay");
	ok_button.set_name ("EditorGTKButton");
	cancel_button.set_name ("EditorGTKButton");

	region_renamed = false;

	entry.activate.connect (bind (slot (*this, &Editor::rename_region_finished), true));
	ok_button.clicked.connect (bind (slot (*this, &Editor::rename_region_finished), true));
	cancel_button.clicked.connect (bind (slot (*this, &Editor::rename_region_finished), false));

	/* recurse */

	dialog.show_all ();
	ARDOUR_UI::instance()->allow_focus (true);
	Main::run ();
	ARDOUR_UI::instance()->allow_focus (false);

	if (region_renamed) {
		(*selection->audio_regions.begin())->region.set_name (entry.get_text());
		redisplay_regions ();
	}
}

void
Editor::rename_region_finished (bool status)

{
	region_renamed = status;
	Main::quit ();
}

void
Editor::audition_playlist_region_via_route (AudioRegion& region, Route& route)
{
	if (session->is_auditioning()) {
		session->cancel_audition ();
	} 

	// note: some potential for creativity here, because region doesn't
	// have to belong to the playlist that Route is handling

	// bool was_soloed = route.soloed();

	route.set_solo (true, this);
	
	session->request_bounded_roll (region.position(), region.position() + region.length());
	
	/* XXX how to unset the solo state ? */
}

void
Editor::audition_playlist_region_standalone (AudioRegion& region)
{
	session->audition_region (region);
}


void *
Editor::_import_thread (void *arg)
{
	Editor *ed = (Editor *) arg;
	return ed->import_thread ();
}

void *
Editor::import_thread ()
{
	session->import_audiofile (import_status);
	return 0;
}

gint
Editor::import_progress_timeout (void *arg)

{
	import_progress_label.set_text (import_status.doing_what);

	if (import_status.doing_what == "building peak files") {
		import_progress_bar.set_activity_mode (true);
		return FALSE;
	} else {
		import_progress_bar.set_percentage (import_status.percent_done);
	}
	return !(import_status.done || import_status.cancel);
}

void
Editor::build_import_progress_window ()
{
	import_progress_window = new ArdourDialog;

	import_progress_bar.set_orientation (GTK_PROGRESS_LEFT_TO_RIGHT);
	
	import_progress_vbox.set_border_width (10);
	import_progress_vbox.set_spacing (5);
	import_progress_vbox.pack_start (import_progress_label, false, false);
	import_progress_vbox.pack_start (import_progress_bar,false, false);
	import_progress_vbox.pack_start (import_cancel_button,false, false);

	import_cancel_button.clicked.connect (slot (*this, &Editor::import_cancel_clicked));
	
	import_progress_window->set_title (_("ardour: audio import in progress"));
	import_progress_window->set_modal (true);
	import_progress_window->set_default_size (200, 100);
	import_progress_window->add (import_progress_vbox);
}

void
Editor::import_cancel_clicked ()
{
	import_status.cancel = true;
}

void
Editor::import_audio ()
{
	if (session == 0) {
		warning << _("You can't import an audiofile until you have a session loaded.") << endmsg;
		return;
	}

	SoundFileSelector& sfdb (ARDOUR_UI::instance()->get_sfdb_window());

	sfdb.run (_("Import selected"), true);

	if (sfdb.run_status()) {
		return;
	}

	sfdb.get_result (import_status.pathname, import_status.multichan);
	
	if (import_progress_window == 0) {
		build_import_progress_window ();
	}
	
	import_progress_window->show_all ();
	
	import_status.done = false;
	import_status.cancel = false;
	import_status.percent_done = 0.0;
	
	import_progress_connection = 
		Gtk::Main::timeout.connect (bind (slot (*this, &Editor::import_progress_timeout), (gpointer) 0), 100);
	
	pthread_t thr;
	
	pthread_create_and_store ("import", &thr, 0, _import_thread, this);
	pthread_detach (thr);
	
	while (!import_status.done && !import_status.cancel) {
		gtk_main_iteration ();
	}

	import_status.done = true;
	import_progress_connection.disconnect ();
	import_progress_window->hide_all ();
}

void 
Editor::embed_audio ()
{
	if (session == 0) {
		warning << _("You can't embed an audiofile until you have a session loaded.") << endmsg;
		return;
	}

	SoundFileSelector& sfdb (ARDOUR_UI::instance()->get_sfdb_window());
	sfdb.run (_("Add to External Region list"), true);

	if (sfdb.run_status()) {
		return;
	}

	SndFileSource *source = 0; /* keep g++ quiet */
	AudioRegion::SourceList sources;
	string idspec;
	string linked_path;
	SNDFILE *sf;
	SF_INFO finfo;
	bool multi;
	string path;

	sfdb.get_result (path, multi);

	/* lets see if we can link it into the session */
	
	linked_path = session->sound_dir();
	linked_path += basename (path);

	if (link (path.c_str(), linked_path.c_str()) == 0) {

		/* there are many reasons why link(2) might have failed.
		   but if it succeeds, we now have a link in the
		   session sound dir that will protect against
		   unlinking of the original path. nice.
		*/

		cerr << "we would have used " << selection << " but now its linked as " << linked_path << endl;
		path = linked_path;
	}

	memset (&finfo, 0, sizeof(finfo));

	/* note that we temporarily truncated _id at the colon */
	
	if ((sf = sf_open (path.c_str(), SFM_READ, &finfo)) == 0) {
		char errbuf[256];
		sf_error_str (0, errbuf, sizeof (errbuf) - 1);
		error << compose(_("Editor: cannot open file \"%1\" (%2)"), selection, errbuf) << endmsg;
		return;
	}
	sf_close (sf);
	sf = 0;

	if (finfo.samplerate != (int) session->frame_rate()) {
		vector<string> choices;

		choices.push_back (_("Embed it anyway"));
		choices.push_back (_("Cancel"));

		Gtkmmext::Choice rate_choice (
      _("This audiofile's sample rate doesn't match the session sample rate!"),
      choices);
		rate_choice.chosen.connect (Main::quit.slot());
		rate_choice.show_all ();

		Main::run ();

		if (rate_choice.get_choice()) {
			return;
		}
	}

	/* make the proper number of channels in the region */

	for (int n=0; n < finfo.channels; ++n)
	{
		idspec = path;
		idspec += compose(":%1", n);
		
		try {
			source = new SndFileSource (idspec.c_str());
			sources.push_back(source);
		} 
		
		catch (failed_constructor& err) {
			error << compose(_("could not open %1"), path) << endmsg;
			return;
		}
	}

	if (sources.size() > 0) {

		string region_name = basename_nosuffix (path);
		region_name += "-0";

		/* The created region isn't dropped.  It emits a signal
		   that is picked up by the session. */
		new AudioRegion (sources, 0, sources[0]->length(), region_name, 0,
						Region::Flag (Region::DefaultFlags|Region::WholeFile|Region::External));
		
		/* make sure we can see it in the list */

		Gtk::CTree_Helpers::RowList::iterator external_node;
		external_node = region_list_display.rows().begin();
		++external_node; /* its the second node, always */
		external_node->expand_recursive ();
	}
}

void
Editor::insert_sndfile ()
{
	if (clicked_trackview == 0) {
		return;
	}

	SoundFileSelector& sfdb (ARDOUR_UI::instance()->get_sfdb_window());

	sfdb.run (_("Insert selected"), true);

	if (sfdb.run_status()) {
		return;
	}

	jack_nframes_t pos;
	SndFileSource *source = 0; /* keep g++ quiet */
	AudioRegion::SourceList sources;
	string idspec;
	SNDFILE *sf;
	SF_INFO finfo;
	string path;
	bool multi;

	sfdb.get_result (path, multi);

	if (ensure_cursor (&pos)) {
		return;
	}

	memset (&finfo, 0, sizeof(finfo));

	/* note that we temporarily truncated _id at the colon */
	
	if ((sf = sf_open (path.c_str(), SFM_READ, &finfo)) == 0) {
		char errbuf[256];
		sf_error_str (0, errbuf, sizeof (errbuf) - 1);
		error << compose(_("Editor: cannot open file \"%1\" (%2)"), path, errbuf) << endmsg;
		return;
	}
	sf_close (sf);
	sf = 0;
	
	/* make the proper number of channels in the region */

	for (int n=0; n < finfo.channels; ++n)
	{
		idspec = path;
		idspec += compose(":%1", n);
		
		try {
			source = new SndFileSource (idspec.c_str());
			sources.push_back(source);
		} 
		
		catch (failed_constructor& err) {
			error << compose(_("could not open %1"), path) << endmsg;
			return;
		}
	}

	if (sources.size() > 0) {

		string region_name = basename (path.c_str());
		region_name += "-0";
		
		AudioRegion *region = new AudioRegion (sources, 0, sources[0]->length(), region_name, 
						       clicked_trackview->playlist()->top_layer(), 
						       Region::Flag (Region::DefaultFlags|Region::WholeFile|Region::External));
		
		begin_reversible_command ("insert sndfile");
		session->add_undo (clicked_trackview->playlist()->get_memento());
		clicked_trackview->playlist()->add_region (*region, pos);
		session->add_redo_no_execute (clicked_trackview->playlist()->get_memento());
		commit_reversible_command ();
	}
}

void
Editor::region_from_selection ()
{
	if (clicked_trackview == 0) {
		return;
	}

	if (selection->time.empty()) {
		return;
	}

	jack_nframes_t start = selection->time[clicked_selection].start;
	jack_nframes_t end = selection->time[clicked_selection].end;

	jack_nframes_t selection_cnt = end - start + 1;
	
	for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {

		AudioRegion *region;
		AudioRegion *current;
		Region* current_r;
		Playlist *pl;

		jack_nframes_t internal_start;
		string new_name;

		if ((pl = (*i)->playlist()) == 0) {
			continue;
		}

		if ((current_r = pl->top_region_at (start)) == 0) {
			continue;
		}

		if ((current = dynamic_cast<AudioRegion*> (current_r)) != 0) {
			internal_start = start - current->position();
			session->region_name (new_name, current->name(), true);
			region = new AudioRegion (*current, internal_start, selection_cnt, new_name);
		}
	}
}	

AudioRegion *
Editor::create_region_from_selection ()
{
	if (selection->time.empty()) {
		return 0;
	}

	jack_nframes_t start = selection->time[clicked_selection].start;
	jack_nframes_t end = selection->time[clicked_selection].end;
	
	if (clicked_trackview == 0) {
		return 0;
	}
	
	AudioRegion* current;
	Region* current_r;
	Playlist* pl;
	jack_nframes_t internal_start;
	string new_name;
	
	if ((pl = clicked_trackview->playlist()) == 0) {
		return 0;
	}

	if ((current_r = pl->top_region_at(start)) == 0) {
		return 0;
	}

	if ((current = dynamic_cast<AudioRegion*>(current_r)) == 0) {
		return 0;
	}
	
	internal_start = start - current->position();
	session->region_name (new_name, current->name(), true);
	
	return new AudioRegion (*current, internal_start, end - start + 1, new_name);
}

void
Editor::split_multichannel_region ()
{
	vector<AudioRegion*> v;

	if (!clicked_regionview || clicked_regionview->region.n_channels() < 2) {
		return;
	}

	clicked_regionview->region.separate_by_channel (*session, v);

	/* nothing else to do, really */
}

void
Editor::new_region_from_selection ()
{
	region_from_selection ();
	cancel_selection ();
}

void
Editor::separate_region_from_selection ()
{
	if (selection->time.empty()) {
		return;
	}

	vector<Playlist*> playlists;
	Playlist *playlist;
		
	if (clicked_trackview != 0) {


		if ((playlist = clicked_trackview->playlist()) == 0) {
			return;
		}

		playlists.push_back (playlist);

	} else {
		
		for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {

			AudioTimeAxisView* atv;

			if ((atv = dynamic_cast<AudioTimeAxisView*> ((*i))) != 0) {

				if (atv->is_audio_track()) {
					
					if ((playlist = atv->playlist()) != 0) {
						playlists.push_back (playlist);
					}
				}
			}
		}
	}

	if (!playlists.empty()) {

		begin_reversible_command ("separate");

		for (vector<Playlist*>::iterator i = playlists.begin(); i != playlists.end(); ++i) {
			
			session->add_undo ((*i)->get_memento());
			
			/* XXX need to consider musical time selections here at some point */

			for (list<AudioRange>::iterator t = selection->time.begin(); t != selection->time.end(); ++t) {
				(*i)->partition ((*t).start, (*t).end, true);
			}

			session->add_redo_no_execute ((*i)->get_memento());
		}

		commit_reversible_command ();
	}
}

void
Editor::crop_region_to_selection ()
{
	if (selection->time.empty()) {
		return;
	}

	vector<Playlist*> playlists;
	Playlist *playlist;

	if (clicked_trackview != 0) {

		if ((playlist = clicked_trackview->playlist()) == 0) {
			return;
		}

		playlists.push_back (playlist);

	} else {
		
		for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {

			AudioTimeAxisView* atv;

			if ((atv = dynamic_cast<AudioTimeAxisView*> ((*i))) != 0) {

				if (atv->is_audio_track()) {
					
					if ((playlist = atv->playlist()) != 0) {
						playlists.push_back (playlist);
					}
				}
			}
		}
	}

	if (!playlists.empty()) {

		jack_nframes_t start;
		jack_nframes_t end;
		jack_nframes_t cnt;

		begin_reversible_command ("trim to selection");

		for (vector<Playlist*>::iterator i = playlists.begin(); i != playlists.end(); ++i) {
			
			Region *region;
			
			start = selection->time.start();

			if ((region = (*i)->top_region_at(start)) == 0) {
				continue;
			}
			
			/* now adjust lengths to that we do the right thing
			   if the selection extends beyond the region
			*/
			
			start = max (start, region->position());
			end = min (selection->time.end_frame(), start + region->length() - 1);
			cnt = end - start + 1;

			session->add_undo ((*i)->get_memento());
			region->trim_to (start, cnt, this);
			session->add_redo_no_execute ((*i)->get_memento());
		}

		commit_reversible_command ();
	}
}		

void
Editor::region_fill_track ()
{
	jack_nframes_t end;

	if (!session || selection->audio_regions.empty()) {
		return;
	}

	end = session->current_end_frame ();

	begin_reversible_command ("region fill");

	for (AudioRegionSelection::iterator i = selection->audio_regions.begin(); i != selection->audio_regions.end(); ++i) {

		AudioRegion& region ((*i)->region);
		Playlist* pl = region.playlist();
		double times = (double) (end - region.last_frame()) / (double) region.length();

		session->add_undo (pl->get_memento());
		pl->add_region (*(new AudioRegion (region)), region.last_frame(), times);
		session->add_redo_no_execute (pl->get_memento());
	}

	commit_reversible_command ();
}

void
Editor::region_fill_selection ()
{
       	if (clicked_audio_trackview == 0 || !clicked_audio_trackview->is_audio_track()) {
		return;
	}

	if (selection->time.empty()) {
		return;
	}

	Region *region;

	Gtk::CTree_Helpers::SelectionList& selected = region_list_display.selection();
	
	if (selected.empty()) {
		return;
	}

	region = reinterpret_cast<Region *> (selected.front().get_data());

	jack_nframes_t start = selection->time[clicked_selection].start;
	jack_nframes_t end = selection->time[clicked_selection].end;

	Playlist *playlist; 

	if (selection->tracks.empty()) {
		return;
	}

	jack_nframes_t selection_length = end - start;
	float times = (float)selection_length / region->length();
	
	begin_reversible_command ("fill selection");
	
	for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {

		if ((playlist = (*i)->playlist()) == 0) {
			continue;
		}		
		
		session->add_undo (playlist->get_memento());
		playlist->add_region (*(createRegion (*region)), start, times);
		session->add_redo_no_execute (playlist->get_memento());
	}
	
	commit_reversible_command ();			
}
	
void
Editor::set_region_sync_from_edit_cursor ()
{
	if (clicked_regionview == 0) {
		return;
	}

	if (!clicked_regionview->region.covers (edit_cursor->current_frame)) {
		error << _("Place the edit cursor at the desired sync point") << endmsg;
		return;
	}

	Region& region (clicked_regionview->region);

	begin_reversible_command ("set sync from edit cursor");
	session->add_undo (region.playlist()->get_memento());
	region.set_sync_offset (edit_cursor->current_frame - region.position());
	session->add_redo_no_execute (region.playlist()->get_memento());
	commit_reversible_command ();
}

void
Editor::remove_region_sync ()
{
	if (clicked_regionview) {
		Region& region (clicked_regionview->region);
		begin_reversible_command ("remove sync");
		session->add_undo (region.playlist()->get_memento());
		region.set_sync_offset (0);
		session->add_redo_no_execute (region.playlist()->get_memento());
		commit_reversible_command ();
	}
}

void
Editor::align ()
{
	align_selection (Start, edit_cursor->current_frame);
}

void
Editor::align_relative ()
{
	align_selection_relative (Start, edit_cursor->current_frame);
}

void
Editor::align_selection_relative (RegionPoint point, jack_nframes_t position)
{
	if (selection->audio_regions.empty()) {
		return;
	}

	jack_nframes_t distance;
	jack_nframes_t pos = 0;
	int dir;
	
	Region& r ((*selection->audio_regions.begin())->region);

	switch (point) {
	case Start:
		pos = r.first_frame ();
		break;

	case End:
		pos = r.last_frame();
		break;

	case SyncPoint:
		pos = r.first_frame() + r.sync_offset();
		break;
	}

	if (pos > position) {
		distance = pos - position;
		dir = -1;
	} else {
		distance = position - pos;
		dir = 1;
	}

	begin_reversible_command ("align selection (relative)");

	for (AudioRegionSelection::iterator i = selection->audio_regions.begin(); i != selection->audio_regions.end(); ++i) {

		Region& region ((*i)->region);

		session->add_undo (region.playlist()->get_memento());
		
		if (dir > 0) {
			region.set_position (region.position() + distance, this);
		} else {
			region.set_position (region.position() - distance, this);
		}

		session->add_redo_no_execute (region.playlist()->get_memento());

	}

	commit_reversible_command ();
}

void
Editor::align_selection (RegionPoint point, jack_nframes_t position)
{
	if (selection->audio_regions.empty()) {
		return;
	}

	begin_reversible_command ("align selection");

	for (AudioRegionSelection::iterator i = selection->audio_regions.begin(); i != selection->audio_regions.end(); ++i) {
		align_region_internal ((*i)->region, point, position);
	}

	commit_reversible_command ();
}

void
Editor::align_region (Region& region, RegionPoint point, jack_nframes_t position)
{
	begin_reversible_command ("align region");
	align_region_internal (region, point, position);
	commit_reversible_command ();
}

void
Editor::align_region_internal (Region& region, RegionPoint point, jack_nframes_t position)
{
	session->add_undo (region.playlist()->get_memento());

	switch (point) {
	case SyncPoint:
		if (position > region.sync_offset()) {
			region.set_position (position - region.sync_offset(), this);
		}
		break;

	case End:
		if (position > region.length()) {
			region.set_position (position - region.length(), this);
		}
		break;

	case Start:
		region.set_position (position, this);
		break;
	}

	session->add_redo_no_execute (region.playlist()->get_memento());
}	

void
Editor::trim_region_to_edit_cursor ()
{
	if (clicked_regionview == 0) {
		return;
	}

	Region& region (clicked_regionview->region);

	begin_reversible_command ("trim to edit");
	session->add_undo (region.playlist()->get_memento());
	region.trim_end (edit_cursor->current_frame, this);
	session->add_redo_no_execute (region.playlist()->get_memento());
	commit_reversible_command ();
}

void
Editor::trim_region_from_edit_cursor ()
{
	if (clicked_regionview == 0) {
		return;
	}

	Region& region (clicked_regionview->region);

	begin_reversible_command ("trim to edit");
	session->add_undo (region.playlist()->get_memento());
	region.trim_front (edit_cursor->current_frame, this);
	session->add_redo_no_execute (region.playlist()->get_memento());
	commit_reversible_command ();
}

void
Editor::unflatten_route ()
{
	if (clicked_audio_trackview == 0 || !clicked_audio_trackview->is_audio_track()) {
		return;
	}
	
	clicked_audio_trackview->audio_track()->unfreeze ();
}

void
Editor::flatten_route ()
{
	if (clicked_audio_trackview == 0 || !clicked_audio_trackview->is_audio_track()) {
		return;
	}
	
	Playlist* playlist;

	if ((playlist = clicked_trackview->playlist()) == 0) {
		return;
	}

	track_canvas_scroller.get_window().set_cursor (GDK_WATCH);
	gdk_flush ();

	begin_reversible_command ("flatten");
	session->add_undo (playlist->get_memento());
	if (session->flatten_whole_track (*clicked_audio_trackview->audio_track()) == 0) {
		session->add_redo_no_execute (playlist->get_memento());
		commit_reversible_command ();
	}

	track_canvas_scroller.get_window().set_cursor (current_canvas_cursor);
}

void
Editor::flatten_selection ()
{
	if (selection->time.empty()) {
		return;
	}

	if (clicked_audio_trackview == 0 || !clicked_audio_trackview->is_audio_track()) {
		return;
	}

	jack_nframes_t start = selection->time[clicked_selection].start;
	jack_nframes_t end = selection->time[clicked_selection].end;
	jack_nframes_t cnt = end - start + 1;

	Playlist* playlist;

	if ((playlist = clicked_trackview->playlist()) == 0) {
		return;
	}

	begin_reversible_command ("flatten selection");
	session->add_undo (playlist->get_memento());
	session->flatten_track (*clicked_audio_trackview->audio_track(), start, cnt);
	session->add_redo_no_execute (playlist->get_memento());
	commit_reversible_command ();
}

void
Editor::cut ()
{
	cut_copy (Cut);
}

void
Editor::copy ()
{
	cut_copy (Copy);
}

void 
Editor::cut_copy (CutCopyOp op)
{
	/* only cancel selection if cut/copy is successful.*/

	string opname;

	switch (op) {
	case Cut:
		opname = _("cut");
		break;
	case Copy:
		opname = _("copy");
		break;
	case Clear:
		opname = _("clear");
		break;
	}
	
	cut_buffer->clear ();

	switch (current_mouse_mode()) {
	case MouseObject: 
		if (!selection->audio_regions.empty() || !selection->points.empty()) {

			begin_reversible_command (opname + _(" objects"));

			if (!selection->audio_regions.empty()) {
				
				cut_copy_regions (op);
				
				if (op == Cut) {
					selection->clear_audio_regions ();
				}
			}

			if (!selection->points.empty()) {
				cut_copy_points (op);

				if (op == Cut) {
					selection->clear_points ();
				}
			}

			commit_reversible_command ();	
		}
		break;
		
	case MouseRange:
		if (!selection->time.empty()) {

			begin_reversible_command (opname + _(" range"));
			cut_copy_ranges (op);
			commit_reversible_command ();

			if (op == Cut) {
				selection->clear_time ();
			}
			
		}
		break;
		
	default:
		break;
	}
}

void
Editor::cut_copy_points (CutCopyOp op)
{
	for (PointSelection::iterator i = selection->points.begin(); i != selection->points.end(); ++i) {

		AutomationTimeAxisView* atv = dynamic_cast<AutomationTimeAxisView*>(&(*i).track);

		if (atv) {
			atv->cut_copy_clear_objects (selection->points, op);
		} else {
			cerr << "point object selection does not belong to a atv\n";
		}
	}
}

void
Editor::cut_copy_regions (CutCopyOp op)
{
	typedef map<AudioPlaylist*,AudioPlaylist*> PlaylistMapping;
	PlaylistMapping pmap;
	jack_nframes_t first_position = max_frames;

	for (AudioRegionSelection::iterator x = selection->audio_regions.begin(); x != selection->audio_regions.end(); ++x) {
		first_position = min ((*x)->region.position(), first_position);
	}

	for (AudioRegionSelection::iterator x = selection->audio_regions.begin(); x != selection->audio_regions.end(); ) {

		AudioPlaylist *pl = dynamic_cast<AudioPlaylist*>((*x)->region.playlist());
		AudioPlaylist* npl;
		AudioRegionSelection::iterator tmp;
		
		tmp = x;
		++tmp;

		if (pl) {

			PlaylistMapping::iterator pi = pmap.find (pl);
			
			if (pi == pmap.end()) {
				npl = new AudioPlaylist (*session, "cutlist", true);
				pmap[pl] = npl;
			} else {
				npl = pi->second;
			}

			switch (op) {
			case Cut:
				session->add_undo (pl->get_memento());
				npl->add_region (*(new AudioRegion ((*x)->region)), (*x)->region.position() - first_position);
				pl->remove_region (&((*x)->region));
				session->add_redo_no_execute (pl->get_memento());
				break;

			case Copy:
				npl->add_region (*(new AudioRegion ((*x)->region)), (*x)->region.position() - first_position);
				break;

			case Clear:
				session->add_undo (pl->get_memento());
				pl->remove_region (&((*x)->region));
				session->add_redo_no_execute (pl->get_memento());
				break;
			}
		}

		x = tmp;
	}

	list<Playlist*> foo;

	for (PlaylistMapping::iterator i = pmap.begin(); i != pmap.end(); ++i) {
		foo.push_back (i->second);
	}

	if (!foo.empty()) {
		cut_buffer->set (foo);
	}
}

void
Editor::cut_copy_ranges (CutCopyOp op)
{
	for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {
		(*i)->cut_copy_clear (*selection, op);
	}
}

void
Editor::paste (float times)
{
	paste_internal (edit_cursor->current_frame, times);
}

void
Editor::paste_internal (jack_nframes_t position, float times)
{
	bool commit = false;

	if (cut_buffer->empty() || selection->tracks.empty()) {
		return;
	}

	if (position == max_frames) {
		position = edit_cursor->current_frame;
	}

	begin_reversible_command ("paste");

	TrackSelection::iterator i;
	size_t nth;

	for (nth = 0, i = selection->tracks.begin(); i != selection->tracks.end(); ++i, ++nth) {
		if ((*i)->paste (position, times, *cut_buffer, nth)) {
			commit = true;
		}
	}

	if (commit) {
		commit_reversible_command ();
	}
}

void
Editor::paste_named_selection (float times)
{
	using Gtk::CList_Helpers;
	Gtk::CList_Helpers::SelectionList& selected = named_selection_display.selection();
	bool commit = false;
	size_t nth;
	TrackSelection::iterator i;

	if (selected.empty() || selection->tracks.empty()) {
		return;
	}

	begin_reversible_command ("paste chunk");

	for (nth = 0, i = selection->tracks.begin(); i != selection->tracks.end(); ++i, ++nth) {
		//if ((*i)->paste (edit_cursor->current_frame, times, , nth)) {
		//commit = true;
		//}
	}

	if (commit) {
		commit_reversible_command();
	}
}

void
Editor::duplicate_some_regions (AudioRegionSelection& regions, float times)
{
	Playlist *playlist; 
	
	begin_reversible_command ("duplicate region");

	for (AudioRegionSelection::iterator i = regions.begin(); i != regions.end(); ++i) {

		Region& r ((*i)->region);
		
 		playlist = (*i)->region.playlist();
		session->add_undo (playlist->get_memento());
		playlist->duplicate (r, r.last_frame(), times);
		session->add_redo_no_execute (playlist->get_memento());

	}

	commit_reversible_command ();
}

void
Editor::duplicate_selection (float times)
{
	if (selection->time.empty()) {
		return;
	}

	Playlist *playlist; 

	Region *region = create_region_from_selection ();

	if (region == 0) {
		return;
	}
	
	begin_reversible_command ("duplicate selection");
	
	for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {
		if ((playlist = (*i)->playlist()) == 0) {
			continue;
		}
		session->add_undo (playlist->get_memento());
		playlist->duplicate (*region, selection->time[clicked_selection].end, times);
		session->add_redo_no_execute (playlist->get_memento());
	}

	commit_reversible_command ();
}

void
Editor::center_playhead ()
{
	center_screen (playhead_cursor->current_frame);
}

void
Editor::center_edit_cursor ()
{
	center_screen (edit_cursor->current_frame);
}

void
Editor::clear_playlist (Playlist& playlist)
{
	begin_reversible_command ("clear playlist");
	session->add_undo (playlist.get_memento());
	playlist.clear ();
	session->add_redo_no_execute (playlist.get_memento());
	commit_reversible_command ();
}

void
Editor::nudge_track (bool use_edit_cursor, bool forwards)
{
	Playlist *playlist; 
	jack_nframes_t distance;
	jack_nframes_t next_distance;
	jack_nframes_t start;

	if (use_edit_cursor) {
		start = edit_cursor->current_frame;
	} else {
		start = 0;
	}

	if ((distance = get_nudge_distance (start, next_distance)) == 0) {
		return;
	}
	
	if (selection->tracks.empty()) {
		return;
	}
	
	begin_reversible_command ("nudge track");
	
	for (TrackSelection::iterator i = selection->tracks.begin(); i != selection->tracks.end(); ++i) {

		if ((playlist = (*i)->playlist()) == 0) {
			continue;
		}		
		
		session->add_undo (playlist->get_memento());
		playlist->nudge_after (start, distance, forwards);
		session->add_redo_no_execute (playlist->get_memento());
	}
	
	commit_reversible_command ();			
}

void
Editor::toggle_auto_xfade ()
{
	if (session) {
		session->set_auto_crossfade (!session->get_auto_crossfade());
	}
}

void
Editor::toggle_follow_playhead ()
{
	set_follow_playhead (!_follow_playhead);
}

void
Editor::set_xfade_visibility (bool yn)
{
	
}

void
Editor::toggle_xfade_visibility ()
{
	set_xfade_visibility (!xfade_visibility());
}

void
Editor::remove_last_capture ()
{
	vector<string> choices;
	string prompt;
	
	if (!session) {
		return;
	}

	prompt  = _("Do you really want to destroy the last capture?"
		    "\n(This is destructive and cannot be undone)");

	choices.push_back (_("Yes, destroy it."));
	choices.push_back (_("No, do nothing."));

	Gtkmmext::Choice prompter (prompt, choices);

	prompter.chosen.connect (Gtk::Main::quit.slot());
	prompter.show_all ();

	Gtk::Main::run ();

	if (prompter.get_choice() == 0) {
		session->remove_last_capture ();
	}
}

void
Editor::reverse_region ()
{
	if (!session) {
		return;
	}

	Reverse rev (*session);
	apply_filter (rev, _("reverse regions"));
}

void
Editor::normalize_region ()
{
	if (!session) {
		return;
	}

	Normalize norm (*session);
	apply_filter (norm, _("normalize regions"));
}

void
Editor::apply_filter (AudioFilter& filter, string command)
{
	if (selection->audio_regions.empty()) {
		return;
	}

	begin_reversible_command (command);

	track_canvas_scroller.get_window().set_cursor (wait_cursor);
	gdk_flush ();

	for (AudioRegionSelection::iterator r = selection->audio_regions.begin(); r != selection->audio_regions.end(); ) {

		AudioRegion& region ((*r)->region);
		Playlist* playlist = region.playlist();

		AudioRegionSelection::iterator tmp;
		
		tmp = r;
		++tmp;

		if (region.apply (filter) == 0) {

			session->add_undo (playlist->get_memento());
			playlist->replace_region (region, *(filter.results.front()), region.position());
			session->add_redo_no_execute (playlist->get_memento());
		} else {
			goto out;
		}

		r = tmp;
	}

	commit_reversible_command ();
	selection->audio_regions.clear ();

  out:
	gdk_window_set_cursor (track_canvas_scroller.get_window(), current_canvas_cursor);
}

