# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package ArchZoom::LiteCGI;

sub new ($%) {
	my $class = shift;
	my %init = @_;

	# work around Apache sending QUERY_STRING as parameter to cgi
	@ARGV = () if $ENV{SCRIPT_NAME};

	my $query_string = $ARGV[1] || $ENV{QUERY_STRING} || "";
	$query_string &&= "?$query_string";

	my $path_info = $ARGV[0] || $ENV{'PATH_INFO'} || "";
	$path_info = "/$path_info" unless $path_info =~ /^\//;

	my $script_name = $ENV{'SCRIPT_NAME'} || $0;
	$script_name = "/$script_name" unless $script_name =~ /^\//;

	my $proto = "http";
	$proto = "https" if $ENV{HTTPS} && $ENV{HTTPS} =~ /^on$/i
		|| $ENV{'SERVER_PROTOCOL'} && $ENV{'SERVER_PROTOCOL'} =~ /^https/i;
	my $default_port = $proto eq "https"? 443: 80;
	my $server_name = $ENV{SERVER_NAME} || "localhost";
	my $server_port = $ENV{SERVER_PORT} || $default_port;
	my $host = $server_port eq $default_port? $server_name:
		"$server_name:$server_port";

	my $content_charset = $init{content_charset} || "iso-8859-1";

	my $self = {
		header_sent => 0,
		script_name => $script_name,
		query_string => $query_string,
		path_info => $path_info,
		proto_host => "$proto://$host",
		user_agent => $ENV{HTTP_USER_AGENT} || "No User Agent",
		content_charset => $content_charset,
	};

	bless $self, $class;
	return $self;
}

sub header ($@) {
	my $self = shift;
	$self->{header_sent} = 1;
	return "Content-Type: text/html; charset=$self->{content_charset}\n\n";
}

sub header_once ($@) {
	my $self = shift;
	return "" if $self->{header_sent};
	return $self->header(@_);
}

sub param_hash ($) {
	my $self = shift;
	return map {
		my ($flag, $value) = split('=');
		$flag? ($flag => defined $value? $value: 1): ()
	} split(/\?|%3f|&|;/i, $self->{query_string});
}

sub param ($$) {
	my $self = shift;
	my $key = shift;
	my %params = $self->param_hash;
	return $params{$key};
}

sub path_info ($;$) {
	my $self = shift;
	$self->{path_info} = shift if defined $_[0];
	return $self->{path_info};
}

sub content_charset ($;$) {
	my $self = shift;
	$self->{content_charset} = shift if defined $_[0];
	return $self->{content_charset};
}

# default is 'absolute', args are 'full' and 'info'
sub url ($%) {
	my $self = shift;
	my %args = @_;
	my $url = "";
	$url .= $self->{proto_host} if $args{'full'};
	$url .= $self->{script_name};
	$url .= $self->{path_info} . $self->{query_string} if $args{'info'};
	return $url;
}

sub redirect ($;$) {
	my $self = shift;
	my $url = shift || $self->url(full => 1, info => 1);
	return "Status: 302 Moved\nLocation: $url\n\n";
}

sub start_debug_mode ($) {
	my $self = shift;
	if (open(STDERR, "|-") == 0) {
		my $debug_html = "";
		while (my $line = <STDIN>) {
			my $date = localtime();
			$date = sprintf("%d-%s-%02d %s", $5, $2, $3, $4)
				if $date =~ /^(\w{3}) (\w{3}) (.\d) (\d\d:\d\d:\d\d) (\d{4})$/;
			$debug_html .= qq(<p class="debug">[$date] $line</p>);
		}
		unless ($self->{header_sent}) {
			my $header = $self->header;
			chomp($header);
			print "<!-- Start-Debug: none\n";
			print $header;
			print "Misc-Header: -->\n\n";
		}
		print $debug_html;

		require Config;
		my $web_server = $ENV{SERVER_SOFTWARE} || 'none';
		my $perl_version = $Config::Config{version} || 'none';
		my $os = `uname -a` || "unknown";
		print qq(<hr>\n);
		print qq(<p class="debug">Web Server: $web_server; Perl: $perl_version\n</p>);
		print qq(<p class="debug">OS: $os</p>);
		exit;
	}
	$ENV{DEBUG} = $self->param('debug') || 1;
}

# guess whether user agent supports js
sub browser_js_support ($) {
	my $self = shift;
	my $browser = (split(/[ \/]/, $self->{user_agent}))[0];
	return $browser =~ /^Lynx$/i || $browser =~ /^E?Links$/i
		|| $browser =~ /^w3m$/i? 0: 1;
}

1;
