//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "common.h"
#include "utils.h"
#include "conf.h"

extern char tmp[MAXMAXLEN];

// Some globals

char logfile [MAXLEN];
static char tofile[MAXLEN];

timerCmd* timers  = NULL;
varData*  vars    = NULL;

char * lastValues    [ID_SETMAX] = {NULL};
int    lastValuesSize[ID_SETMAX] = {-1};

// used by Flush() command to temporary store old configuration
static mode       *flushModes   = NULL;
static type_alias *flushAliases = NULL;
static type_key   *flushAlarms  = NULL;


//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to logging
//
//////////////////////////////////////////////////////////////////////////////////

static void initFile(char* what, const char* name)
{
	// store data in first arg
        
        char *prefix = getenv("AR_TMPDIR");
        
        if (prefix) {
        	strcat(what, prefix);
        } else {
        	char *h = getenv("HOME");
		if (h) {
        		strcat(what, h);
        		strcat(what, "/.anyRemote");
		} else {			// could it ever happen ?
        		strcat(what, "/tmp");
		}
        }
        strcat(what, name);
        char *u = getenv("USER");
        if (prefix && u) {
        	strcat(what, ".");
        	strcat(what, u);
        }        
        return;
}

void initLog()
{
        if (getLog()) {
        
        	initFile(logfile, LOGFILE);
                
                printf("INFO: log file is %s\n",logfile);
                        
                // Just  truncate file
                FILE *fplog = fopen(logfile, "w");
                if (fplog) {
                	fclose(fplog);
                }      
                printConf();
        }
}

void logger(char *head, char *str)
{
	FILE *fplog;
        time_t logtime;
        char *timestr;

	if (logfile[0] == '\0') {
        	return;
        }
        
        if (getLog()) {
        	
                if (strcmp(head, "DEBUG") == 0 && getDebug() == 0) {
                    return;
                }

                fplog = fopen(logfile, "a");
                if (fplog!=NULL) {
                	if (strcmp(head, "CFG") != 0) {
                        	time(&logtime);
                                timestr = (char *) ctime(&logtime);
                                timestr[strlen(timestr)-1]=0;
                                fprintf(fplog, "[%s] - ",timestr);
                        }   
                        if(head!=NULL) {
                        	fprintf(fplog, "%s - ",head);
                        }
                        fprintf(fplog, "%s\n",str);
                
                        fclose(fplog);
                } else {
                	printf("Can't open log file >%s<\n", logfile);
                }
        }
}

void setResFile()
{
	initFile(tofile, TOFILE);
}

char* getResFile()
{
	return tofile;
}

//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to timer commands
//
//////////////////////////////////////////////////////////////////////////////////

timerCmd* findTimer(char *id, timerCmd **prev) 
{
	if (id == NULL) {
        	return NULL;
        }

	timerCmd* ptr = timers;

        while (ptr != NULL && strncmp(ptr->timer->descr,id,MTEXTLEN) != 0) {
        	if (prev != NULL) {
        		*prev = ptr;
                }
                ptr  = ptr->next;
        }

        return ptr;
}

timerCmd* createTimer(cmdItem *ci)
{
	if (ci->descr == NULL || ci->descr[0] == '\0' || ci->tdata == NULL) {
        	logger("DEBUG", "createTimer(): wrong input");                
                return NULL;
        }

        if (findTimer(ci->descr, NULL) != NULL) {
        	logger("DEBUG", "createTimer(): timer already exists");                
                return NULL;
        } 
        timerCmd *tm = calloc(1,sizeof(timerCmd));

        tm->ticks     = 0;
        tm->times     = 1;
        tm->timer     = ci;
        tm->status    = TIMER_RUN;

        // Insert in head
        if (timers == NULL) { // Insert first
        	tm->next = NULL;
        } else {
        	tm->next = timers;
        }
        timers = tm;

        return tm;
}

int cancelTimer(char *id) 
{
	DEBUG2("cancelTimer() >%s<", id);

        timerCmd *prev = NULL;
        timerCmd *tm = findTimer(id, (timerCmd **) &prev);
        if (tm == NULL) {
            logger("DEBUG", "Cant find timer to cancel!");
            return EXIT_NOK;
        }    
        if (tm == timers) { //remove in head
        	timers = tm->next;
        } else {
        	if (prev != NULL) {
                	prev->next = tm->next;
                } else {
                	logger("ERROR", "Previous item absent for non-first element");
                        return EXIT_NOK;
                }
        }       
         
	tm->timer = NULL;
        free(tm);

        return EXIT_OK;
}

void freeTimers()
{
	logger("DEBUG", "freeTimers()");
        timerCmd *tm;

        while (timers != NULL) {
        	tm     = timers;
                timers = timers->next;
                
        	tm->timer = NULL;
                free(tm);
        }
}

extern int macroCmd(char *macro, cmdParams* params);

int verifyTimers(int ticksInSec) 
{
	timerCmd *tm   = timers;

        while (tm != NULL) {
                if (tm->status == TIMER_RUN) {	// do not process paused timers
                	if (tm->ticks >= (tm->timer->tdata->timeout * ticksInSec)) {
                		DEBUG2("verifyTimers(): it is time to execute >%s<", (tm->timer->exec != NULL ? tm->timer->exec : "empty command"));

				int ret = macroCmd(tm->timer->descr,NULL);
               			if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                        		return ret;
                        	}
                
                        	tm->ticks = 0;
                
                        	if (tm->timer->tdata->times > 0) {
                        		tm->times++;
                                	if (tm->times >= tm->timer->tdata->times) {    // Cancel this timer 
                                		DEBUG2("verifyTimers(): timer %s self-canceled", tm->timer->descr);
                                
                                        	// Cancel this timer
                                        	timerCmd *tmc = tm;
                                        	tm = tm->next;

                                        	cancelTimer(tmc->timer->descr);
                                
                                        	continue;
                                	}
                        	}
                	} else {
                		tm->ticks++;
                	}
                }
                tm   = tm->next;
        }
	
        return EXIT_OK;
}

//////////////////////////////////////////////////////////////////////////////////
//
// To minimize data transfer cache results for some commands and send data only if data were changed
//
//////////////////////////////////////////////////////////////////////////////////

int isDataNew(int what, char *data, int size)
{
	if (what < 0 || what >= ID_SETMAX) {
 		return EXIT_OK;				// consider as new value
	}
        
        int isNew = EXIT_NOK;
        if (lastValues[what] == NULL) {
        	isNew = EXIT_OK;		     	// consider as new value
        } else if (lastValues[what] != NULL &&
                   (lastValuesSize[what] != size ||
                    memcmp(lastValues[what],data,lastValuesSize[what]) != 0)) {
             
        	free(lastValues[what]);
                lastValues[what]     = NULL;
                lastValuesSize[what] = -1;
                
                isNew = EXIT_OK;		     	// consider as new value
        }
        
        if (isNew == EXIT_OK) {
        	lastValues[what] = calloc(size + 1, 1);
        	memcpy(lastValues[what],data,size);
        	lastValuesSize[what] = size;
        }
        return isNew;
}

void freeCachedData()
{
	int i = 0;
        for (;i<ID_SETMAX;i++) {
        	if (lastValues[i] != NULL) {
                	free(lastValues[i]);lastValues[i] = NULL;
                }
        }
}

//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to internal "variables" handling
//
//////////////////////////////////////////////////////////////////////////////////

varData* searchVar(const char *id, int len) 
{
	
	if (id == NULL || len <= 0) {
        	return NULL;
        }
	strcpy(tmp, "searchVar() >");
	strncat(tmp,id,len);
	strcat(tmp,"<");
	logger("DEBUG", tmp);

	varData* ptr = vars;

        while (ptr != NULL && (strlen(ptr->name) != len || strncmp(ptr->name,id,len) != 0)) {
                ptr  = ptr->next;
        }
        return ptr;
}

static int addVar(const char *name, const char *val, int sz)
{
	DEBUG2("addVar() >%s<", name);
                
        varData * v = (varData *) calloc(sizeof(varData),1);
        v->name  = strdup(name);

	if (sz > 0) {
		v->value = calloc(1,sz);
		memcpy((void*)v->value, (const void *) val, sz); // can not user strdup() since val can contains binary data 
        }
        v->size = sz;
	
        // Insert in head
        if (vars == NULL) { 		// Insert first
        	v->next = NULL;
        } else {
        	v->next = vars;
        }
        vars = v;
        
	return EXIT_OK;
}

int setVar(const char *name, const char *val, int sz)
{
        if (name == NULL) {
              return EXIT_NOK;
        }
	DEBUG2("setVar() >%s->%s<", name, (val == NULL ? "NULL" : val));
        
	varData * v = searchVar(name, strlen(name));
        if (v == NULL) {		// Add variable if it not yet defined
        	return addVar(name, val, sz);
        }
        
        free(v->value); 	// Store new value in already defined variable
        if (sz > 0) {
		v->value = calloc(1,sz);
		memcpy((void*)v->value, (const void *) val, sz);
	}
        v->size = sz;

        return EXIT_OK;
}
 
void freeVars()
{
	//logger("DEBUG", "freeVars()");
        varData *vv;

        while (vars != NULL) {
        	vv   = vars;
                vars = vars->next;
                
        	free(vv->name); vv->name  = NULL;
        	free(vv->value);vv->value = NULL;
                
                free(vv);
        }
}

//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to Flush() command
//
//////////////////////////////////////////////////////////////////////////////////

extern mode       *currentMode;
extern mode       *modes;
extern type_alias *aliases;
extern type_key   *alarms;

extern int  flushConf;

int flushData() 
{
	//logger("DEBUG","flushData");
        
        flushConf = 1;
        
        flushModes = modes;
        modes = NULL;
        
        flushAliases = aliases;
        aliases = NULL;

        flushAlarms = alarms;
        alarms = NULL;
        
        currentMode = NULL;
        
        // Now we ready to load new cfg.
        printConf();
        
	return EXIT_OK;
}

void flushOldConf() {

	freeTimers();
        freeCfgEx(flushModes, flushAlarms, flushAliases);
        flushModes   = NULL;
        flushAlarms  = NULL;
        flushAliases = NULL;
}

//////////////////////////////////////////////////////////////////////////////////
