/* Copyright (C) 2002, 2003, 2004, 2005 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
/****************************************************************************
 ** ui.h extension file, included from the uic-generated form implementation.
 **
 ** If you wish to add, delete or rename slots use Qt Designer which will
 ** update this file, preserving your code. Create an init() slot in place of
 ** a constructor, and a destroy() slot in place of a destructor.
 *****************************************************************************/

#ifdef XERCES_HAS_CPP_NAMESPACE
using namespace XERCES_CPP_NAMESPACE;
#endif
#ifdef XALAN_HAS_CPP_NAMESPACE
using namespace XALAN_CPP_NAMESPACE;
#endif

void AnyMeal::helpIndex()
{
  
}

void AnyMeal::helpContents()
{

}

void AnyMeal::helpAbout()
{
  KAboutApplication aboutDialog( this, "aboutDialog", true );
  aboutDialog.exec();
  statusBar()->message( i18n( "License was displayed." ), STATUSDISPLAYTIME );
}


void AnyMeal::fileImportMM()
{
  /// Select Mealmaster file to be imported.
  assert( cookBook );

  ImportMealMasterDialog importMealMasterDialog( this );
  importMealMasterDialog.cookBook = cookBook;
  
  if ( importMealMasterDialog.exec() )
    statusBar()->message( i18n( "Imported Mealmaster file(s)." ),
                          STATUSDISPLAYTIME );
  else
    statusBar()->message( i18n( "Import aborted." ), STATUSDISPLAYTIME );
}

extern boost::shared_ptr< KSplashScreen > splash;

void AnyMeal::init()
{
  assert( splash );
  splash->message( i18n( "Initialising docbook export ..." ) );
  recipeToDocbook =
    XSLCompilerPtr
    ( new XSLCompiler( findAnyMealFile( "appdata", "scripts/recipeToDocbook.xsl" ),
                       FormatterListener::OUTPUT_METHOD_XML,
                       HTMLENCODING,
                       findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) ) );
  recipeToDocbook->setParam( "lfile", std::string( "\"file://" ) +
                             dictionaryFile() + "\"" );
  recipeToDocbook->setParam( "lname", std::string( "\"" ) + anyMealLanguage() +
                             "\"" );

  splash->message( i18n( "Initialising Mealmaster export ..." ) );
  recipeToMealMaster =
    XSLCompilerPtr
    ( new XSLCompiler( findAnyMealFile( "appdata", "scripts/recipeToMealMaster.xsl" ),
                       FormatterListener::OUTPUT_METHOD_TEXT,
                       MMENCODING,
                       findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) ) );
  recipeToMealMaster->setParam( "mfile", std::string( "\"file://" ) +
                                mealmasterMapFile() + "\"" );

  splash->message( i18n( "Initialising HTML export ..." ) );
  docbookToHTML =
    CompilerPtr
    ( new XSLCompiler( DOCBOOK"/html/docbook.xsl",
                       FormatterListener::OUTPUT_METHOD_HTML,
                       HTMLENCODING, "" ) );

  splash->message( i18n( "Initialising fast HTML ..." ) );
  recipeToHTML =
    XSLCompilerPtr
    ( new XSLCompiler( findAnyMealFile( "appdata", "scripts/recipeToHTML.xsl" ),
                       FormatterListener::OUTPUT_METHOD_HTML,
                       HTMLENCODING,
                       findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) ) );
  recipeToHTML->setParam( "lfile", std::string( "\"file://" ) +
                          dictionaryFile() + "\"" );
  recipeToHTML->setParam( "lname", std::string( "\"" ) + anyMealLanguage() +
                          "\"" );

  splash->message( i18n( "Initialising Formatted Objects export ..." ) );
  docbookToFO =
    XSLCompilerPtr
    ( new XSLCompiler( findAnyMealFile( "appdata", "scripts/docbookToFo.xsl" ),
                       FormatterListener::OUTPUT_METHOD_XML,
                       HTMLENCODING, "" ) );
  docbookToFO->setParam( "paper.type", "\"A4\"" );

  splash->message( i18n( "Initialising RecipeML export ..." ) );
  recipeToRecipeML =
    CompilerPtr
    ( new XSLCompiler( findAnyMealFile( "appdata", "scripts/recipeToRecipeML.xsl" ),
                       FormatterListener::OUTPUT_METHOD_XML,
                       HTMLENCODING,
                       findAnyMealFile( "appdata", "scripts/anymeal.xsd" ) ) );

  recipeMLToRecipe =
    XSLCompilerPtr
    ( new XSLCompiler( findAnyMealFile( "appdata",
                                        "scripts/recipeMLToRecipe.xsl" ),
                       FormatterListener::OUTPUT_METHOD_XML,
                       "UTF-8",
                       findAnyMealFile( "appdata", "scripts/recipeML.xsd" ) ) );
}


void AnyMeal::destroy()
{
  recipeMLToRecipe.reset();
  recipeToRecipeML.reset();
  docbookToFO.reset();
  recipeToHTML.reset();
  docbookToHTML.reset();
  recipeToMealMaster.reset();
  recipeToDocbook.reset();

  cookBook.reset();
  mysqlDatabase.reset();
  networkServer.reset();
  socketServer.reset();
}


void AnyMeal::fileConnect()
{
  ConnectDialog connectDialog( this, "connectDialog" );

  if ( connectDialog.exec() )
    connectDatabase( connectDialog );
  else
    statusBar()->message
      ( i18n( "Connecting to database was aborted." ), STATUSDISPLAYTIME );

}

void AnyMeal::connectDatabase( ConnectDialog &connectDialog )
{
  MySQLDatabasePtr newMySQLDatabase;

  try {

    DisplayWaitCursor w;

    MySQLNetworkPtr newNetworkServer;
    MySQLSocketPtr newSocketServer;
    MySQLServerPtr server;
    switch ( connectDialog.serverTypeCombo->currentItem() ) {
    case 0:
      newNetworkServer = MySQLNetworkPtr
        ( new MySQLNetwork( connectDialog.userCombo->currentText(),
                            connectDialog.passWordCheckBox->isChecked() ?
                            (const char *)connectDialog.passWordEdit->text() :
                            "",
                            connectDialog.serverCombo->currentText(),
                            MYSQL_PORT ) );
      server = newNetworkServer;
      break;
    case 1:
      newSocketServer = MySQLSocketPtr
        ( new MySQLSocket
          ( connectDialog.userCombo->currentText(),
            connectDialog.passWordCheckBox->isChecked() ?
            (const char *)connectDialog.passWordEdit->text() :
            "",
            connectDialog.socketCombo->currentText(),
            MySQLDaemonPtr( new MySQLDaemon
                            ( connectDialog.folderEdit->text(), 
                              connectDialog.socketCombo->currentText() ) ) ) );
      server = newSocketServer;
      break;
    default:
      assert(false);
    };

    newMySQLDatabase = MySQLDatabasePtr
      ( new MySQLDatabase
        ( server, connectDialog.databaseCombo->currentText() ) );

    // Try to connect to database.
    CookBookPtr newCookBook
      ( new CookBook
        ( newMySQLDatabase,
          findAnyMealFile( "appdata", "scripts/mysqlIn.xsl" ),
          findAnyMealFile( "appdata", "scripts/mysqlOut.xsl" ) ) );
      
    // Count recipes.
    int number = getNumRecipes( newCookBook );

    // Close existing connections.
    cookBook = newCookBook;
    mysqlDatabase = newMySQLDatabase;
    networkServer = newNetworkServer;
    socketServer = newSocketServer;

    // Close all windows.
    workSpace->closeAllWindows();
    searchDialog.reset();

    // Store data in kwallet after successful connection.
    connectDialog.updateWallet();

    // Enable/disable some qt actions.
    importMMAction->setEnabled( true );
    importRecipeMLAction->setEnabled( true );
    fileExportMealmasterAction->setEnabled( true );
    advancedSearchAction->setEnabled( true );
    editNewRecipeAction->setEnabled( true );
    databaseInfoAction->setEnabled( true );
    databaseLogAction->setEnabled( socketServer );
    fileConnectAction->setEnabled( false );
    fileDisconnectAction->setEnabled( true );
    fileDeleteAction->setEnabled( false );

    statusBar()->message
      ( i18n( "Connected to database \"%1\" containing %2 recipes." ).
        arg( connectDialog.databaseCombo->currentText() ).
        arg( number ),
        STATUSDISPLAYTIME );

  } catch ( CookBookDeprecated &e ) {
    
    try {
      
      int version = e.getVersion();

      if ( KMessageBox::warningContinueCancel
           ( this,
             i18n( "Do you want to update the database from version %1 to "
                   "version %2? It is recommended to do a backup before." ).
             arg( versionText( version ) ).
             arg( versionText( DATABASEVERSION ) ),
             i18n( "Update Database" ) ) ==
           KMessageBox::Continue ) {
        
        DisplayWaitCursor w;
         
        // View as cookbook without performing version-check and update.
        CookBook( newMySQLDatabase,
                  findAnyMealFile( "appdata", "scripts/mysqlIn.xsl" ),
                  findAnyMealFile( "appdata", "scripts/mysqlOut.xsl" ),
                  false ).update();
            
        // On success first disconnect.
        newMySQLDatabase.reset();
        
        // Retry connecting.
        connectDatabase( connectDialog );
        
      } else
        throw e;

    } catch ( Error &e ) {
      
      KMessageBox::error( this, e.what() );
      statusBar()->message( e.what(), STATUSDISPLAYTIME );

    };

  } catch ( Error &e ) {

    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );

  };
}

void AnyMeal::displayRecipe( int id )
{
  try {

    DisplayWaitCursor w;
    
    // Query recipe with specified identity number.
    std::stringstream query;
    query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
          << "<query><selection><single>" << id
          << "</single></selection></query>" << std::endl;
    
    std::ostringstream xmlOut;
    
    cookBook->getXMLLayer()->translate( query, xmlOut );
    
    // Display result.
    RecipeDisplay *recipeDisplay =
      new RecipeDisplay( workSpace, 0, WDestructiveClose );
    recipeDisplay->setRecipeToHTML
      ( recipeToHTML );
    recipeDisplay->setRecipe( id, xmlOut.str() );
    connect( this,
             SIGNAL( changeRecipeId( int, int, const QString &, const QString & ) ),
             recipeDisplay,
             SLOT( recipeIdChanged( int, int, const QString &, const QString & ) ) );
    connect( this, SIGNAL( deleteRecipeId( int ) ),
             recipeDisplay, SLOT( recipeIdDeleted( int ) ) );
    recipeDisplay->show();
    
    statusBar()->message( i18n( "Displaying recipe number %1." ).arg( id ),
                          STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::editRecipe( int id )
{

  try {

    std::ostringstream recipe;
    {
      DisplayWaitCursor w;

      std::stringstream query;
      // Query recipe with specified identity number.
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<query><selection><single>" << id
            << "</single></selection></query>" << std::endl;
      
      cookBook->getXMLLayer()->translate( query, recipe );
    };

#ifndef NDEBUG
    std::cerr << recipe.str() << std::endl;
#endif

    RecipeEditor recipeEditor( this, "recipeEditor" );
    recipeEditor.setRecipe( recipe.str() );

    if ( recipeEditor.exec() ) {
        
      DisplayWaitCursor w;

      // Replace recipe.
      std::ostringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>"
            << std::endl << "<replace>" << std::endl
            << recipeEditor.getRecipe() << std::endl
            << "<selection><single>" << id << "</single></selection>"
            << std::endl << "</replace>" << std::endl;

#ifndef NDEBUG
      std::cerr << query.str() << std::endl;
#endif

      XMLDocument xmlDocument( "" );
      xmlDocument.fromString( cookBook->getXMLLayer()->
                              translate( query.str() ) );

      int newId = atoi( xmlDocument.getDocumentElement().
                        selectNode( "recipe/id" ).getNodeText().c_str() );
      std::string newTitle( xmlDocument.getDocumentElement().
                            selectNode( "recipe/title" ).getNodeText() );

      // Inform other windows.
      std::ostringstream xmlRecipe;
      xmlRecipe << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
                << "<insert>" << std::endl
                << recipeEditor.getRecipe() << std::endl
                << "</insert>";
      emit changeRecipeId( id, newId, newTitle.c_str(),
                           xmlRecipe.str().c_str() );

      // Update statusbar.
      statusBar()->message( i18n( "Edited recipe number %1 (was %2)." ).
                            arg( newId ).arg( id ),
                            STATUSDISPLAYTIME );

    }  else
      // Update statusbar.
      statusBar()->message( i18n( "Editing of recipe was aborted." ),
                            STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
    
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
    
  };

};

void AnyMeal::exportMealMasterRecipes( const std::string &selection )
{
  try {

    QString fileName =
      checkOverWrite
      ( KFileDialog::getSaveFileName
        ( ":export",
          i18n( "*.mm *.mmf|UTF-8 Mealmaster (*.mm *.mmf)\n"
                "*|All Files (*)" ),
          this, i18n( "Export to MealMaster File" ) ), this );
                                    
    if ( fileName != QString::null ) {

      DisplayWaitCursor w;

      std::ofstream mealMasterFile( fileName, std::ios::binary );

      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<query><selection>" << selection << "</selection></query>"
            << std::endl;
        
      std::ostringstream mealMasterOut;
      
      ChainedCompiler( cookBook->getXMLLayer(),
                       recipeToMealMaster ).translate( query,
                                                       mealMasterOut );
      mealMasterFile << mealMasterOut.str();
      
      ERRORMACRO( mealMasterFile, Error, ,
                  QString( i18n( "Error writing to file \"%1\"." ) ).
                  arg( fileName ) );
      
      statusBar()->message( i18n( "Exported Mealmaster file." ),
                            STATUSDISPLAYTIME );
      
    } else
      statusBar()->message( i18n( "Export aborted." ), STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::exportDocbookRecipes( const std::string &selection )
{
  try {

    QString fileName =
      checkOverWrite
      ( KFileDialog::getSaveFileName
        ( ":export",
          i18n( "*.xml|XML (*.xml)\n"
                "*|All Files (*)" ),
          this, i18n( "Export to Docbook File" ) ), this );
                                    
    if ( fileName != QString::null ) {

      DisplayWaitCursor w;

      std::ofstream docbookFile( fileName, std::ios::binary );

      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<query><selection>" << selection << "</selection></query>"
            << std::endl;
        
      std::ostringstream docbookOut;
      
      ChainedCompiler( cookBook->getXMLLayer(),
                       recipeToDocbook ).translate( query,
                                                    docbookOut );
      docbookFile << docbookOut.str();
      
      ERRORMACRO( docbookFile, Error, ,
                  QString( i18n( "Error writing to file \"%1\"." ) ).
                  arg( fileName ) );
      
      statusBar()->message( i18n( "Exported Docbook file." ),
                            STATUSDISPLAYTIME );
      
    } else
      statusBar()->message( i18n( "Export aborted." ), STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::exportFORecipes( const std::string &selection )
{
  try {

    QString fileName =
      checkOverWrite
      ( KFileDialog::getSaveFileName
        ( ":export",
          i18n( "*.fo|FO (*.fo)\n"
                "*.xml|XML (*.xml)\n"
                "*|All Files (*)" ),
          this, i18n( "Export to FO File" ) ), this );
                                    
    if ( fileName != QString::null ) {

      DisplayWaitCursor w;

      std::ofstream foFile( fileName, std::ios::binary );

      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<query><selection>" << selection << "</selection></query>"
            << std::endl;
        
      std::ostringstream foOut;
      
      ChainedCompiler( cookBook->getXMLLayer(),
                       CompilerPtr( new ChainedCompiler
                                    ( recipeToDocbook,
                                      docbookToFO ) ) ).translate( query,
                                                                   foOut );
      foFile << foOut.str();
      
      ERRORMACRO( foFile, Error, ,
                  QString( i18n( "Error writing to file \"%1\"." ) ).
                  arg( fileName ) );
      
      statusBar()->message( i18n( "Exported FO file." ),
                            STATUSDISPLAYTIME );
      
    } else
      statusBar()->message( i18n( "Export aborted." ), STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::exportHTMLRecipes( const std::string &selection )
{
  try {

    QString fileName =
      checkOverWrite
      ( KFileDialog::getSaveFileName
        ( ":export",
          i18n( "*.html *.htm|HTML (*.html *.htm)\n"
                "*|All Files (*)" ),
          this, i18n( "Export to HTML File" ) ), this );
                                    
    if ( fileName != QString::null ) {

      DisplayWaitCursor w;

      std::ofstream htmlFile( fileName, std::ios::binary );

      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<query><selection>" << selection << "</selection></query>"
            << std::endl;
        
      std::ostringstream htmlOut;
      
      ChainedCompiler( cookBook->getXMLLayer(),
                       CompilerPtr( new ChainedCompiler
                                    ( recipeToDocbook,
                                      docbookToHTML ) ) ).translate( query,
                                                                     htmlOut );
      htmlFile << htmlOut.str();
      
      ERRORMACRO( htmlFile, Error, ,
                  QString( i18n( "Error writing to file \"%1\"." ) ).
                  arg( fileName ) );
      
      statusBar()->message( i18n( "Exported HTML file." ),
                            STATUSDISPLAYTIME );
      
    } else
      statusBar()->message( i18n( "Export aborted." ), STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::exportRecipeMLRecipes( const std::string &selection )
{
  try {

    QString fileName =
      checkOverWrite
      ( KFileDialog::getSaveFileName
        ( ":export",
          i18n( "*.xml *.recipeml|RecipeML (*.xml *.recipeml)\n"
                "*|All Files (*)" ),
          this, i18n( "Export to RecipeML File" ) ), this );
                                    
    if ( fileName != QString::null ) {

      DisplayWaitCursor w;

      std::ofstream xmlFile( fileName, std::ios::binary );

      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<query><selection>" << selection << "</selection></query>"
            << std::endl;
        
      std::ostringstream xmlOut;
      
      ChainedCompiler( cookBook->getXMLLayer(),
                       recipeToRecipeML ).translate( query, xmlOut );
      xmlFile << xmlOut.str();
      
      ERRORMACRO( xmlFile, Error, ,
                  QString( i18n( "Error writing to file \"%1\"." ) ).
                  arg( fileName ) );
      
      statusBar()->message( i18n( "Exported RecipeML file." ),
                            STATUSDISPLAYTIME );
      
    } else
      statusBar()->message( i18n( "Export aborted." ), STATUSDISPLAYTIME );
    
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::advancedSearch()
{
  try {
    
    if ( !searchDialog ) {
      searchDialog =
        boost::shared_ptr< SearchDialog >
        ( new SearchDialog( this, "searchDialog" ) );
      searchDialog->cookBook = cookBook;
      searchDialog->reloadCategories();
    };
    
    if ( searchDialog->exec() ) {

      DisplayWaitCursor w;

      XMLDocument xmlDocument( "" );
      xmlDocument.fromString( cookBook->getXMLLayer()->
                              translate( searchDialog->xmlQuery() ) );
      XMLNodeReferenceList nodeList =
        xmlDocument.getDocumentElement().selectNodes( "results/table/row" );

      // Display result.

      RecipeListDisplay *recipeListDisplay =
        new RecipeListDisplay( workSpace, 0, WDestructiveClose );
      recipeListDisplay->recipeList->setSorting( 1, true );
      for ( int i=0; i<nodeList.getLength(); i++ ) {

        XMLReference< XalanElement > item
          ( dynamic_pointer_cast< XalanElement >( nodeList.item( i ) ) );
        new RecipeListItem( recipeListDisplay->recipeList,
                            atoi( item.selectNode( "column[position()=1]" ).
                                  getNodeText().c_str() ),
                            item.selectNode( "column[position()=2]" ).
                            getNodeText().c_str(),
                            item.selectNode( "column[position()=3]" ).
                            getNodeText() == "1" );
      };

      connect( recipeListDisplay,
               SIGNAL( recipesSelected( const std::string &, bool ) ),
               this, SLOT( selectRecipes( const std::string &, bool ) ) );
      connect( recipeListDisplay,
               SIGNAL( displayRecipe( int ) ),
               this, SLOT( displayRecipe( int ) ) );
      connect( recipeListDisplay,
               SIGNAL( editRecipe( int ) ),
               this, SLOT( editRecipe( int ) ) );
      connect( recipeListDisplay,
               SIGNAL( deleteRecipes( const std::string & ) ),
               this, SLOT( deleteRecipes( const std::string & ) ) );
      connect( recipeListDisplay,
               SIGNAL( exportMealMasterRecipes( const std::string & ) ),
               this, SLOT( exportMealMasterRecipes( const std::string & ) ) );
      connect( recipeListDisplay,
               SIGNAL( exportRecipeMLRecipes( const std::string & ) ),
               this, SLOT( exportRecipeMLRecipes( const std::string & ) ) );
      connect( recipeListDisplay,
               SIGNAL( exportDocbookRecipes( const std::string & ) ),
               this, SLOT( exportDocbookRecipes( const std::string & ) ) );
      connect( recipeListDisplay,
               SIGNAL( exportFORecipes( const std::string & ) ),
               this, SLOT( exportFORecipes( const std::string & ) ) );
      connect( recipeListDisplay,
               SIGNAL( exportHTMLRecipes( const std::string & ) ),
               this, SLOT( exportHTMLRecipes( const std::string & ) ) );
      connect( this, SIGNAL( recipesSelected( const std::string &, bool ) ),
               recipeListDisplay, SLOT( selectRecipes( const std::string &, bool ) ) );
      connect( this, SIGNAL( deleteRecipeId( int ) ),
               recipeListDisplay, SLOT( recipeIdDeleted( int ) ) );
      connect( this,
               SIGNAL( changeRecipeId( int, int, const QString &, const QString & ) ),
               recipeListDisplay,
               SLOT( recipeIdChanged( int, int, const QString &, const QString & ) ) );

      recipeListDisplay->show();

      statusBar()->message( i18n( "Displaying %1 search results." ).
                            arg( nodeList.getLength() ),
                            STATUSDISPLAYTIME );
    };
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}


void AnyMeal::fileExportMealmaster()
{
  try {

    QString fileName =
      checkOverWrite
      ( KFileDialog::getSaveFileName
        ( ":export",
          i18n( "*.mm *.mmf|UTF-8 Mealmaster (*.mm *.mmf)\n"
                "*|All Files (*)" ),
          this, i18n( "Export to MealMaster File" ) ), this );
                                    
    if ( fileName != QString::null ) {

      DisplayWaitCursor w;

      std::ofstream mealMasterFile( fileName, std::ios::binary );

      const int
        numRecipes = getNumRecipes( cookBook ),
        count = 20;

      QProgressDialog progress( i18n( "Exporting Mealmaster file ... " ),
                                i18n( "&Cancel" ), numRecipes, this,
                                "progress", true );

      for ( int i=0; i<numRecipes; i+=count ) {

        progress.setProgress( i );
        ERRORMACRO( !progress.wasCanceled(), Error, ,
                    i18n( "Export aborted." ) );

        std::stringstream query;
        query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
              << "<query><selection><range><skip>" << i << "</skip><count>"
              << count << "</count></range></selection></query>" << std::endl;
        
        std::ostringstream mealMasterOut;

        ChainedCompiler( cookBook->getXMLLayer(),
                         recipeToMealMaster ).translate( query,
                                                         mealMasterOut );
        mealMasterFile << mealMasterOut.str();

        ERRORMACRO( mealMasterFile, Error, ,
                    QString( i18n( "Error writing to file \"%1\"." ) ).
                    arg( fileName ) );

      };
      
      statusBar()->message( i18n( "Exported Mealmaster file." ),
                            STATUSDISPLAYTIME );

    } else
      statusBar()->message( i18n( "Export aborted." ), STATUSDISPLAYTIME );

  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}

void AnyMeal::deleteRecipes( const std::string &selection )
{
  try {

    if ( KMessageBox::warningContinueCancel
         ( this, 
           i18n( "Do you really want to delete all "
                 "highlighted recipes?" ),
           i18n( "Delete Recipes" ) ) ==
         KMessageBox::Continue ) {

      DisplayWaitCursor w;
    
      // Delete recipe.
      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
            << "<delete><selection>" << selection << "</selection></delete>"
            << std::endl;
      std::ofstream nullDevice( "/dev/null" );
      cookBook->getXMLLayer()->translate( query, nullDevice );
      
      XMLDocument xmlDocument( "" );
      xmlDocument.fromString( cookBook->getXMLLayer()->
                              translate( query.str() ) );
      XMLNodeReferenceList nodeList =
        xmlDocument.getDocumentElement().selectNodes( "delete/id" );
      for ( int i=0; i<nodeList.getLength(); i++ )
        emit deleteRecipeId( atoi( nodeList.item( i ).
                                   getNodeText().c_str() ) );
      
      // Update statusbar.
      statusBar()->message( i18n( "Deleted recipes." ), STATUSDISPLAYTIME );

    } else
      statusBar()->message( i18n( "Deleting of recipes was aborted" ),
                            STATUSDISPLAYTIME );

  } catch ( Error &e ) {

#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );

  };
}


void AnyMeal::selectRecipes( const std::string &selection, bool state )
{
  try {
    assert( mysqlDatabase );
    
    // Store selection in database.
    std::stringstream query;
    query << "<?xml version='1.0' encoding='UTF-8'?>" << std::endl
          << "<select user='" << mysqlDatabase->getServer()->getUserName()
          << "'><selection>" << selection << "</selection><state>"
          << state << "</state></select>" << std::endl;
    std::ofstream nullDevice( "/dev/null" );
    cookBook->getXMLLayer()->translate( query, nullDevice );
    
    // Inform other windows.
    emit recipesSelected( selection, state );
  } catch ( Error &e ) {
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
  };
}


void AnyMeal::fileDelete()
{
  assert( !mysqlDatabase );

  try {

    DestroyDialog destroyDialog( this, "destroyDialog" );

    if ( destroyDialog.exec() ) {

      DisplayWaitCursor w;

      MySQLServerPtr server;
      switch ( destroyDialog.serverTypeCombo->currentItem() ) {
      case 0:
        server = MySQLServerPtr
          ( new MySQLNetwork( destroyDialog.rootUserEdit->text(),
                              destroyDialog.rootPassWordCheckBox->isChecked() ?
                              (const char *)destroyDialog.rootPassWordEdit->
                              text() : "",
                              destroyDialog.serverEdit->text(),
                              MYSQL_PORT ) );
        break;
      case 1:
        server = MySQLServerPtr
          ( new MySQLSocket
            ( destroyDialog.rootUserEdit->text(),
              destroyDialog.rootPassWordCheckBox->isChecked() ?
              (const char *)destroyDialog.rootPassWordEdit->
              text() : "",
              destroyDialog.socketEdit->text(),
              MySQLDaemonPtr( new MySQLDaemon
                              ( destroyDialog.folderEdit->text(), 
                                destroyDialog.socketEdit->text() ) ) ) );
        break;
      default:
        assert(false);
      };

      std::ostringstream destroyQuery;
      destroyQuery << "DROP DATABASE "
                   << destroyDialog.databaseEdit->text() << ';';

      ERRORMACRO( mysql_query( server->getConnection(),
                               destroyQuery.str().c_str() ) == 0,
                  Error, , i18n( "Error destroying database: %1" ).
                  arg( mysql_error( server->getConnection() ) ) );

      statusBar()->message
        ( i18n( "Database \"%1\" was deleted." ).
          arg( destroyDialog.databaseEdit->text() ),
          STATUSDISPLAYTIME );
    };

  } catch ( Error &e ) {

    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );

  };
}


int AnyMeal::getNumRecipes( CookBookPtr cookBook )
{
  std::string number;
  XMLDocument xmlDocument( "" );
  xmlDocument.fromString
    ( cookBook->getXMLLayer()->translate
      ( "<?xml version='1.0' encoding='UTF-8'?>"
        "<count><selection><all/></selection></count>" ) );
  number = xmlDocument.getDocumentElement().getNodeText();
  return atoi( number.c_str() );
}


void AnyMeal::editNewRecipe()
{
  assert( cookBook );
  
  try {
    
    RecipeEditor recipeEditor( this, "recipeEditor" );
    
    if ( recipeEditor.exec() ) {

      // Insert recipe.
      std::stringstream query;
      query << "<?xml version='1.0' encoding='UTF-8'?>"
            << std::endl << "<insert>" << std::endl
            << recipeEditor.getRecipe() << std::endl
            << std::endl << "</insert>" << std::endl;
#ifndef NDEBUG
      std::cerr << query.str() << std::endl;
#endif
      std::ofstream nullDevice( "/dev/null" );
      cookBook->getXMLLayer()->translate( query, nullDevice );

      // Update statusbar.
      statusBar()->message( i18n( "Created new recipe." ), STATUSDISPLAYTIME );

    }  else
      // Update statusbar.
      statusBar()->message( i18n( "Creating of recipe was aborted." ),
                            STATUSDISPLAYTIME );

  } catch ( Error &e ) {
    
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
    
  };
}


void AnyMeal::databaseInfo()
{
  assert( cookBook );

  try {
    XMLDocument xmlDocument( "" );
    {
      DisplayWaitCursor wait;
      xmlDocument.fromString( cookBook->getXMLLayer()->translate
                              ( "<?xml version='1.0' encoding='UTF-8'?>"
                                "<statistics/>" ) );
    };
    XMLReference< XalanElement > docElement
      ( dynamic_pointer_cast< XalanElement >
        ( xmlDocument.getDocumentElement().selectNode( "statistics" ) ) );
    
    KMessageBox::
      information( this,
                   i18n( "The database contains %1 recipes.\n"
                         "There are %2 categories.\n"
                         "The recipes are making use of %3 different "
                         "ingredients." ).
                   arg( docElement.selectNode( "recipes" ).getNodeText() ).
                   arg( docElement.selectNode( "categories" ).getNodeText() ).
                   arg( docElement.selectNode( "edibles" ).getNodeText() ),
                   i18n( "Database information" ) );
  } catch ( Error &e ) {
    
#ifndef NDEBUG
    std::cerr << e.what() << std::endl;
#endif
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
    
  };
}


void AnyMeal::fileImportRecipeML()
{
#if 0
  try {

    QStringList fileNames =
      KFileDialog::getOpenFileNames( ":import",
                                     i18n( "*.recipeml *.xml|RecipeML (*.recipeml *.xml)\n"
                                           "*|All Files (*)" ),
                                     this, i18n( "Import RecipeML File" ) );
                                    
    if ( !fileNames.empty() ) {

      std::ofstream nullDevice( "/dev/null" );

      QProgressDialog progress( i18n( "Importing RecipeML-files ... " ),
                                i18n( "&Cancel" ), fileNames.size(), this,
                                "progress", true );

      for ( int i=0; i<(signed)fileNames.size(); i++ ) {

        progress.setProgress( i );
        ERRORMACRO( !progress.wasCanceled(), Error, ,
                    i18n( "Import aborted." ) );

        try {
          std::ifstream inputStream( fileNames[i], std::ios::binary );
          progress.setLabelText( i18n( "Importing file \"%1\" ..." ).
                                 arg( fileNames[i] ) );
          
          ChainedCompiler
            ( recipeMLToRecipe,
              cookBook->getXMLLayer() ).translate( inputStream, nullDevice );

        } catch ( Error &e ) {
          
          ERRORMACRO( KMessageBox::warningContinueCancel
                      ( this,
                        i18n( "Error importing file %1: %2. "
                              "Continue importing recipes?" ).
                        arg( fileNames[i] ).
                        arg( e.what() ),
                        i18n( "Error importing recipe" ) ) ==
                      KMessageBox::Continue,
                      Error, ,
                      i18n( "Import was aborted" ) );
        };
      };
    };

  } catch ( Error &e ) {
    
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
    
  };
#else

  try {

    ImportRecipeMLDialog importRecipeMLDialog( this );
    importRecipeMLDialog.cookBook = cookBook;
    importRecipeMLDialog.recipeMLToRecipe = recipeMLToRecipe;
    if ( importRecipeMLDialog.exec() )
      statusBar()->message( i18n( "Imported RecipeML file(s)." ),
                            STATUSDISPLAYTIME );
    else
      statusBar()->message( i18n( "Import aborted." ), STATUSDISPLAYTIME );

  } catch ( Error &e ) {
    
    KMessageBox::error( this, e.what() );
    statusBar()->message( e.what(), STATUSDISPLAYTIME );
    
  };
#endif
}


void AnyMeal::databaseLog()
{
  if( socketServer )
    if ( socketServer->getDaemon() ) {
      LogWindow w( this );
#ifndef NDEBUG
      std::cerr << socketServer->getDaemon()->getLog() << std::endl;
#endif
      w.textBrowser->setText( socketServer->getDaemon()->getLog() );
      w.exec();
    };
}


void AnyMeal::fileDisconnect()
{
  DisplayWaitCursor w;
  
  // Close connection, if there is any.
  cookBook.reset();
  mysqlDatabase.reset();
  networkServer.reset();
  socketServer.reset();
  
  // Close all windows.
  workSpace->closeAllWindows();
  searchDialog.reset();

  // Enable/disable some qt actions.
  importMMAction->setEnabled( false );
  importRecipeMLAction->setEnabled( false );
  fileExportMealmasterAction->setEnabled( false );
  advancedSearchAction->setEnabled( false );
  editNewRecipeAction->setEnabled( false );
  databaseInfoAction->setEnabled( false );
  databaseLogAction->setEnabled( false );
  fileConnectAction->setEnabled( true );
  fileDisconnectAction->setEnabled( false );
  fileDeleteAction->setEnabled( true );
}
