// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#include "clp.h"
#include "poa.h"
#include <string>
#include <list>
#include <cctype>
#include <fstream>

// Create a map with all global Parameters
std::map< std::string, Parameter::param_user_t > clp::global_param_map;
// Create a map with all local Parameters
std::map< std::string, Parameter * > clp::local_param_map;

int clp::global_flags;

void clp::scanAllParameter( Parameter &po )
{
	std::map< std::string, Parameter::param_t >::iterator i;
	for ( i = po.Param().begin(); i != po.Param().end(); i++ )
	{
		if ( i->second.Typ == Parameter::G_FLAG ||       // Check is this a GLOBAL parameter.
		        i->second.Typ == Parameter::G_STRING ||
		        i->second.Typ == Parameter::G_INTEGER ||
		        i->second.Typ == Parameter::G_DATE )
		{
			if ( global_param_map.find( i->first ) != global_param_map.end() )   // Already exist?
			{
				if ( global_param_map[ i->first ].Param.Typ != i->second.Typ )
				{
					std::cerr << "Error: Can't add global parameter \"" << i->first << "\" of module \"" << po.Prefix() << "\"." << std::endl;
					std::cerr << "  The type of this global parameter is different to other" << std::endl;
					std::cerr << "  global parameters with the same name (prefix)." << std::endl;
				}
				else
				{
					global_param_map[ i->first ].User.push_back( &po );
				}
			}
			else
			{
				global_param_map[ i->first ].Param = i->second;
				global_param_map[ i->first ].User.push_back( &po );
			}
		}
		else
		{
			// Create local name with prefix.
			std::string local_name = po.Prefix();
			local_name += "-";
			local_name += i->first;

			if ( local_param_map.find( local_name ) != local_param_map.end() )   // Already exist?
			{
				std::cerr << "Error: Can't add local parameter \"" << local_name << "\" of module \"" << po.Prefix() << "\"." << std::endl;
				std::cerr << "  A local parameter with the same name already exists." << std::endl;
			}
			else
			{
				local_param_map[ local_name ] = &po;
			}
		}
	}
}

bool clp::parse( int argc, char **argv )
{
	// need new concept. read now fisrt all parameters in a universal format
	// from the commandline.
	// here some examples to understand the raw format:
	// -x                -> "-x",""
	// --test            -> "test",""
	// --test-one        -> "test-one",""
	// -fhello           -> "-f","hello"
	// -f hello          -> "-f","hello"
	// -f -hello         -> "-f","" -> "-h","ello" ;-)
	// --test=hmhm       -> "test","hmhm"
	// --test-one=hm-hm  -> "test","hm-hm" this was a bug in 0.3
	// and inlater version i try to regonize the doublequotes too:
	// --test="hm hm"    -> "test","hm hm"
	std::map< std::string, std::string > raw;

	if ( argc < 2 )   // do nothing.
	{
		// i don't know ... is it possible run anteater without any parameter?
	}
	else
	{

		argc--; // without the cmd name!

		// create a new array for all cmd line arguments as c++ strings
		std::string * arg = new std::string[ argc ];

		// copy all arguments from c char array to the new string array.
		int i;
		for ( i = 1; i < ( argc + 1 ); i++ ) arg[ i - 1 ] = argv[ i ];

		i = 0;
		do
		{
			// no argument can be less than 2 chars.
			if ( arg[ i ].size() < 2 )
				return parseError( arg[ i ], __LINE__ );

			// the first char must be a '-'.
			if ( arg[ i ][ 0 ] != '-' )
				return parseError( arg[ i ], __LINE__ );

			// is the second char a '-' too?
			if ( arg[ i ][ 1 ] != '-' )
			{
				if ( !isalpha( arg[ i ][ 1 ] ) )
					return parseError( arg[ i ], __LINE__ );

				// Add short parameter to the list.
				if ( arg[ i ].size() > 2 )
				{
					// Test for doublequotes.
					if ( arg[ i ][ 2 ] == '\"' )
					{
						// check for the closing quote in the same arg.
						// and for a size of the arg > 3.
						if ( arg[ i ][ arg[ i ].size() - 1 ] == '\"' && arg[ i ].size() > 3 )
						{
							if ( raw.find( arg[ i ].substr( 0, 2 ) ) != raw.end() )
								return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );

							raw[ arg[ i ].substr( 0, 2 ) ] = arg[ i ].substr( 3, arg[ i ].size() - 4 );
						}
						else // no " at the end or less than 3 chars.
						{
							// is this the last argument? >= paranoia ;-)
							if ( i >= argc - 1 )
								return parseError( arg[ i ] += "<- no closing doublequote!", __LINE__ );

							// check all next arg's for the second doublequote.
							int j;
							std::string value = arg[ i ].size() > 3 ? arg[ i ].substr( 3 ) : "";
							for ( j = i + 1; j < argc; j++ )
							{
								value += " ";
								if ( arg[ j ][ arg[ j ].size() - 1 ] == '\"' )
								{
									if ( arg[ j ].size() > 1 ) value += arg[ j ].substr( 0, arg[ j ].size() - 1 );
									break;
								}
								else
								{
									value += arg[ j ];
								}
							}
							// break why?
							if ( arg[ j - 1 ][ arg[ j - 1 ].size() - 1 ] != '\"' )
								return parseError( arg[ i ] += "<- no closing doublequote!", __LINE__ );

							// the string is ok, add it to the list.
							if ( raw.find( arg[ i ].substr( 0, 2 ) ) != raw.end() )
								return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );
							raw[ arg[ i ].substr( 0, 2 ) ] = value;
							i = j - 1; // this is the new index.

						} // if - check closing doublequote and size > 3

					}
					else // no doublequotes at pos 3
					{
						// add parameter with value...
						if ( raw.find( arg[ i ].substr( 0, 2 ) ) != raw.end() )
							return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );
						raw[ arg[ i ].substr( 0, 2 ) ] = arg[ i ].substr( 2 );
					}
				}
				else  // size is 2 chars or less..
				{
					// add parameter without a value.
					if ( raw.find( arg[ i ] ) != raw.end() )
						return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );
					raw[ arg[ i ] ] = "";
				}

			}
			else // now the parameter is something like "--xxxx"
			{
				if ( arg[ i ].size() < 3 )   // only "--" ?
					return parseError( arg[ i ], __LINE__ );

				// search for a "=" between the start and the first char not
				// isalnum or -
				std::string::size_type eqpos = arg[ i ].find( '=' );
				std::string name;
				if ( eqpos == std::string::npos )
					name = arg[ i ].substr( 2 );
				else
					name = arg[ i ].substr( 2, eqpos - 2 );

				// check invalid chars in the name.
				if ( name.find_first_not_of( "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-" ) != std::string::npos )
					return parseError( arg[ i ], __LINE__ );

				// the name is sure ok now, check again for =
				if ( eqpos != std::string::npos )   // yes, found...
				{
					// first case, = is the last char (error).
					if ( eqpos == ( arg[ i ].size() - 1 ) )
						return parseError( arg[ i ] += "<- no value.", __LINE__ );

					// second case, after = follow a doublequote
					if ( arg[ i ][ eqpos + 1 ] == '\"' )   // yes
					{
						// search the end of the doublequote (take part above)
						// first case, the " is in the same argument.
						if ( ( arg[ i ].size() > ( eqpos + 2 ) ) && ( arg[ i ][ arg[ i ].size() - 1 ] == '\"' ) )
						{
							if ( raw.find( name ) != raw.end() )
								return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );

							// add parameter with value.
							raw[ name ] = arg[ i ].substr( eqpos + 1, arg[ i ].size() - eqpos - 2 );
						}
						else // no quote in this arg.
						{
							// second case, this is the last arg (error)
							if ( i >= ( argc - 1 ) )
								return parseError( arg[ i ] += "<- no closing doublequote!", __LINE__ );

							// ok, search the next arg's.
							int j;
							std::string value = arg[ i ].size() > ( eqpos + 1 ) ? arg[ i ].substr( eqpos + 1 ) : "";
							for ( j = i + 1; j < argc; j++ )
							{
								value += " ";
								if ( arg[ j ][ arg[ j ].size() - 1 ] == '\"' )
								{
									if ( arg[ j ].size() > 1 ) value += arg[ j ].substr( 0, arg[ j ].size() - 1 );
									break;
								}
								else
								{
									value += arg[ j ];
								}
							}
							// break why?
							if ( arg[ j - 1 ][ arg[ j - 1 ].size() - 1 ] != '\"' )
								return parseError( arg[ i ] += "<- no closing doublequote!", __LINE__ );

							if ( raw.find( name ) != raw.end() )
								return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );

							// Add parameter with value
							raw[ name ] = value;
						}
					}
					else // no doublequote after the =
					{
						if ( raw.find( name ) != raw.end() )
							return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );

						// Add parameter with value
						raw[ name ] = arg[ i ].substr( eqpos + 1 );
					}
				} else
				{ // no = found...
					if ( raw.find( name ) != raw.end() )
						return parseError( arg[ i ] += "<- this is the second one.", __LINE__ );

					// add parameter without a value.
					raw[ name ] = "";
				}
			};

			// get next argument.
			i++;
		}
		while ( i < argc );

		// free the array.
		delete[] arg;
	}

	
	// whow very tricky, now some debug information:
	// [2003/10/18] and in the meanwhile there are good libraries for :-(

#ifdef DEBUG
	std::cerr << "Parameter parsing - Step 1 (preparse)" << std::endl;
	std::cerr << "-------------------------------------" << std::endl;
	std::map<std::string, std::string>::iterator mi;
	int mc = 1;
	for ( mi = raw.begin(); mi != raw.end(); mi++ )
	{
		std::cerr.width( 4 );
		std::cerr << mc << " ";
		std::cerr.width( 20 );
		std::cerr << mi->first.c_str() << " ";
		std::cerr << "\"" << mi->second.c_str() << "\"" << std::endl;
		mc++;
	}
#endif

	// ok, check if the parameter -c is set.
	// with the parameter -c it's possible to read more parameters
	// from a configuration file.
	// Format:
	// #comment
	//   (empty line)
	// paramater value
	// Only --param parameters are allowed, without the --
	// Example:
	// The parameter --domain=profzone.ch in the config file:
	// domain: profzone.ch

	// this is a very cheap solution, never trust the following lines!

	if ( raw.find( "-c" ) != raw.end() )
	{
		// open file.
		if ( raw[ "-c" ] == "" )
		{
			std::cerr << "Error: Paremeter -c with no filename" << std::endl;
			return false;
		}
		std::string name, value, line;
		std::ifstream cf( raw[ "-c" ].c_str() );
		if ( !cf )
		{
			std::cerr << "Error: Can't open File \"" << raw[ "-c" ] << "\"." << std::endl;
			return false;
		}
		// read file.
		int c;
		while ( cf )
		{
			if ( ( c = cf.get() ) == '\n' ) continue;
			if ( c == '#' )
			{
				cf.ignore( 4096, '\n' );
				continue;
			}
			else
			{
				cf.putback( c );

				// Ok, keine leerzeile - kein kommentar.
				// ganze Zeile lesen.
				while ( ( c = cf.get() ) != '\n' )
					if ( !isspace( c ) ) break;
				if ( ( c == '\n' ) || ( c == EOF ) ) continue;

				name = ( char ) c;
				while ( ( c = cf.get() ) != '\n' )
				{
					if ( ( c == ':' ) || ( c == EOF ) ) break;
					name += ( char ) c;
				};
				if ( ( c == EOF ) || ( c == '\n' ) ) continue;

				// Parameter gefunden:
				if ( raw.find( name ) != raw.end() )
					return parseError( name += "<- this is the second one.", __LINE__ );
				raw[ name ] = "";

				while ( ( c = cf.get() ) != '\n' )
					if ( !isspace( c ) ) break;
				if ( ( c == '\n' ) || ( c == EOF ) ) continue;

				value = ( char ) c;
				while ( ( c = cf.get() ) != '\n' )
				{
					if ( ( c == ':' ) || ( c == EOF ) ) break;
					value += ( char ) c;
				};

				raw[ name ] = value;
			}
		}


#ifdef DEBUG
		std::cerr << "Parameter parsing - Step 2 (after reading file)" << std::endl;
		std::cerr << "-----------------------------------------------" << std::endl;
		std::map<std::string, std::string>::iterator mi;
		int mc = 1;
		for ( mi = raw.begin(); mi != raw.end(); mi++ )
		{
			std::cerr.width( 4 );
			std::cerr << mc << " ";
			std::cerr.width( 20 );
			std::cerr << mi->first.c_str() << " ";
			std::cerr << "\"" << mi->second.c_str() << "\"" << std::endl;
			mc++;
		}
#endif

	}
#ifdef DEBUG
	else
	{
		std::cerr << "Paremeter parsing - No Step 2" << std::endl;
		std::cerr << "-----------------------------" << std::endl;
	}
#endif


	// check now the flag "-p" to printout all parameters and
	// exit.
	if ( raw.find( "-p" ) != raw.end() )
	{
		bool html = false;
		if ( raw[ "-p" ] == "html" ) html = true;

		if ( html )
		{
			std::cout << "<html>" << std::endl << "<head>" << std::endl;
			std::cout << "  <title>Anteater V" VERSION " - parameter summary</title>" << std::endl;
			std::cout << "</head>" << std::endl << "<body>" << std::endl;
			std::cout << "<h1>Anteater V" VERSION " - parameter summary</h1>" << std::endl;
			std::cout << "<table>" << std::endl;
		}
		std::map<std::string, std::string>::iterator i;
		unsigned int max = 1;
		for ( i = raw.begin(); i != raw.end(); i++ )
			if ( max < i->first.size() ) max = i->first.size();
		for ( i = raw.begin(); i != raw.end(); i++ )
		{
			if ( html )
			{
				std::cout << "<tr><td><strong>" << i->first.c_str() << "</strong></td><td>&quot;"
				<< i->second.c_str() << "&quot;</td></tr>" << std::endl;
			}
			else
			{
				std::cout.width( max );
				std::cout.setf( std::ios::left );
				std::cout << i->first.c_str();
				std::cout << " - \"" << i->second.c_str() << "\"" << std::endl;
			}
		}
		if ( html )
		{
			std::cout << "</table>" << std::endl << "</body>" << std::endl << "</html>" << std::endl;
		}
		return false;
	}

	// Check the flag "-v" to printout the Version and exit.
	if ( raw.find( "-v" ) != raw.end() )
	{
		std::cout << "*** ANTEATER V" VERSION << " *** dynamic MTA loganalyser" << std::endl << std::endl;
		std::cout << "Anteater version " VERSION ", Copyright (C) 2000-2003 Tobias Erbsland" << std::endl;
		std::cout << "Anteater comes with ABSOLUTELY NO WARRANTY; for details see the file COPYING." << std::endl;
		std::cout << "This is free software, and you are welcome to redistribute it under certain " << std::endl;
		std::cout << "conditions; see file COPYING for details." << std::endl;

		return false;
	}

	// Check the flag "-h" for a short help and exit.
	if ( raw.find( "-h" ) != raw.end() )
	{
		std::cout << "With the parameter -m you will get a list of all possible parameters." << std::endl;
		std::cout << std::endl;
		std::cout << "  anteater -mhtml > anteater-parameters.html" << std::endl;
		std::cout << std::endl;

		return false;
	}

	if ( raw.find( "-m" ) != raw.end() )
	{
		poa::dumpObjectParameter( raw[ "-m" ] == "html" );
		return false;
	}

	// ok, all parameters are now read.
	// Now we must distribute the right parameter to the right
	// place.

	std::map<std::string, std::string>::iterator i;
	std::string localname, name, value;
	for ( i = raw.begin(); i != raw.end(); i++ )
	{
		name = i->first;
		value = i->second;
		if ( global_param_map.find( name ) != global_param_map.end() )   // Global-Param?
		{
			// Here should we test the type of the parameter (in later versions...)

			if ( global_param_map[ name ].Param.Typ == Parameter::G_FLAG && value == "" )
			{
				value = "1";
			}
			std::list< Parameter * >::iterator j;
			for ( j = global_param_map[ name ].User.begin(); j != global_param_map[ name ].User.end(); j++ )
			{
				( *j ) ->Param() [ name ].Value = value;
			}
		}
		else if ( local_param_map.find( name ) != local_param_map.end() )   // Local-Param?
		{
			localname = name.substr( name.find( "-" ) + 1 ); // cut prefix...
			if ( ( local_param_map[ name ] ) ->Param() [ localname ].Typ == Parameter::FLAG && value == "" )
			{
				value = "1";
			}
			( local_param_map[ name ] ) ->Param() [ localname ].Value = value;
		}
		else
		{
			if ( name != "-c" )
				std::cerr << "Ignore unknown Parameter \"" << name << "\"." << std::endl;
		}
	}

	return true;
};

bool clp::parseError( const std::string &wrong, long int )
{
	std::cerr << "Ooops! I don't understand your parameter \"" << wrong << "\"..." << std::endl;
	std::cerr << "Please check the syntax or read the manual." << std::endl << std::endl;
	return false;
};
