# $Id: SMTP2.pm,v 1.1 2003/11/26 10:23:11 bengen Exp $

#
# MTA module for pure SMTP setup
#

package AMAVIS::MTA::SMTP2;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

# For receiving mail
use IO::Socket;
use IO::Socket::INET;

# For sending mail
use Net::SMTP;

use POSIX qw(setsid);

use vars qw(
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line

	    $cfg_smtp_in
	    $cfg_smtp_out
	    $cfg_smtp_port_in
	    $cfg_smtp_port_out

	    $cfg_smtp_session_timeout

	    $cfg_daemon
	    $cfg_pidfile
	    $hostname

	    $smtpserver
	    $conn_in
	    $conn_out

	    $saved_args
	    $mta_result

	    $server_pid
	    $running
	    $signame
	   );

sub init {
  my $class = shift;
  my $args = shift;

  $cfg_smtp_in = ($AMAVIS::cfg->val('SMTP', 'input address') or 'localhost');
  $cfg_smtp_out = ($AMAVIS::cfg->val('SMTP', 'output address') or 'localhost');
  $cfg_smtp_port_in = ($AMAVIS::cfg->val('SMTP', 'input port') or '10025');
  $cfg_smtp_port_out = ($AMAVIS::cfg->val('SMTP', 'output port') or '10026');

  if ((!defined $cfg_smtp_port_in ) &&
      (!defined $cfg_smtp_port_out )) {
    writelog($args,LOG_CRIT,__PACKAGE__.
	     ": SMTP input and output ports must be specified.");
    return 0;
  }

  $cfg_daemon = ($AMAVIS::cfg->val('SMTP', 'daemon') or 'yes');
  $cfg_pidfile = $AMAVIS::cfg->val('SMTP', 'pidfile');

  # Session timeout:
  # undef: accept message immediately
  # =0:    no timeout. Response is sent when We Are Done.
  # >0:    timeout is set to n seconds. After n seconds, processing 
  #        is aborted if it has not been finished.
  $cfg_smtp_session_timeout=$AMAVIS::cfg->val('SMTP','session timeout');

  writelog($args,LOG_DEBUG,__PACKAGE__.
	   ": Input  $cfg_smtp_in:$cfg_smtp_port_in");
  writelog($args,LOG_DEBUG,__PACKAGE__.
	   ": Output $cfg_smtp_out:$cfg_smtp_port_out");

  # $hostname=hostname();
  $hostname = $cfg_smtp_in;

  my $pid;

  if ($cfg_daemon eq 'yes') {
    (open(STDIN, "< /dev/null") && open(STDOUT, "> /dev/null") && open(STDERR, "> /dev/null")) || do {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Error closing stdin, stdout, or stderr: $!");
      return 0;
    };
    if (!defined ($pid = fork())) {
      writelog($args,LOG_ERR, __PACKAGE__.": fork() failed.");
      return 0;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      writelog($args,LOG_DEBUG, __PACKAGE__.
	       ": fork() successful, child's PID=$pid.");
      exit 0;
    }
  }

  # We are the child.
  # So we become a daemon.

  $smtpserver = IO::Socket::INET->new (
				       LocalAddr=>$cfg_smtp_in,
				       LocalPort=>$cfg_smtp_port_in,
				       Listen=>20,
				       Proto=>'tcp',
				       ReuseAddr=>1
				      )
    || do {
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": Unable to create SMTP server on $cfg_smtp_in:".
	       "$cfg_smtp_port_in: $!");
      return 0;
    };

  # Enter chroot environment if desired
  if ($cfg_chroot) {
    chroot($cfg_chroot)
      or die "Error in chroot($cfg_chroot): $!\n";
  }

  # Drop privileges if desired
  if ($> == 0) {
    if (defined $cfg_gid) {
      writelog($args,LOG_DEBUG,__PACKAGE__.": Dropping GID");
      $)=$cfg_gid;
      if ($) != $cfg_gid) {
	writelog($args,LOG_ERR, __PACKAGE__.": Can't drop GID to $cfg_gid");
	die;
      }
    }
    if (defined $cfg_uid) {
      writelog($args,LOG_DEBUG,__PACKAGE__.": Dropping UID");
      $>=$cfg_uid;
      if ($> != $cfg_uid) {
	writelog($args,LOG_ERR, __PACKAGE__.": Can't drop UID to $cfg_uid");
	die;
      }
    }
  }

  setsid() or do {
    writelog($args,LOG_ERR,__PACKAGE__.": setsid() failed: $!");
    return 0;
  };
  chdir("/");

  $server_pid=$$;

  # crate PID file.
  my $pidfile=IO::File->new(">$cfg_pidfile");
  unless (defined $pidfile) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to create PID file: $!");
    return 0;
  }
  else {
    ($pidfile->print("$$\n") && $pidfile->close()) || do {
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": Unable to write to PID file: $!");
    }
  }
  $0='amavisd';

  if ($AMAVIS::cfg->val('global', 'x-header') eq 'true') {
    $cfg_x_header = 1
  };
  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');

  $SIG{TERM} = \&inthandler;

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  if ($$ == $server_pid) {
    writelog($args,LOG_DEBUG,__PACKAGE__.
	     ": Received SIG$signame. Cleaning up.");
    unlink $cfg_pidfile;
  }
  return 1;
}

# Create temp dir and write mail
sub get_directory($) {
  my $self = shift;
  my $args = shift;

  my $prefix = "$AMAVIS::cfg_unpack_dir/amavis-unpack-";
  my $i = 0;
  my $message_id;

  # Main loop for accepting SMTP connections
  while(eval{$conn_in = $smtpserver->accept();}) {
    my $pid;
    writelog($args,LOG_DEBUG, "Accepting connection.");
    if (!defined ($pid = fork)) {
      writelog($args,LOG_ERR, "fork() failed.");
      send_smtp_answer($conn_in, "421", "fork() failed.",
		       "$hostname AMaViS Service not available, closing transmission channel");
      $conn_in->close();
      next;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      # The parent will (should) only have children that have been
      # forked for accepting socket connections.
      $SIG{CHLD} = 'IGNORE';
      writelog($args,LOG_DEBUG, "fork() successful, child's PID=$pid.");
      $conn_in->close();
      next;
    }
    else {
      # We are the child
      # Our children won't be automatically reaped.
      $SIG{CHLD} = 'DEFAULT';
      $SIG{TERM} = 'IGNORE';

      # Empty log
      $$args{'log'} = '';

      # Make sure that no result has been set for the message.
      undef $mta_result;

      # Create temp directory. Try creating $prefix[date]-[pid] 
      # If unsuccessful after 10 tries, abort
      while (1) {
	$message_id = sprintf("%.8x-%.4x",time,$$);
	unless (defined mkpath ($prefix.$message_id, 0, 0770)) {
	  if (++$i > 10) {
	    writelog($args,LOG_ERR,
		     __PACKAGE__.": Unable to create unpacking directory.");
	    send_smtp_answer($conn_in, "421", "Unable to create unpacking directory.",
			     "$hostname AMaViS Service not available, closing transmission channel");
	    # We "return" to AMAVIS.pm, telling it that message
	    # unpacking failed
	    return 0;
	  }
	  else {
	    next;
	  }
	}
	last;
      }

      $$args{'message_id'}=$message_id;
      my $directory = $prefix.$message_id;
      mkdir "$directory/parts", 0777;
      $$args{'directory'} = $directory;

      # Send welcome message
      unless ($conn_out = IO::Socket::INET->new(
						PeerAddr=>$cfg_smtp_out,
						PeerPort=>$cfg_smtp_port_out,
					       )) {
	send_smtp_answer($conn_in, "421","Unable to create socket for back connection.",
			 "$hostname AMaViS Service not available, closing transmission channel");
	$conn_in->close();
	return 0;
      } else {
	my ($code, $text) = get_smtp_answer($conn_out);
	foreach ($code) {
	  /^2/ && do {
	    send_smtp_answer($conn_in, "220", "$hostname AMaViS SMTP Pleased to meet you. Gimme some malicious code, already.");
	    last;
	  };
	  # Otherwise...
	  send_smtp_answer($conn_in, "421", "Unable to talk to MTA for back connection.",
			   "$hostname AMaViS Service not available, closing transmission channel");
	}
      }

      # email.txt will be written here.
      if (handle_smtp_connection($conn_in,$conn_out,$args)) {
      }
      else {
	writelog($args,LOG_ERR,__PACKAGE__.
		 ": Incoming SMTP session failed");
	return 0;
      };

      # Return successfully
      return 1;
    }
  }

  # We reach this point only if there was a problem in the accept loop. 
  $$args{'directory'}='END';
  return 0;
}

# Called from within AMAVIS.pm to continue message delivery
sub accept_message( $ ) {
  my $self = shift;
  my $args = shift;

  my ($code, $text);

  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");
  $mta_result='accept';

  my $result;

  ($code, $text)=send_smtp_command($conn_out, "DATA");
  if ($code!~/^3/) {
    writelog($args, LOG_ERR, "MTA did not want DATA: \"$code $text\"");
    send_smtp_anser($conn_in, "421", "MTA did not accept the message");
    return 0;
  }

  my $headers=1;
  my $fh = $$args{'filehandle'};
  while (<$fh>) {
    chomp;
    # Insert X- header at the end of the headers.
    if ($headers && /^\s*$/) {
      $headers=0;
      if ($cfg_x_header) {
	$result=$conn_out->print("$cfg_x_header_tag: $cfg_x_header_line\r\n");
	return 0 unless ($result);
      }
    }
    $result = $conn_out->print("$_\r\n");
    return 0 unless ($result);
  }
  ($code, $text)=send_smtp_command($conn_out, ".");
  writelog($args,LOG_DEBUG, "DATA done.");
  foreach ($code) {
    /^2/ && do {
      # Tell client that we have taken responsibility for the message
      writelog($args,LOG_DEBUG, "Message accepted by MTA.");
      send_smtp_answer($conn_in, "250", "I got it. Seems virus-free.");
      last;
    };
    # Otherwise 
    writelog($args,LOG_DEBUG, "MTA gave error message: \"$code $text\".");
    send_smtp_answer($conn_in, $code, $text);
  }
  send_smtp_answer($conn_in, "421", "Closing connection");
  $conn_in->close();
  send_smtp_command($conn_out, "QUIT");
  $conn_out->close();
  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to throw message away
sub drop_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.
	   ": Dropping message (Message-ID: ".$$args{'Message-ID'}.")");

  $mta_result='drop';

  # We won't care if the timeout alarm hits us now.
  writelog($args,LOG_DEBUG, __PACKAGE__.
	   ": Accepting message at SMTP level, but DROPPING it");
  send_smtp_answer($conn_in, "250", "I WILL DROP THIS MESSAGE.");
  send_smtp_answer($conn_in, "421", "Closing connection");
  $conn_in->close();
  send_smtp_command($conn_out, "QUIT");
  $conn_out->close();
  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to freeze message delivery
sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  $mta_result='freeze';

  # First try to put the message into the problems directory.
  if (AMAVIS->quarantine_problem_message($args)) {
    send_smtp_answer($conn_in, "220", "Put message into quarantine directory");
    send_smtp_answer($conn_in, "421", "Closing connection");
    $conn_in->close();
    send_smtp_command($conn_out, "QUIT");
    $conn_out->close();
    return 1;
  }
  # If that fails, reject the message
  else {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ': Unable to put message into problem dir ');
    # Instead of freezing, reject the message at SMTP level.
    writelog($args,LOG_DEBUG, __PACKAGE__.
	     ": Rejecting message at SMTP level with temporary failure");
    send_smtp_answer($conn_in, "453", "Error in processing the message");
    send_smtp_answer($conn_in, "421", "Closing connection");
    $conn_in->close();
    send_smtp_command($conn_out, "QUIT");
    $conn_out->close();
    return 1;
  }
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message( $$$ ) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my $smtpclient = new Net::SMTP("$cfg_smtp_out:$cfg_smtp_port_out",
				 Hello => 'localhost',
				 Timeout => 30);
  unless (defined $smtpclient) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to connect to SMTP server on ".
	     "$cfg_smtp_out:$cfg_smtp_port_out.");
    return 0;
  }

  # FIXME: Error checking
  my $result;

  writelog($args,LOG_DEBUG, "Setting sender: $sender");
  $sender =~ s/^<(.*)>$/$1/;
  $result=$smtpclient->mail($sender);
  unless ($result) {
    writelog($args,LOG_ERR, __PACKAGE__.": Error while MAIL FROM:");
    return 0;
  }
  writelog($args,LOG_DEBUG, "Setting recipients: ".
	   join(', ',@recipients));
  $result=$smtpclient->recipient(@recipients);
  unless ($result) {
    writelog($args,LOG_ERR, __PACKAGE__.": Error while RCPT TO:");
    return 0;
  }
  writelog($args,LOG_DEBUG, "Sending DATA");
  $result=$smtpclient->data($message); #split(/\n/m, $message));
  unless ($result) {
    writelog($args,LOG_ERR, __PACKAGE__.": Error while DATA");
    return 0;
  }
  $smtpclient->quit();

  # Return successfully
  return 1;
}

sub get_smtp_answer {
  my $socket = shift;
  my $code;
  my $text;
  while (<$socket>) {
    /^(...)-(.*)$/ && do {
      $code = $1;
      $text = $2;
      next;
    };
    /^(...) (.*)$/ && do {
      $code = $1;
      $text = $2;
      last;
    };
  }
  return $code, $text;
}

sub send_smtp_command {
  my $socket = shift;
  my $command = shift;
  my $code;
  my $text;
  $socket->print("$command\r\n");
  return (get_smtp_answer($socket));
}

sub send_smtp_answer {
  my $socket = shift;
  my $code = shift;
  my @text = @_;
  foreach my $i (0..$#text-1) {
    $socket->print("$code-$text[$i]\r\n");
  }
  $socket->print("$code $text[-1]\r\n");
}

sub handle_smtp_connection {
  my $conn_in = shift;
  my $conn_out = shift;
  my $args = shift;

  my $code;
  my $text;

  my $ret;

  # ready, sender, recipient, data, closed
  my $state = "ready";

  my $from=undef;
  my @to=();

 SMTP:while (<$conn_in>) {
    # Remove leading, trailing spaces and \r\n
    chomp;
    s/^\s+//;
    s/\s+$//;
    foreach ($_) {
      # RFC 822 4.1.1
      /^(?:HE|EH)LO (.*)/i && do {
	writelog($args, LOG_DEBUG, "HELO");
	if ($state eq "ready") {
	  ($code,$text) = send_smtp_command($conn_out, "HELO $hostname");
	  foreach ($code) {
	    /^2/ && do {
	      send_smtp_answer($conn_in, "250", "Hi");
	      next;
	    };
	    # Otherwise...
	    writelog($args,LOG_ERR, __PACKAGE__.
		     ": MTA responded to HELO with \"$code $text\"");
	    send_smtp_answer($conn_in, "421", "Closing connection");
	    return 0;
	  }
	} else {
	  send_smtp_answer($conn_in, "550", "Huh?");
	}
	next;
      };
      /^MAIL FROM:(.*)$/i && do {
	writelog($args, LOG_DEBUG, "MAIL FROM:$1");
	if ($state eq "ready") {
	  my $tmpfrom = p_addr($1);
	  writelog($args, LOG_DEBUG, "MAIL FROM:$tmpfrom");
	  ($code, $text) = (send_smtp_command($conn_out, "MAIL FROM: $tmpfrom"));
	  foreach ($code) {
	    /^2/ && do {
	      $state = "sender";
	      $from = $tmpfrom;
	      @to=();
	      next;
	    };
	    # Otherwise...
	    writelog($args, LOG_ERR, "MAIL FROM not accepted : \"$code $text\"");
	  }
	  send_smtp_answer($conn_in, $code, $text);
	} else {
	  send_smtp_answer($conn_in, "550", "Huh?");
	}
	next;
      };
      /^RCPT TO:(.*)$/i && do {
	writelog($args, LOG_DEBUG, "RCPT TO:$1");
	if (($state eq "sender") or ($state eq "recipient")) {
	  my $to = p_addr($1);
	  writelog($args, LOG_DEBUG, "RCPT TO:$to");
	  ($code, $text) = send_smtp_command($conn_out, "RCPT TO: $to");
	  foreach ($code) {
	    /^2/ && do {
	      $state = "recipient";
	      push @to, $to;
	      next;
	    };
	    # Otherwise...
	    writelog($args, LOG_ERR, "RCPT TO not accepted : \"$code $text\"");
	  }
	  send_smtp_answer($conn_in, $code, $text);
	} else {
	  send_smtp_answer($conn_in, "550", "Huh?");
	}
	next;
      };
      /^DATA$/i && do {
	writelog($args, LOG_DEBUG, "DATA");
	if (($state eq "recipient") && ($#to >= 0)) {
	  send_smtp_answer($conn_in, "354", "End data with <CR><LF>.<CR><LF>");

	  # Open message file that is later going to be disected and scanned
	  my $output = IO::File->new("+>$$args{'directory'}/email.txt");
	  my $done=undef;
	  my $headers=1;
	DATA:while (<$conn_in>) {
	    s/\r\n$//;
	    if(/^\.$/) {
	      writelog($args,LOG_DEBUG,__PACKAGE__.
		       ": Received end of DATA.");
	      $done = 1;
	      last DATA;
	    }
	    # Replace '..' at beginning of line with '.'
	    s/^\.(.+)$/$1/;
	    if (/^$/) {
	      $headers=0;
	    }
	    if ($headers==1) {
	      if (/^Message-ID:\s*(.*)\s*$/i) {
		$$args{'Message-ID'} = $1;
	      }
	      $$args{'headers'}.="$_\n";
	    }
	    $output->print("$_\n");
	  }
	  # Message file has been written, reset file pointer and put it into
	  # the record.
	  $output->seek(0,0);
	  $$args{'filehandle'} = $output;
	  # Body and dot have been received.
	  $$args{'Message-ID'} = "Unknown" unless defined ($$args{'Message-ID'});
	  last SMTP;
	  # FIXME: This should of course be the top loop...
	}
	else {
	  send_smtp_answer($conn_in, "500", "foo");
	}
	next;
      };
      /^RSET/i && do {
	writelog($args, LOG_DEBUG, "RSET");
	$state = "ready";
	$from=undef;
	@to=();
	send_smtp_answer($conn_in, (send_smtp_command($conn_out,"RSET")));
	next;
      };
      # Do not confirm that recipient exists.
      /^VRFY/i && do {
	writelog($args, LOG_DEBUG, "VRFY");
	send_smtp_answer($conn_in, "252", "No.");
	next;
      };
      # Do not confirm that recipient expands to a mailing list.
      /^EXPN/i && do {
	writelog($args, LOG_DEBUG, "EXPN");
	send_smtp_answer($conn_in, "252", "No.");
	next;
      };
      /^HELP/i && do {
	writelog($args, LOG_DEBUG, "HELP");
	send_smtp_answer($conn_in, "214", "Help yourself");
	next;
      };
      /^NOOP/i && do {
	writelog($args, LOG_DEBUG, "NOOP");
	send_smtp_answer($conn_in, "250", "Yawn");
	next;
      };
      /^QUIT/i && do {
	writelog($args, LOG_DEBUG, "QUIT");
	send_smtp_answer($conn_in, "221", "See you later");
	$conn_in->close();
	return 0;
      };
      # If none of the above commands was recognized, tell the other
      # side so.
      writelog($args, LOG_ERR, "Unknown command: $_");
      send_smtp_answer($conn_in, "500", "Unknown command");
    }
  }

  $$args{'sender'} = $from;
  $$args{'sender'} = "<>" if (!$$args{'sender'});
  push @{$$args{'recipients'}}, @to;
  return 1;
}

sub inthandler {
  $signame = shift;
  $running=0;
  die "Somebody sent me a SIG$signame";
}

sub p_addr {
  (my $addr = shift || "") =~ s/^\s*<?\s*|\s*>?\s*$//sg ;
  "<$addr>";
}

1;
