/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  Authors: Jeffrey Stedfast <fejj@ximian.com>
 *
 *  Copyright 2003 Ximian, Inc. (www.ximian.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "vgskinview.h"


static void vg_skin_view_class_init (VgSkinViewClass *klass);
static void vg_skin_view_init (VgSkinView *view);
static void vg_skin_view_destroy (GtkObject *obj);
static void vg_skin_view_finalize (GObject *obj);

static void skin_view_clear (VgSkinView *view);
static void skin_view_reset (VgSkinView *view);
static void skin_view_connect (VgSkinView *view, int sockfd);
static int  skin_view_step (VgSkinView *view);
static void skin_view_disconnect (VgSkinView *view);
static int  skin_view_save_log (VgSkinView *view, int fd);
static void skin_view_cut (VgSkinView *view);
static void skin_view_copy (VgSkinView *view);
static void skin_view_paste (VgSkinView *view);
static void skin_view_show_rules (VgSkinView *view);


static GtkVBoxClass *parent_class = NULL;


GType
vg_skin_view_get_type (void)
{
	static GType type = 0;
	
	if (!type) {
		static const GTypeInfo info = {
			sizeof (VgSkinViewClass),
			NULL, /* base_class_init */
			NULL, /* base_class_finalize */
			(GClassInitFunc) vg_skin_view_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (VgSkinView),
			0,    /* n_preallocs */
			(GInstanceInitFunc) vg_skin_view_init,
		};
		
		type = g_type_register_static (GTK_TYPE_VBOX, "VgSkinView", &info, 0);
	}
	
	return type;
}

static void
vg_skin_view_class_init (VgSkinViewClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (klass);
	
	parent_class = g_type_class_ref (GTK_TYPE_VBOX);
	
	object_class->finalize = vg_skin_view_finalize;
	gtk_object_class->destroy = vg_skin_view_destroy;
	
	/* virtual methods */
	klass->clear = skin_view_clear;
	klass->reset = skin_view_reset;
	klass->connect = skin_view_connect;
	klass->step = skin_view_step;
	klass->disconnect = skin_view_disconnect;
	klass->save_log = skin_view_save_log;
	klass->cut = skin_view_cut;
	klass->copy = skin_view_copy;
	klass->paste = skin_view_paste;
	klass->show_rules = skin_view_show_rules;
}

static void
vg_skin_view_init (VgSkinView *view)
{
	view->argv = NULL;
	view->symtab = NULL;
	view->srcdir = NULL;
	view->rules = NULL;
}

static void
vg_skin_view_finalize (GObject *obj)
{
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void
vg_skin_view_destroy (GtkObject *obj)
{
	GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}


void
vg_skin_view_set_argv (VgSkinView *view, const char **argv)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	view->argv = argv;
}


void
vg_skin_view_set_srcdir (VgSkinView *view, const char **srcdir)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	view->srcdir = srcdir;
}


void
vg_skin_view_set_symtab (VgSkinView *view, SymTab *symtab)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	view->symtab = symtab;
}


static void
skin_view_clear (VgSkinView *view)
{
	;
}


void
vg_skin_view_clear (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->clear (view);
}


static void
skin_view_reset (VgSkinView *view)
{
	;
}


void
vg_skin_view_reset (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->reset (view);
}


static void
skin_view_connect (VgSkinView *view, int sockfd)
{
	;
}


void
vg_skin_view_connect (VgSkinView *view, int sockfd)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->connect (view, sockfd);
}


static int
skin_view_step (VgSkinView *view)
{
	;
}


int
vg_skin_view_step (VgSkinView *view)
{
	g_return_val_if_fail (VG_IS_SKIN_VIEW (view), -1);
	
	return VG_SKIN_VIEW_GET_CLASS (view)->step (view);
}


static void
skin_view_disconnect (VgSkinView *view)
{
	;
}


void
vg_skin_view_disconnect (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->disconnect (view);
}


static int
skin_view_save_log (VgSkinView *view, int fd)
{
	errno = ENOTSUP;
	return -1;
}


int
vg_skin_view_save_log (VgSkinView *view, int fd)
{
	g_return_val_if_fail (VG_IS_SKIN_VIEW (view), -1);
	
	return VG_SKIN_VIEW_GET_CLASS (view)->save_log (view, fd);
}


static void
skin_view_cut (VgSkinView *view)
{
	;
}


void
vg_skin_view_cut (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->cut (view);
}


static void
skin_view_copy (VgSkinView *view)
{
	;
}


void
vg_skin_view_copy (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->copy (view);
}


static void
skin_view_paste (VgSkinView *view)
{
	;
}


void
vg_skin_view_paste (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->paste (view);
}


static void
skin_view_show_rules (VgSkinView *view)
{
	if (view->rules == NULL)
		return;
	
	if (GTK_WIDGET_MAPPED (view->rules))
		gdk_window_raise (view->rules->window);
	else
		gtk_widget_show (view->rules);
}


void
vg_skin_view_show_rules (VgSkinView *view)
{
	g_return_if_fail (VG_IS_SKIN_VIEW (view));
	
	VG_SKIN_VIEW_GET_CLASS (view)->show_rules (view);
}



static char *
path_concat (const char *dirname, int dirlen, const char *basename, int baselen)
{
	char *path, *p;
	
	p = path = g_malloc (dirlen + baselen + 2);
	memcpy (path, dirname, dirlen);
	p += dirlen;
	*p++ = '/';
	memcpy (p, basename, baselen);
	p[baselen] = '\0';
	
	return path;
}

static gboolean
path_is_rx (const char *path)
{
	struct stat st;
	
	if (stat (path, &st) != -1 && S_ISREG (st.st_mode)) {
		if (access (path, R_OK | X_OK) != -1)
			return TRUE;
	}
	
	return FALSE;
}


char *
vg_skin_view_scan_path (const char *program)
{
	const char *pathenv, *path, *p;
	struct stat st;
	char *filename;
	int len;
	
	if (program[0] == '/') {
		if (path_is_rx (program))
			return g_strdup (program);
		
		return NULL;
	}
	
	if (!(pathenv = getenv ("PATH")))
		return NULL;
	
	path = pathenv;
	len = strlen (program);
	while ((p = strchr (path, ':'))) {
		if (p > path) {
			filename = path_concat (path, (p - path), program, len);
			if (path_is_rx (filename))
				return filename;
			
			g_free (filename);
		}
		
		path = p + 1;
	}
	
	if (path[0] != '\0') {
		filename = g_strdup_printf ("%s/%s", path, program);
		if (path_is_rx (filename))
			return filename;
		
		g_free (filename);
	}
	
	return NULL;
}
