# This is very simple library for handling XMPP/Jabber protocol.
# The protocol is itended to be very extendable, thus this library
# provides very basics functionality - connecting to server, handling
# stream :stream tag and converting protocol xml tags into events.
# It also provides helper functions for generating XML tags, XML escaping, etc.
#
# Copyright (c) Ihar Viarheichyk, 2004

package require easyxml
package provide xmpp 0.1

namespace eval xmpp {
	# Export helper procedures
	namespace export tag escape stamp2time time2stamp
	variable defaults { -port 5222 -event "" -role client  -server localhost id "" connected 0 -lang "" -proxy "" -ssl 0}
}

if {[catch {package require tls} reason]} { set xmpp::nossl $reason }

namespace eval xmpp {

	# Handle incoming network packets '''1
	proc Read {name} {
		upvar #0 $name data
		if {[catch { set chunk [read $data(socket)] } reason]} {
			xmpp::event $name Log warning $reason
			public:disconnect $name
			return
		}
		if {$chunk==""} {
			puts "chunk: $chunk"
			if {[fblocked $data(socket)] || ![eof $data(socket)]} return
			xmpp::event $name Log warning "Connection closed [eof $data(socket)]"
			public:disconnect $name
			return
		}
		xmpp::event $name Log {debug dump} "<-- $chunk"
		if {[catch { $data(parser) parse $chunk } reason]} {
			puts "Error in data stream: $reason, $::errorInfo"
			#xmpp::event $name Error "Error in data stream: $reason, $::errorInfo"
			public:disconnect $name
		}
	}
	
	# Send initiating stream tag on connect '''1
	proc Greeting {name} {
		upvar #0 $name data
		fileevent $data(socket) writable ""
		set lang ""
		if {$data(-lang)!=""} { set lang xml:lang='$data(-lang)' }
		puts $data(socket) "<?xml version='1.0'?><stream:stream xmlns='jabber:$data(-role)' xmlns:stream='http://etherx.jabber.org/streams' to='$data(-server)' $lang version='1.0'>"
		stream::parser $data(parser) [namespace tail $name]
		fileevent $data(socket) readable [namespace code [list\
			Read $name]]
	}

	proc SpawnSession {cmd sock addr port} {
		fconfigure $sock -blocking no -encoding utf-8 -buffering none
		xmpp x-$sock -role server socket $sock -event $cmd
		stream::parser [x-$sock cget parser] x-$sock
		fileevent $sock readable [nc Read [namespace current]::x-$sock]
	}

	proc server {cmd {port 5222}} {
		set sock [socket -server [nc SpawnSession $cmd] $port]
	}

	# Connection command constructor '''1
	proc xmpp {name args} {
		variable defaults
		set ns [namespace current]
		lappend args parser [easyxml::parser parser-${name}]
		eval [list dispatch $name configure] $defaults $args
		interp alias {} $name {} ${ns}::dispatch $name
		set name
	}

	# Connection command dispatcher '''1
	proc dispatch {name cmd args} {
		set cmd public:$cmd
		if {[info commands $cmd]==$cmd} {
			eval [list $cmd [namespace current]::$name] $args
		} else {
			foreach x [info commands public:*] {
				lappend valid [string range\
					[namespace tail $x] 7 end]
			}
			return -code "Unknown method $cmd. Should be one of: $valid"
		}
	}
	
	# Event passing '''1
	proc event {name args} {
		upvar #0 $name data
		if {$data(-event)!=""} {
			if {[catch {eval [list $data(-event) [namespace tail $name]] $args} r]} {
				puts "error while handling $args: $r"
			}
		}
	}
	#```

	# A set of methods available via connection command '''1

	# Configure connection options '''2
	proc public:configure {name args} {
		upvar #0 $name data
		array set data $args
	}
	
	# Get connection option by name '''2
	proc public:cget {name key} {
		upvar #0 $name data
		if {[info exists data($key)]} {
			return $data($key)
		} else {
			set valid [array names data -*]
			return -code error "Unknown property $key. Should be one of $valid"
		}
	}

	# Send raw XML data via connection '''2
	proc public:send {name val} {
		upvar #0 $name data
		if {!$data(connected)} { return 0 }
		puts -nonewline $data(socket) $val
		event $name Log {debug dump outgoing}\
			"--> $val"
		return 1
	}

	# Send IQ of type set '''2
	proc public:set {name id val} {
		public:send $name [tag iq type set id $id $val]
	}
	# Send IQ of type get '''2
	proc public:get {name id val} {
		public:send $name [tag iq type get id $id $val]
	}
	# Initiate connection '''2
	proc public:connect {name} {
		upvar #0 $name data
		if {![info exists data(-server)]} {
			return -code error "Server to connect is not given"
		}
		if {$data(-proxy)!=""} {
			if {[catch { package require proxy } reason]} {
				event $name Log {error proxy}\
					"Proxy is disabled: $reason"
			} else {
				::proxy::connect $data(-proxy)\
					$data(-server) $data(-port)\
					[nc ConfigureSocket $name]\
					[nc ProxyError $name]\
					[nc event $name Log]
				return	
			}		
		}
		if {[catch {
			set socket [socket -async $data(-server) $data(-port)]
			} reason]} {
				event $name Log error $reason 
				return 
		}
		if {$data(-ssl)} {
			if {[info exists nossl]} {
				event $name Log {error SSL} "No SSL support: $nossl"
			} else {
				tls::import $socket -command [nc TLS $name] 
				#fconfigure $socket
				fileevent $socket writable [nc handshake $name $socket]
				#handshake $name $socket
			}
		} else { ConfigureSocket $name $socket }
	}

	proc handshake {name socket} { 
		if {[eof $socket]} {
			puts "connection closed"
			return
		}
		if { [catch { set res [tls::handshake $socket ] } reason]} {
			puts "handshake: $reason"
		} elseif { $res } { ConfigureSocket $name $socket }
	}
	
	proc ConfigureSocket {name sock} {
		upvar #0 $name data
		set data(socket) $sock
		fconfigure $data(socket) -buffering none -encoding utf-8\
			-blocking no
		fileevent $data(socket) writable [namespace code [list\
			Greeting $name]]
	}
	proc ProxyError {name code descr} {
		event $name Error:proxy:$code $descr
		if {$code!="auth"} { 
			public:disconnect $name 
		}
	}

	proc TLS {name cmd args} {
		update idletasks
		if {[info commands TLS:$cmd]=="TLS:$cmd"} {
			eval [list TLS:$cmd $name] $args 
		} else { event $name Log {debug TLS} "TLS callback $cmd not present" }
	}

	proc TLS:info {name sock major minor message} {
		event $name Log {debug TLS} "$major/$minor: $message "
	}
	proc TLS:verify {name sock depth sert status error} {
		event $name Log {info TLS } "Validate sertificate $sert"
		return 1
	}
	proc TLS:error {name sock reason} {
		puts "disconnect $name $reason"
		#public:disconnect $name $reason
	}

	# Check if connection is established '''2
	proc public:connected? {name} {
		upvar #0 $name data
		set data(connected)
	}
	# Force disconnect '''2
	proc public:disconnect {name {reason ""}} {
		upvar #0 $name data
		if {![info exists data(socket)]} { return 0 }
		# If socket is open, greterful disconnect
		catch { 
			if {$reason!=""} { set reason [tag status $reason] }
			public:send $name "[tag presence type unavailable $reason]</stream:stream>"
			close $data(socket)
		}
		unset data(socket)
		array set data [list id "" connected 0]
		xmpp::event $name Disconnected $reason
		return 1
	}
	# Get connection ID '''2
	proc public:id {name} { set ${name}(id) }

	# Helper procedures '''1
	proc tag {args} {
		set len [llength $args]
		if {!$len} { return -code error "Nothing to send" }
		# If even number of arguments, last item is a body
		# otherwise all arguments are attribute list
		if {!($len&1)} {
			set body [lindex $args end]
			set args [lrange $args 0 end-1]
		}
		set name [lindex $args 0]
		if {[llength $name]==2} {
			lappend args xmlns [lindex $name 0]
			set name [lindex $name end]
		}
		set tag $name
		foreach {key val} [lrange $args 1 end] {
			append tag " $key='$val'"
		}
		if {[info exists body]} {
			 set res "<$tag>$body</$name>"
		} else { set res "<$tag/>" }
		set res
	}

	proc stamp2time {stamp} { clock scan $stamp -gmt 1 }

	proc time2stamp {time} { clock format $time -format "%Y%m%dT%T" -gmt 1}

	interp alias {} [namespace current]::escape {} string map\
		{ < &lt; > &gt; & &amp; \" &quot; ' &apos; }
		
	proc nc {args} { namespace code $args }
}

# Namespace containing XML handlers for tags within stream '''1
namespace eval xmpp::flow {
	proc parser {parser id} {
		$parser configure\
			-start-tag [namespace code [list StartTag $id]]\
			-end-tag [namespace code [list EndTag $id]]
	}

	proc StartTag {id tag attrs} {
		upvar #0 [namespace current]::$id name
		array set aux $attrs
		if {[info exists aux(xmlns)]} { set tag [list $aux(xmlns) $tag]}
		if {![info exists name] || ![llength $name]} {
			if {$tag=="iq"} {
				if {[info exists aux(type)]} {
					set tag ${tag}($aux(type)) 
				}
			}
			xmpp::event [namespace parent]::$id Stanza:Start\
				$tag $attrs
		}
		lappend name $tag
	}

	proc EndTag {id tag attrs val} {
		upvar #0 [namespace current]::$id name

		set n [namespace parent]::$id
		xmpp::event $n xmpp $name $attrs $val
		set name [lrange $name 0 end-1]
		if {![llength $name]} { xmpp::event $n Stanza:End }
	}
}

# Namespace containing handlers of stream:stream and stream:error tags '''1
namespace eval xmpp::stream {
	proc parser {parser name} {
		$parser configure -start-tag [namespace code [list stream $name]] -end-tag {}
		$parser reset	
	}

	proc stream {name tag attrs} {
		set n [namespace parent]::$name
		upvar #0 $n data
		if {$tag!="stream:stream"} {
			return -code error "Invalid tag $tag"
		}
		$data(-role) $n $attrs
		[namespace parent]::flow::parser $data(parser) $name
		set data(connected) 1	
	}
	proc client {n attrs} {
		upvar #0 $n data
		array set aux $attrs
		if {![info exists aux(id)]} {
			return -code error "No stream ID, disconnect"
		}
		set data(id) $aux(id)
		after idle [list xmpp::event $n Established]
	}
	proc server {n attrs} {
		upvar #0 $n data
		array set aux $attrs
		if {![info exists aux(xmlns:stream)] ||
		    ![info exists aux(xmlns)] ||
		    $aux(xmlns:stream)!="http://etherx.jabber.org/streams" ||
		    $aux(xmlns)!="jabber:client"} {
			return -code error "Invalid namespace"
		}
		set data(id) [clock clicks]
		puts $data(socket) "<?xml version='1.0'?><stream:stream xmlns='jabber:server' xmlns:stream='http://etherx.jabber.org/streams' id=$data(id) to='you' version='1.0'>"
	}
}

