// ---------------------------------------------------------------------------
// - Regex.hpp                                                               -
// - standard object library - regex class definition                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_REGEX_HPP
#define  ALEPH_REGEX_HPP

#ifndef  ALEPH_THRMAP_HPP
#include "Thrmap.hpp"
#endif

namespace aleph {

  /// The Regex class is a builtin object used to represent and evaluate
  /// regular expression. A regular expression is constructed with the
  /// a string implicitely or explicitely with operators.
  /// @author amaury darsch

  class Regex : public Literal {
  private:
    /// the regex value
    String d_reval;
    /// the regex structure
    struct s_regex* p_recni;
    /// the group vector in map
    mutable Thrmap d_gmap;

  public:
    /// create a new regex
    Regex (void);

    /// create a new regex from a string
    /// @param re the regex description
    Regex (const String& re);

    /// copy construct a regex
    /// @param that the regex to copy
    Regex (const Regex& that);

    /// destroy this regex
    ~Regex (void);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this regex
    String toliteral (void) const;

    /// @return a string representation of this regex
    String tostring (void) const;

    /// @return a clone of this regex
    Object* clone (void) const;

    /// @return the regex serial code
    t_byte serialid (void) const;

    /// serialize this regex to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a regex from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// compile a regex as a string
    /// @param re the regex string
    void compile (const String& re);

    /// match a regex against a string
    /// @param s the string to match
    bool operator == (const String& s);

    /// reverse match of regex against a string
    /// @param s the string to match
    bool operator != (const String& s);

    /// match a regex partially against a string
    /// @param s the string to match
    bool operator < (const String& s);

    /// match a regex with a string and return the substring
    /// @param s the string to match
    String match (const String& s);

    /// replace a regex match with a string
    /// @param s the string to match
    /// @param val the value to replace
    String replace (const String& s, const String& val);

    /// @return the length of the group vector
    long length (void) const;

    /// get a group object at a certain index
    /// @param index the group index
    Object* getobj (const long index) const;

    /// get a group string at a certain index
    /// @param index the group index
    String getstr (const long index) const;

    /// get a group integer at a certain index
    /// @param index the group index
    t_long getint (const long index) const;

    /// get a group real at a certain index
    /// @param index the group index
    t_real getreal (const long index) const;

    /// generate a new regex
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this regex with a literal object
    /// @param robj   the current runnable
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (Runnable* robj, t_oper type, Object* object);

    /// apply this integer with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  private:
    // make the assignment operator private
    Regex& operator = (const Regex&);
  };
}

#endif
