// ---------------------------------------------------------------------------
// - Digest.cpp                                                              -
// - aleph:txt library - message digest class implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Digest.hpp"
#include "Buffer.hpp"

namespace aleph {

  // the digest supported quarks
  static const long QUARK_COMPUTE = String::intern ("compute");

  // -------------------------------------------------------------------------
  // - MD5 algorithm - rfc 1321 - Ron Rivest                                 -
  // -------------------------------------------------------------------------

  // basic functions
  static inline t_quad rsl (t_quad x, long n) {
    return ((x << n) | (x >> (32 - n)));
  }
  static void btoq (t_quad* dst, t_byte* src, long size) {
    for (long i = 0, j = 0; j < size; i++, j += 4)
      dst[i] =   ((t_quad) src[j])          | 
	        (((t_quad) src[j+1]) << 8)  |
	        (((t_quad) src[j+2]) << 16) | 
	        (((t_quad) src[j+3]) << 24);
  }
  static void qtob (t_byte* dst, t_quad* src, long size) {
    for (long i = 0, j = 0; i < size; i++, j += 4) {
      dst[j]   = (t_byte)  (src[i]        & 0xff);
      dst[j+1] = (t_byte) ((src[i] >> 8)  & 0xff);
      dst[j+2] = (t_byte) ((src[i] >> 16) & 0xff);
      dst[j+3] = (t_byte) ((src[i] >> 24) & 0xff);
    }
  }
  static void otob (t_byte* dst, t_octa* src, long size) {
    for (long i = 0, j = 0; i < size; i++, j += 8) {
      dst[j]   = (t_byte)  (src[i]        & 0xff);
      dst[j+1] = (t_byte) ((src[i] >> 8)  & 0xff);
      dst[j+2] = (t_byte) ((src[i] >> 16) & 0xff);
      dst[j+3] = (t_byte) ((src[i] >> 24) & 0xff);
      dst[j+4] = (t_byte) ((src[i] >> 32) & 0xff);
      dst[j+5] = (t_byte) ((src[i] >> 40) & 0xff);
      dst[j+6] = (t_byte) ((src[i] >> 48) & 0xff);
      dst[j+7] = (t_byte) ((src[i] >> 56) & 0xff);
    }
  }
  static char btoc (const char byte, bool flag) {
    char data = flag ? byte & 0x0f : (byte >> 4) & 0x0f;
    if (data < 10) return '0' + data;
    return 'A' + (data - 10);
  }
  static String dtos (const t_byte* src, long size) {
    String result;
    for (long i = 0; i < size; i++) {
      result = result + btoc (src[i], false);
      result = result + btoc (src[i], true);
    }
    return result;
  }
  // MD5 constants
  const long   S11 =  7;
  const long   S12 = 12;
  const long   S13 = 17;
  const long   S14 = 22;
  const long   S21 =  5;
  const long   S22 =  9;
  const long   S23 = 14;
  const long   S24 = 20;
  const long   S31 =  4;
  const long   S32 = 11;
  const long   S33 = 16;
  const long   S34 = 23;
  const long   S41 =  6;
  const long   S42 = 10;
  const long   S43 = 15;
  const long   S44 = 21;
  // MD5 message padding
  const t_byte PAD[64] = {
    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };
  // MD5 basic functions
  static inline t_quad F (t_quad x, t_quad y, t_quad z) {
    return ((x & y) | ((~x) & z));
  }
  static inline t_quad G (t_quad x, t_quad y, t_quad z) {
    return ((x & z) | (y & (~z)));
  }
  static inline t_quad H (t_quad x, t_quad y, t_quad z) {
    return (x ^ y ^ z);
  }
  static inline t_quad I (t_quad x, t_quad y, t_quad z) {
    return (y ^ (x | (~z)));
  }
  // transformations for rounds 1, 2, 3, 4
  static inline void FF (t_quad& a, t_quad b, t_quad c, t_quad d, t_quad x, 
			 long s, t_quad ac) {
    a += F   (b, c, d) + x + ac;
    a  = rsl (a, s);
    a += b;
  }
  static inline void GG (t_quad& a, t_quad b, t_quad c, t_quad d, t_quad x, 
			 long s, t_quad ac) {
    a += G   (b, c, d) + x + ac;
    a  = rsl (a, s);
    a += b;
  }
  static inline void HH (t_quad& a, t_quad b, t_quad c, t_quad d, t_quad x, 
			 long s, t_quad ac) {
    a += H   (b, c, d) + x + ac;
    a  = rsl (a, s);
    a += b;
  }
  static inline void II (t_quad& a, t_quad b, t_quad c, t_quad d, t_quad x, 
			 long s, t_quad ac) {
    a += I   (b, c, d) + x + ac;
    a  = rsl (a, s);
    a += b;
  }

  // MD5 context
  struct s_mdctx {
    // ABCD states
    t_quad d_state[4];
    // the message length
    long d_length;
    // number of bytes in buffer
    long d_count;
    // block buffer
    t_byte p_buffer[64];
    // initialize the context
    s_mdctx (void) {
      reset ();
    }
    // reset this context
    void reset (void) {
      d_state[0] = 0x67452301;
      d_state[1] = 0xefcdab89;
      d_state[2] = 0x98badcfe;
      d_state[3] = 0x10325476;
      d_count    = 0;
      d_length   = 0;
    }
    // copy some bytes in the buffer and update the counter
    void copy (const t_byte* data, long& size) {
      long i = 0;
      while (d_count != 64) {
	p_buffer[d_count++] = data[i++];
	d_length++;
	if (--size == 0) break;
      }
    }
    // update the context with a context block
    void update (void) {
      // initialize state values
      t_quad a = d_state[0];
      t_quad b = d_state[1];
      t_quad c = d_state[2];
      t_quad d = d_state[3];

      // decode a block in 16 quads
      t_quad x[16]; btoq (x, p_buffer, 64);
      // round 1
      FF (a, b, c, d, x[ 0], S11, 0xd76aa478);
      FF (d, a, b, c, x[ 1], S12, 0xe8c7b756);
      FF (c, d, a, b, x[ 2], S13, 0x242070db);
      FF (b, c, d, a, x[ 3], S14, 0xc1bdceee);
      FF (a, b, c, d, x[ 4], S11, 0xf57c0faf);
      FF (d, a, b, c, x[ 5], S12, 0x4787c62a);
      FF (c, d, a, b, x[ 6], S13, 0xa8304613);
      FF (b, c, d, a, x[ 7], S14, 0xfd469501);
      FF (a, b, c, d, x[ 8], S11, 0x698098d8);
      FF (d, a, b, c, x[ 9], S12, 0x8b44f7af);
      FF (c, d, a, b, x[10], S13, 0xffff5bb1);
      FF (b, c, d, a, x[11], S14, 0x895cd7be);
      FF (a, b, c, d, x[12], S11, 0x6b901122);
      FF (d, a, b, c, x[13], S12, 0xfd987193);
      FF (c, d, a, b, x[14], S13, 0xa679438e);
      FF (b, c, d, a, x[15], S14, 0x49b40821);
      // round 2
      GG (a, b, c, d, x[ 1], S21, 0xf61e2562);
      GG (d, a, b, c, x[ 6], S22, 0xc040b340);
      GG (c, d, a, b, x[11], S23, 0x265e5a51);
      GG (b, c, d, a, x[ 0], S24, 0xe9b6c7aa);
      GG (a, b, c, d, x[ 5], S21, 0xd62f105d);
      GG (d, a, b, c, x[10], S22,  0x2441453);
      GG (c, d, a, b, x[15], S23, 0xd8a1e681);
      GG (b, c, d, a, x[ 4], S24, 0xe7d3fbc8);
      GG (a, b, c, d, x[ 9], S21, 0x21e1cde6);
      GG (d, a, b, c, x[14], S22, 0xc33707d6);
      GG (c, d, a, b, x[ 3], S23, 0xf4d50d87);
      GG (b, c, d, a, x[ 8], S24, 0x455a14ed);
      GG (a, b, c, d, x[13], S21, 0xa9e3e905);
      GG (d, a, b, c, x[ 2], S22, 0xfcefa3f8);
      GG (c, d, a, b, x[ 7], S23, 0x676f02d9);
      GG (b, c, d, a, x[12], S24, 0x8d2a4c8a);
      // round 3
      HH (a, b, c, d, x[ 5], S31, 0xfffa3942);
      HH (d, a, b, c, x[ 8], S32, 0x8771f681);
      HH (c, d, a, b, x[11], S33, 0x6d9d6122);
      HH (b, c, d, a, x[14], S34, 0xfde5380c);
      HH (a, b, c, d, x[ 1], S31, 0xa4beea44);
      HH (d, a, b, c, x[ 4], S32, 0x4bdecfa9);
      HH (c, d, a, b, x[ 7], S33, 0xf6bb4b60);
      HH (b, c, d, a, x[10], S34, 0xbebfbc70);
      HH (a, b, c, d, x[13], S31, 0x289b7ec6);
      HH (d, a, b, c, x[ 0], S32, 0xeaa127fa);
      HH (c, d, a, b, x[ 3], S33, 0xd4ef3085);
      HH (b, c, d, a, x[ 6], S34,  0x4881d05);
      HH (a, b, c, d, x[ 9], S31, 0xd9d4d039);
      HH (d, a, b, c, x[12], S32, 0xe6db99e5);
      HH (c, d, a, b, x[15], S33, 0x1fa27cf8);
      HH (b, c, d, a, x[ 2], S34, 0xc4ac5665);
      // round 4
      II (a, b, c, d, x[ 0], S41, 0xf4292244);
      II (d, a, b, c, x[ 7], S42, 0x432aff97);
      II (c, d, a, b, x[14], S43, 0xab9423a7);
      II (b, c, d, a, x[ 5], S44, 0xfc93a039);
      II (a, b, c, d, x[12], S41, 0x655b59c3);
      II (d, a, b, c, x[ 3], S42, 0x8f0ccc92);
      II (c, d, a, b, x[10], S43, 0xffeff47d);
      II (b, c, d, a, x[ 1], S44, 0x85845dd1);
      II (a, b, c, d, x[ 8], S41, 0x6fa87e4f);
      II (d, a, b, c, x[15], S42, 0xfe2ce6e0);
      II (c, d, a, b, x[ 6], S43, 0xa3014314);
      II (b, c, d, a, x[13], S44, 0x4e0811a1);
      II (a, b, c, d, x[ 4], S41, 0xf7537e82);
      II (d, a, b, c, x[11], S42, 0xbd3af235);
      II (c, d, a, b, x[ 2], S43, 0x2ad7d2bb);
      II (b, c, d, a, x[ 9], S44, 0xeb86d391);
      // state update
      d_state[0] += a;
      d_state[1] += b;
      d_state[2] += c;
      d_state[3] += d;
      // fix counter
      d_count   = 0;
    }
    // finish a message
    t_byte* finish (void) {
      // how much padding do we have to do - the message must be congruent
      // to 56 modulo 64 (aka 448 bits modulo 512)
      long len = d_length;
      long res = len % 64;
      long pad = (res < 56) ? 56 - res : 120 - res;
      // process with the padding
      process (PAD, pad);
      // now add the length - this is in bits !!!
      t_byte blen[8];
      t_octa bits = (t_octa) len << 3;
      otob (blen, &bits, 1);
      process (blen, 8);
      // create the message digest
      t_byte* result = new t_byte[16];
      qtob (result, d_state, 4);
      // reset the context and return
      reset ();
      return result;
    }
    // process a message - we copy/update until the buffer is empty
    void process (const t_byte* msg, long size) {
      while (size != 0) {
	copy (msg, size);
	if (d_count == 64) {
	  update ();
	  msg += 64;
	}
      }
    }
  };

  // -------------------------------------------------------------------------
  // - generic digest class                                                  -
  // -------------------------------------------------------------------------

  // create a default digest

  Digest::Digest (void) {
    d_algo = MD5;
    d_dctx = new s_mdctx;
  }

  // destroy this digest

  Digest::~Digest (void) {
    switch (d_algo) {
    case MD5:
      delete (s_mdctx*) d_dctx;
      break;
    }
  }

  // return the class name

  String Digest::repr (void) const {
    return "Digest";
  }

  // compute a message digest from a string

  String Digest::compute (const String& msg) {
    wrlock ();
    long    mlen   = msg.length ();
    t_byte* data   = (t_byte*) msg.tochar ();
    t_byte* digest = nilp;
    // check for digest algorithm
    if (d_algo == MD5) {
      s_mdctx* ctx = (s_mdctx*) d_dctx;
      ctx->process (data, mlen);
      digest = ctx->finish ();
    }
    String result = dtos (digest, 16);
    unlock ();
    return result;
  }

  // create a new digest object

  Object* Digest::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Digest;
    // invalid arguments
    throw Exception ("argument-error", "too many arguments for digest");
  }

  // apply a digest method with a set of arguments and a quark
  
  Object* Digest::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 arguments
    if (argc == 1) {
      if (quark == QUARK_COMPUTE) {
	Object* obj = argv->get (0);
	// check for a literal
	Literal* lval = dynamic_cast <Literal*> (obj);
	if (lval != nilp) {
	  String msg = lval->tostring ();
	  return new String (compute (msg));
	}
	// check for a buffer
	Buffer* bval = dynamic_cast <Buffer*> (obj);
	if (bval != nilp) {
	  String msg = bval->tostring ();
	  return new String (compute (msg));
	}
	throw Exception ("type-error", "invalid object for digest compute",
			 Object::repr (obj));
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

}
