// ---------------------------------------------------------------------------
// - Options.hpp                                                             -
// - aleph interpreter - options class implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "System.hpp"
#include "Options.hpp"
#include "OutputTerm.hpp"

// default assert flag
#ifdef DEBUG
const bool ASSERT_DEFAULT = true;
#else
const bool ASSERT_DEFAULT = false;
#endif

namespace aleph {

  // print a usage message when options are wrongs
  static void usage (void) {
    OutputTerm terr (OutputTerm::ERROR);
  
    terr << "usage: aleph [options] [file] [arguments]\n";
    terr << "             [-h]             print this help message\n";
    terr << "             [-v]             print version information\n";
    terr << "             [-i  path]       add a path to the resolver\n";
    terr << "             [-f  assert]     enable assertion checking\n";
    terr << "             [-f  nopath]     do not set initial path\n";
  }

  // print the aleph system version
  static void version (void) {
    OutputTerm terr (OutputTerm::ERROR);
    terr << "aleph interpreter " << System::osname () << " revision ";
    terr << System::version () << "\n";
  }

  // create a new option class by parsing the argument array

  Options::Options (int argc, char** argv) {
    // set default values
    d_assert = ASSERT_DEFAULT;
    d_noipth = false;
    // register the program name
    System::setpgm (argv[0]);
    // prepare for options
    OutputTerm terr(OutputTerm::ERROR);
    String     fvalue;
    // process options
    char optsep = System::optsep ();
    int  count  = 1;
    bool fflag  = false;
    while (count < argc) {
      String arg = argv[count];
      // check if we have an option
      if ((fflag == false) && (arg[0] == optsep)) {
	// check option length
	if (arg.length () != 2) {
	  terr << "fatal: illegal option " << arg << "\n";
	  System::exit (1);
	}
	// switch between options
	switch (arg[1]) {
	case 'h':
	  // print a usage message
	  usage ();
	  System::exit (0);	  
	case 'v':
	  // print system information
	  version ();
	  System::exit (0);	
	case 'f':
	  // check for enough arguments
	  if (++count >= argc) {
	    terr << "fatal: missing option name after option [f]\n";
	    System::exit (1);
	  }	
	  // process the option value
	  fvalue = argv[count++];
	  if (fvalue == "assert") {
	    d_assert = true;
	    continue;
	  }
	  if (fvalue == "nopath") {
	    d_noipth = true;
	    continue;
	  }
	  terr << "fatal: illegal flag " << fvalue << "\n";
	  System::exit (1);	  
	case 'i':
	  // check for enough arguments
	  if (++count >= argc) {
	    terr << "fatal: missing path name after option [i]\n";
	    System::exit (1);
	  }
	  d_path.add (argv[count++]);
	  continue;
	default:
	  // invalid option
	  terr << "fatal: illegal option " << arg << "\n";
	  System::exit (1);
	}      
      } else {
	if (fflag == false) {
	  d_fname = argv[count++];
	  fflag   = true;
	} else
	  d_argv.add (argv[count++]);
	continue;
      }
    }
  }
}
