;;; aleph-mode.el --- major mode for editing aleph source in emacs

;;  This program is  free software;  you can  redistribute it and/or modify
;;  it under the  terms of the GNU General Public License  as published  by
;;  the  Free Software Foundation; either version  2 of the License, or any
;;  later version.
;;  This  program  is  distributed in the hope  that it will be useful, but
;;  without  any  warranty;  without  even   the   implied    warranty   of
;;  merchantability or fitness for a particular purpose.
;;  See the GNU General Public License for more details.
;;
;;  Copyright (C) 1999-2003 Philippe Troin
;;
;;  last updated May 2003 with more gizmos.

(eval-when-compile 
  (require 'font-lock))

;;  first things first - basic definitions

(defgroup aleph nil
  "Major mode for editing Aleph source in Emacs"
  :group 'languages)

(defvar aleph-mode-abbrev-table nil
  "Abbreviation table used in Aleph mode")
(define-abbrev-table 'aleph-mode-abbrev-table ())

(defvar aleph-mode-map nil
  "Keymap used in Aleph mode")

(defvar aleph-mode-syntax-table nil
  "Syntax table used in Aleph mode")

(defvar aleph-indent-level 2
  "Amount by which aleph subexpressions are indented.")

;; mode map definition

(unless aleph-mode-map
  (setq aleph-mode-map (make-sparse-keymap))
  (define-key aleph-mode-map "\C-c\C-c" 'aleph-comment-region))

;; syntax table definition

(unless aleph-mode-syntax-table
  (setq aleph-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?+   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?-   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?*   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?/   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?\.  "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?:   "_"  aleph-mode-syntax-table)
  (modify-syntax-entry ?=   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?<   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?>   "w"  aleph-mode-syntax-table)
  (modify-syntax-entry ?\{  "(}" aleph-mode-syntax-table)
  (modify-syntax-entry ?\}  "){" aleph-mode-syntax-table)
  (modify-syntax-entry ?\(  "()" aleph-mode-syntax-table)  
  (modify-syntax-entry ?\)  ")(" aleph-mode-syntax-table)
  (modify-syntax-entry ?\[  "(]" aleph-mode-syntax-table)  
  (modify-syntax-entry ?\]  ")[" aleph-mode-syntax-table)
  (modify-syntax-entry ?\#  "<"  aleph-mode-syntax-table)
  (modify-syntax-entry ?\n  ">"  aleph-mode-syntax-table))

;; font-lock support

(defconst aleph-font-lock-keywords
  (eval-when-compile
    (list
     ;; keywords
     (cons
      (concat 
       "\\b\\("
       (mapconcat (lambda (x) x) 
		  '("do" "if" "trans" "eval" "try" "for" "switch" "launch"
		    "while" "const" "throw" "protect" "block" "class" "assert"
		    "lambda" "gamma" "else" "force" "global" "sync" "daemon"
		    "delay" "nameset" "return" "loop" "println" "print"
		    "errorln" "error" "args")
		  "\\|")
       "\\)\\b")
      font-lock-keyword-face)
     ;; Built-in functions (this face is not supported by X-Emacs)
     (unless (featurep 'xemacs)
       (cons
	(concat 
	 "\\b\\("
	 (mapconcat (lambda (x) x) 
		    '("\\sw+-p")
		    "\\|")
	 "\\)\\b")
	font-lock-builtin-face))
     ;; Built-in constants (this face is not supported by X-Emacs)
     (unless (featurep 'xemacs)
       (cons
	(concat 
	 "\\b\\("
	 (mapconcat (lambda (x) x) 
		    '("nil" "true" "false" "self" "this")
		    "\\|")
	 "\\)\\b")
	font-lock-constant-face))
     ;; Built-in types (this face is not supported by X-Emacs)
     (unless (featurep 'xemacs)
       (cons
	(concat 
	 "\\b\\("
	 (mapconcat (lambda (x) x) 
		    '("Real" "Boolean" "Integer" "String" "Character" "Cons"
		      "Vector" "List" "Regex" "Graph" "Node" "Edge")
		    "\\|")
	 "\\)\\b")
	font-lock-constant-face))
     ;; Function declaration
     '("\\b\\(trans\\|const\\)\\s-+\\(\\(\\sw\\|\\s_\\)+\\)\\s-*\\((\\(([^)
]*)\\|[^()
]\\)*)\\|nil\\)\\s-*[({]" 
       2 font-lock-function-name-face keep)
     ;; Variable declaration
     '("\\b\\(const\\|trans\\)\\s-+\\(\\(\\sw\\|\\s_\\)+\\)" 
       2 font-lock-variable-name-face keep)
     ))
  "aleph-mode font lock definitions")

;; major mode starts here

(defun aleph-mode ()
  "This is the major mode for editing aleph code in Emacs"

  ; house-keeping first
  (interactive)
  (kill-all-local-variables)

  ; install keymaps
  (use-local-map aleph-mode-map)

  ; set local abbreviation table
  (setq local-abbrev-table aleph-mode-abbrev-table)

  ; set syntax table
  (set-syntax-table aleph-mode-syntax-table)

  ; indentation command
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'aleph-indent-line)

  ; set comment start & end
  (make-local-variable 'comment-start)
  (setq comment-start "# ")
  (make-local-variable 'comment-end)
  (setq comment-end "")

  ; sexp ignore comment
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments t)

  ; define paragraph
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "$\\|" page-delimiter))

  ; install font lock
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(aleph-font-lock-keywords nil nil))

  ; let emacs know who we are
  (setq mode-name "aleph")
  (setq major-mode 'aleph-mode)

  ; run a hook if it exists
  (run-hooks 'aleph-mode-hook))

;; compute the line indentation

(defun aleph-count-blanks-at-bol (&optional pos blank-is-zero)
  "Returns the number of blanks at the beginning of POS (or the point if POS
is nil). If line is only blank-filled and BLANK-IS-ZERO is non-nil, returns 0."
  (save-excursion
    (save-match-data
      (if pos (goto-char pos))
      (beginning-of-line)
      (if blank-is-zero
	  (while (looking-at "^\\s-*$")
	    (beginning-of-line 0)))
      (re-search-forward "^\\s-*")
      (if (and (eolp) blank-is-zero)
	  0
	(current-column)))))

(defun aleph-get-indent-delta-of-region (begin end)
  "Returns the delta in indent-level between BEGIN and END."
  (save-excursion
    (save-match-data
      (goto-char begin)
      (while (looking-at "\\s-*$")
	(beginning-of-line 0))
      (re-search-forward "^\\s-*\\s)*")
      (let ((count 0))
	(while (re-search-forward "\\s(\\|\\s)" end t)
	  (let 
	      ((matched-char-syntax 
		(char-syntax (string-to-char (match-string 0)))))
	    (cond
	     ((eq matched-char-syntax ?\() (setq count (1+ count)))
	     ((eq matched-char-syntax ?\)) (setq count (1- count))))))
	count))))

(defun aleph-get-indent-delta-of-cc-at-bol (&optional pos)
  "Returns the (always negative) indent level taking into account
only closing parenthesis characters at the beginning of the line containing
by POS (or the point if POS is nil)."
  (save-excursion
    (save-match-data
    (if pos (goto-char pos))
    (beginning-of-line)
    (re-search-forward "^\\s-*")
    (let ((level 0))
      (while (eq (char-syntax (let ((ca (char-after)))
				(or ca 0)))
		 ?\))
	(setq level (1- level))
	(forward-char))
      level))))

(defun aleph-put-point-at-bol-after-blanks (&optional pos)
  "Put the point in the current buffer at the beginning of the line
contained by POS (or the point if POS is nil), but after the leading blanks."
  (if pos (goto-char pos))
  (beginning-of-line)
  (re-search-forward "^\\s-*"))
  
(defun aleph-get-theorical-indent-of-line (&optional pos)
  "Returns the indentation offset of line at POS (or the point)."
  (save-excursion
    (if pos (goto-char pos))
    (let (c-line-pos p-line-pos)
    (beginning-of-line)
    (setq c-line-pos (point))
    (if (eq (point) (point-min))
	0
      (beginning-of-line 0)
      (setq p-line-pos (point))
      (+ (aleph-count-blanks-at-bol p-line-pos t)
	 (* aleph-indent-level
	    (+ (aleph-get-indent-delta-of-region p-line-pos c-line-pos)
	       (aleph-get-indent-delta-of-cc-at-bol c-line-pos))))))))

(defun aleph-indent-line ()
  "Indent current line as aleph code."
  (let ((c-indent (aleph-count-blanks-at-bol))
	(t-indent (aleph-get-theorical-indent-of-line)))
    (save-excursion
      (if (< t-indent 0) 
	  (setq t-indent 0))
      (aleph-put-point-at-bol-after-blanks)
      (cond
       ((< c-indent t-indent)
	(insert-char ?\  (- t-indent c-indent)))
       ((> c-indent t-indent)
	(backward-delete-char-untabify (- c-indent t-indent)))))
    (if (looking-at "\\s-*$")
	(end-of-line))))

;; and tell emacs we are here

(provide 'aleph-mode)
