;; AIscm - Guile extension for numerical arrays and tensors.
;; Copyright (C) 2013, 2014, 2015, 2016, 2017 Jan Wedekind <jan@wedesoft.de>
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;
(use-modules (aiscm element)
             (aiscm bool)
             (aiscm int)
             (aiscm jit)
             (oop goops)
             (guile-tap))
(define bool-false (make <bool> #:value #f))
(define bool-true (make <bool> #:value #t))
(ok (equal? bool-false bool-false)
    "equality of booleans")
(ok (not (equal? bool-false bool-true))
    "equality of booleans")
(ok (not (equal? bool-true bool-false))
    "equality of booleans")
(ok (equal? bool-true bool-true)
    "equality of booleans")
(ok (not (get bool-false))
    "get boolean value from bool-false")
(ok (get bool-true)
    "get boolean value from bool-true")
(ok (not (equal? bool-true bool-false))
    "unequal boolean objects")
(ok (eqv? 1 (size-of <bool>))
    "storage size of booleans")
(ok (equal? #vu8(0) (pack bool-false))
    "pack 'false' value")
(ok (equal? #vu8(1) (pack bool-true))
    "pack 'true' value")
(ok (eqv? 1 (size bool-true))
    "querying element size of boolean")
(ok (null? (shape bool-true))
    "querying shape of boolean")
(ok (equal? bool-false (unpack <bool> #vu8(0)))
    "unpack 'false' value")
(ok (equal? bool-true (unpack <bool> #vu8(1)))
    "unpack 'true' value")
(ok (equal? "#<<bool> #f>"
            (call-with-output-string (lambda (port) (display bool-false port))))
    "display boolean object")
(ok (equal? "#<<bool> #f>"
            (call-with-output-string (lambda (port) (write bool-false port))))
    "write boolean object")
(ok (equal? <bool> (native-type #f))
    "type matching for #f")
(ok (equal? <bool> (native-type #t))
    "type matching for #t")
(ok (equal? <bool> (native-type #f #t))
    "type matching for multiple booleans")
(ok (get bool-true)
    "get value of true")
(ok (not (get bool-false))
    "get value of false")
(ok (let [(b (make <bool> #:value #f))] (set b #t) (get b))
    "set boolean to true")
(ok (not (let [(b (make <bool> #:value #t))] (set b #f) (get b)))
    "set boolean to false")
(ok (set (make <bool> #:value #f) #t)
    "return value of setting boolean to true")
(ok (not (set (make <bool> #:value #t) #f))
    "return value of setting boolean to false")
(ok (eq? #t (build <bool> '(#t)))
    "build boolean 'true'")
(ok (eq? #f (build <bool> '(#f)))
    "build boolean 'false'")
(ok (equal? '(0) (unbuild <bool> #f))
    "'unbuild' returns 0 for false")
(ok (equal? '(1) (unbuild <bool> #t))
    "'unbuild' returns 1 for true")
(ok (equal? '(#f #f #f #t) (map && '(#f #f #t #t) '(#f #t #f #t)))
    "'&&' behaves like 'and'")
(ok (not (&& #t #t #f))
    "'&&' with three arguments")
(ok (&& #t #t #t #t)
    "'&&' with four arguments")
(ok (equal? '(#f #t #t #t) (map || '(#f #f #t #t) '(#f #t #f #t)))
    "'||' behaves like 'or'")
(ok (not (|| #f #f #f))
    "'||' with three arguments")
(ok (|| #f #f #f #t)
    "'||' with four arguments")
(ok (equal? '(#t #f) (map ! '(#f #t)))
   "'!' is like 'not'")
(ok (pointerless? <bool>)
    "boolean memory is pointerless")
(run-tests)
