/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Copyright (c) 2003, Snax
*/

// wlancap.c : Defines the entry point for the DLL application.
//

#include "wlancap.h"
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

LONG lastError;

OSVERSIONINFO osversion;
CRITICAL_SECTION criticalSection;
HMODULE hModule;

LPLONG pRefCount;
HANDLE hFileMappingObject;
UINT uidEvent;

__int64 performanceCount;

__int64 timeStamp;

__int64 performanceFrequency;

BOOL APIENTRY DllMain( HANDLE hinstDLL, DWORD  reason, LPVOID lpReserved) {
   hModule = (HMODULE)hinstDLL;
   switch (reason) {
		case DLL_PROCESS_ATTACH:
         return WincapInitializeLibrary();
		case DLL_PROCESS_DETACH:
         return WincapCleanupLibrary();
    }
    return TRUE;
}

//set the error state of the library
LONG setError(LONG value) {
   return InterlockedExchange(&lastError, value);
}

LONG doIOCTL(HANDLE hFile, WincapReqBuffer *prb, OVERLAPPED *overlapped, UINT code) {
   LONG error = 0xC0010012;
   DWORD bytes;
   if (hFile == INVALID_HANDLE_VALUE) {
      setError(error);
      return error;
   }
   if (prb && overlapped && overlapped->hEvent) {
      if (!ResetEvent(overlapped->hEvent)) {
         error = GetLastError();
         setError(error);
         return error;
      }
      code = code ? 0x80000018 : 0x80000010;
      prb->result = 0xC0000001;
      if (DeviceIoControl(hFile, code, prb, sizeof(WincapReqBuffer), prb,
                      sizeof(WincapReqBuffer), &bytes, overlapped) == 0) {
         error = GetLastError();
         if (error == 0x3E5) {
            if (GetOverlappedResult(hFile, overlapped, &bytes, 1) == 0) {
               error = GetLastError();
               setError(error);
               return error;
            }
         }
         else if (error) {
            setError(error);
            return error;
         }
      }
      error = prb->result;
      setError(error);
      if (WincapIsWindowsNT() && !error && (prb->field_14 == 0 || prb->field_14 == 2)) {
         memmove(prb->pDest, &prb->dataBuf, prb->count);
      }
      return error;
   }
//errorReturn:
   setError(0x57);
   return 0x57;
}

BOOL checkFile(char *prefix) {
   char fName[260];
   HANDLE h;
   LONG error;
   if (prefix == NULL) {
      return FALSE;
   }
   sprintf(fName, "%s%s", "\\\\.\\", prefix);
   if ((h = CreateFile(fName, 0xC0000000, 0, NULL, 3, 0x60000080, 0)) != INVALID_HANDLE_VALUE) {
      CloseHandle(h);
      return TRUE;
   }
   error = GetLastError();
   setError(error);
   return error == 0;
}

BOOL loadService(char *displayName, char *binPathName, char *alternateName) {
   LONG error;
   SC_HANDLE mgr;
   SC_HANDLE svc;
   SC_LOCK lock;
   char *alt;
   if ((mgr = OpenSCManager(NULL, NULL, 0xF003F)) == 0) {
      setError(GetLastError());
      return FALSE;
   }
   while ((lock = LockServiceDatabase(mgr)) == 0) {
      Sleep(100);
   }
   svc = OpenService(mgr, displayName, 0x20000000);
   error = GetLastError();
   if (svc == 0) {
      alt = alternateName ? alternateName : displayName;
      svc = CreateService(mgr, displayName, alt, 0x20000000, 1, 3, 1,
                              binPathName, NULL, NULL, NULL, NULL, NULL);
      error = GetLastError();
   }
   UnlockServiceDatabase(lock);
   if (svc) {
      error = 0;
      CloseServiceHandle(svc);
   }
   else if (error == 0x431) {
      error = 0;
   }
   CloseServiceHandle(mgr);
   setError(error);
   return error == 0;
}

BOOL startService(char *displayName, char *binPathName, char *alternateName) {
   char *svcName;
   SC_HANDLE mgr;
   SC_HANDLE svc;
   LONG error = 0;
   BOOL result;
   setError(0);
   svcName = WincapIsWindows2000() ? "PEEK5" : "PEEK4";
   if (checkFile(svcName)) {
      return TRUE;
   }
   loadService(displayName, binPathName, alternateName);
   if ((mgr = OpenSCManager(NULL, NULL, 2)) == 0) {
      setError(GetLastError());
      return FALSE;
   }
   if ((svc = OpenService(mgr, displayName, 0xF01FF)) == 0) {
      setError(GetLastError());
      CloseServiceHandle(mgr);
      return FALSE;
   }
   if ((result = StartService(svc, 0, NULL)) == 0) {
      error = GetLastError();
      if (error == 0x420) {
         error = 0;
         result = TRUE;
      }
   }
   CloseServiceHandle(mgr);
   CloseServiceHandle(svc);
   setError(error);
   return result;
}

BOOL stopService() {
   SERVICE_STATUS ss;
   SC_HANDLE mgr;
   SC_HANDLE svc;
   SC_LOCK lock;
   char *svcName;
   if ((mgr = OpenSCManager(NULL, NULL, 0xF003F)) == 0) {
      setError(GetLastError());
      return FALSE;
   }
   svcName = WincapIsWindows2000() ? "PEEK5" : "PEEK4";
   if ((svc = OpenService(mgr, svcName, 0x100A0)) == 0) {
      setError(GetLastError());
      CloseServiceHandle(mgr);
      return FALSE;
   }
   ControlService(svc, 1, &ss);
   while ((lock = LockServiceDatabase(mgr)) == 0) {
      Sleep(100);
   }
   DeleteService(svc);
   UnlockServiceDatabase(lock);
   ControlService(svc, 4, &ss);
   CloseServiceHandle(svc);
   CloseServiceHandle(mgr);
   return TRUE;
}

int ya9xFunc(HANDLE hFile) {
   DWORD dest;
   OVERLAPPED overlapped;
   WincapReqBuffer prb;
   LONG error;
   if (hFile == INVALID_HANDLE_VALUE) {
      setError(0x57);
      return FALSE;
   }
   memset(&overlapped, 0, sizeof(OVERLAPPED));
   if ((overlapped.hEvent = CreateEvent(NULL, 0, 0, NULL)) == 0) {
      setError(GetLastError());
      return FALSE;
   }
   memset(&prb, 0, sizeof(WincapReqBuffer));
   prb.command = 0xF0020101;
   prb.pDest = &dest;
   prb.destLength = 4;
   if (error = WincapPrivateRequest(hFile, &prb, &overlapped)) {
      setError(error);
      CloseHandle(overlapped.hEvent);
      return FALSE;
   }
   CloseHandle(overlapped.hEvent);
   dest >>= 16;
   return dest == 0x401;
}

HANDLE getDeviceHandle() {
   char dir[256];
   char path[256];
   char drive[4];
   HANDLE hFile;
   char *svcName = WincapIsWindows2000() ? "PEEK5" : "PEEK4";
   setError(0);
   sprintf(path, "%s%s", "\\\\.\\", svcName);
   hFile = CreateFile(path, 0xC0000000, 0, NULL, 3, 0x60000080, 0);
   if (hFile == INVALID_HANDLE_VALUE) {
      GetModuleFileName(hModule, path, 260);
      _splitpath(path, drive, dir, NULL, NULL);
//      strcpy(path, "\\\\.\\");
      _makepath(path, drive, dir, svcName, ".SYS");
      GetShortPathName(path, path, 260);
      if (WincapLoadDriver(svcName, path, svcName) == 0) {
         return hFile;
      }
      sprintf(path, "%s%s", "\\\\.\\", svcName);
      if ((hFile = CreateFile(path, 0xC0000000, 3, NULL, 3, 0x60000080, 0)) == INVALID_HANDLE_VALUE) {
         setError(GetLastError());
         return hFile;
      }
   }
   if (ya9xFunc(hFile) == 0) {
      CloseHandle(hFile);
      setError(0x80040001);
      return INVALID_HANDLE_VALUE;
   }
   setError(0);
   return hFile;

}

void PeekTimerFunc(HWND h, UINT i, UINT j, DWORD d) {
   WincapSynchronizeTimeStamps();
}

BOOL WinNTCloser(HANDLE hFileAdapter) {
   LONG result;
   WincapReqBuffer prb;
   OVERLAPPED overlapped;
   int var_244 = 0;
   if (hFileAdapter == INVALID_HANDLE_VALUE) {
      setError(0x57);
      return FALSE;
   }
   memset(&overlapped, 0, sizeof(OVERLAPPED));
   if ((overlapped.hEvent = CreateEvent(NULL, 0, 0, NULL)) == 0) {
      return FALSE;
   }
   prb.field_14 = 0;
   prb.command = 0xF0020103;
   prb.pDest = &var_244;
   prb.destLength = 4;
   prb.count = 0;
   prb.field_28 = 0;
   result = WincapPrivateRequest(hFileAdapter, &prb, &overlapped);
   CloseHandle(overlapped.hEvent);
   if (result) {
      return FALSE;
   }
   if (var_244 & 1) {
      return FALSE;
   }
//   var_244 &= 0xFF;   //does nothing
   var_244 >>= 1;
   var_244 &= 1;
   return var_244;
}

LONG WinNTcloseAdapter(HANDLE hFileAdapter) {
   if (hFileAdapter == INVALID_HANDLE_VALUE) {
      setError(0x57);
      return 0x57;
   }
   if (pRefCount) {
      pRefCount[1] = WinNTCloser(hFileAdapter);
   }
   if (!CloseHandle(hFileAdapter)) {
      LONG error = GetLastError();
      setError(error);
      return error;
   }
   setError(0);
   return 0;
}

LONG Win9xcloseAdapter(HANDLE hFileAdapter) {
   if (hFileAdapter == INVALID_HANDLE_VALUE) {
      setError(0x57);
      return 0x57;
   }
   if (!CloseHandle(hFileAdapter)) {
      LONG error = GetLastError();
      setError(error);
      return error;
   }
   setError(0);
   return 0;
}

HANDLE checkDevice2(char *devName) {
   OVERLAPPED overlapped;
   DWORD bytes;
   LONG error = 0;
   HANDLE result;
   setError(0);
   if ((result = getDeviceHandle()) == INVALID_HANDLE_VALUE) {
      return result;
   }
   memset(&overlapped, 0, sizeof(OVERLAPPED));
   if ((overlapped.hEvent = CreateEvent(NULL, 1, 0, NULL)) == 0) {
      setError(GetLastError());
      CloseHandle(result);
      return INVALID_HANDLE_VALUE;
   }
   if (DeviceIoControl(result, 0x80000000, devName, strlen(devName), NULL, 0, &bytes, &overlapped) == 0) {
      error = GetLastError();
      if (error == 0x3E5) {
         if (GetOverlappedResult(result, &overlapped, &bytes, 1)) {
            CloseHandle(overlapped.hEvent);
            setError(0);
            return result;
         }
         error = GetLastError();
      }
      if (error) {
         CloseHandle(result);
         result = INVALID_HANDLE_VALUE;
      }
   }
   CloseHandle(overlapped.hEvent);
   setError(error);
   return result;
}

HANDLE WinNTcheckDevice(char *devName) {
   OVERLAPPED overlapped;
   HANDLE result;
   DWORD bytes;
   LONG error;
   setError(0);
   if ((result = checkDevice2(devName)) != INVALID_HANDLE_VALUE) {  //the only differenc is this function call
      memset(&overlapped, 0, sizeof(OVERLAPPED));
      if ((overlapped.hEvent = CreateEvent(NULL, 1, 0, NULL)) == 0) {
         setError(GetLastError());
         CloseHandle(result);
         return INVALID_HANDLE_VALUE;
      }

      if (DeviceIoControl(result, 0x80000008, devName, strlen(devName),
                          NULL, 0, &bytes, &overlapped) == 0) {
         if ((error = GetLastError()) == 0x3E5) {
            if (GetOverlappedResult(result, &overlapped, &bytes, 1)) {
               CloseHandle(overlapped.hEvent);
               setError(0);
               return result;
            }
            error = GetLastError();
         }
         if (error) {
            CloseHandle(result);
            result = INVALID_HANDLE_VALUE;
         }
      }
      CloseHandle(overlapped.hEvent);
      setError(error);
   }
   return result;
}

HANDLE Win9xLoadDriver() {
   char path[260];
   char dir[256];
   char drive[4];
   HANDLE hFile;
   sprintf(path, "%s%s", "\\\\.\\", "PEEK3");
   hFile = CreateFile(path, 0xC0000000, 3, NULL, 3, 0x40000080, 0);
   if (hFile == INVALID_HANDLE_VALUE) {
      GetModuleFileName(hModule, path, 260);
      GetShortPathName(path, path, 260);
      _splitpath(path, drive, dir, NULL, NULL);
      strcpy(path, "\\\\.\\");
      _makepath(path + 4, drive, dir, "PEEK3", ".VXD");
      if ((hFile = CreateFile(path, 0xC0000000, 3, NULL, 1, 0x44000080, 0)) == INVALID_HANDLE_VALUE) {
         setError(GetLastError());
         return hFile;
      }
   }
   if (!ya9xFunc(hFile)) {
      CloseHandle(hFile);
      setError(0x80040001);
      return INVALID_HANDLE_VALUE;
   }
   setError(0);
   return hFile;
}

HANDLE Win9xCheckDevice2(char *devName) {
   OVERLAPPED overlapped;
   DWORD bytes;
   HANDLE result;
   LONG error = 0;
   setError(0);
   if ((result = Win9xLoadDriver()) != INVALID_HANDLE_VALUE) {
      memset(&overlapped, 0, sizeof(OVERLAPPED));
      if ((overlapped.hEvent = CreateEvent(NULL, 1, 0, NULL)) == 0) {
         setError(GetLastError());
         CloseHandle(result);
         return INVALID_HANDLE_VALUE;
      }

      if (DeviceIoControl(result, 0x80000000, devName, 4,
                          NULL, 0, &bytes, &overlapped) == 0) {
         if ((error = GetLastError()) == 0x3E5) {
            if (GetOverlappedResult(result, &overlapped, &bytes, 1) == 0) {
               error = GetLastError();
               if (error) {
                  CloseHandle(result);
                  result = INVALID_HANDLE_VALUE;
               }
            }
            else {
               error = 0;
            }
         }
         else {
            if (error) {
               CloseHandle(result);
               result = INVALID_HANDLE_VALUE;
            }
         }
      }
      CloseHandle(overlapped.hEvent);
      if (overlapped.Offset) {
         error = overlapped.Offset;
         CloseHandle(result);
         result = INVALID_HANDLE_VALUE;
      }
      setError(error);
   }
   return result;
}

HANDLE Win9xCheckDevice(char *devName) {
   OVERLAPPED overlapped;
   HANDLE result;
   DWORD bytes;
   LONG error;
   setError(0);
   if ((result = Win9xCheckDevice2(devName)) != INVALID_HANDLE_VALUE) {  //the only differenc is this function call
      memset(&overlapped, 0, sizeof(OVERLAPPED));
      if ((overlapped.hEvent = CreateEvent(NULL, 1, 0, NULL)) == 0) {
         setError(GetLastError());
         CloseHandle(result);
         return INVALID_HANDLE_VALUE;
      }

      if (DeviceIoControl(result, 0x80000008, devName, 4,
                          NULL, 0, &bytes, &overlapped) == 0) {
         if ((error = GetLastError()) == 0x3E5) {
            if (GetOverlappedResult(result, &overlapped, &bytes, 1)) {
               CloseHandle(overlapped.hEvent);
               setError(0);
               return result;
            }
            error = GetLastError();
         }
         if (error) {
            CloseHandle(result);
            result = INVALID_HANDLE_VALUE;
         }
      }
      CloseHandle(overlapped.hEvent);
      setError(error);
   }
   return result;
}

HANDLE openDevice(char *devName) {
   char deviceName[260];
   if (WincapIsWindowsNT()) {
      strcpy(deviceName, devName);
      return WinNTcheckDevice(deviceName);
   }
   if (WincapIsWindows9X()) {
      return Win9xCheckDevice(devName);
   }
   setError(0x78);
   return INVALID_HANDLE_VALUE;
}

WLANCAP_API BOOL WincapCleanupLibrary() {
   if (pRefCount) {
      InterlockedDecrement(pRefCount);
      if (!pRefCount[0] && pRefCount[1]) {
         WincapUnloadDriver();
      }
      UnmapViewOfFile(pRefCount);
      pRefCount = 0;
      CloseHandle(hFileMappingObject);
      hFileMappingObject = 0;
   }
   if (uidEvent) {
      KillTimer(0, uidEvent);
      uidEvent = 0;
   }
   DeleteCriticalSection(&criticalSection);
   return TRUE;
}

WLANCAP_API LONG WincapClose(HANDLE hFileAdapter) {
   if (WincapIsWindowsNT()) {
      WinNTcloseAdapter(hFileAdapter);
   }
   if (WincapIsWindows9X()) {
      Win9xcloseAdapter(hFileAdapter);
   }
   setError(0x78);
   return 0x78;
}

WLANCAP_API Context *WincapCreateContext(HANDLE hFileAdapter, WincapCallback cBack,
                                              int ringSize, int channelMask, void *userPtr) {
   OVERLAPPED overlapped;
   WincapReqBuffer prb;
   Context *result;
   if (hFileAdapter != INVALID_HANDLE_VALUE && hFileAdapter != 0 && cBack != NULL) {
      if (result = (Context*) HeapAlloc(GetProcessHeap(), 0, sizeof(Context))) {
         memset(result, 0, sizeof(Context));
         result->ringSize = ringSize;
         result->channelMask = channelMask;
         result->cBack = cBack;
         result->hFileAdapter = hFileAdapter;
         result->userData = userPtr;
         memset(&overlapped, 0, sizeof(OVERLAPPED));
         if (overlapped.hEvent = CreateEvent(NULL, 0, 0, NULL)) {
            memset(&prb, 0, sizeof(WincapReqBuffer));
            prb.command = 0xFF636703;
            prb.pDest = &result->wincapRequestResult1;
            prb.destLength = 4;
            if (!WincapRequest(hFileAdapter, &prb, &overlapped)) {
               memset(&prb, 0, sizeof(WincapReqBuffer));
               prb.command = 0xFF636702;
               prb.pDest = &result->wincapRequestResult2;
               prb.destLength = 4;
               WincapRequest(hFileAdapter, &prb, &overlapped);
            }
            CloseHandle(overlapped.hEvent);
         }
         setError(0);
         return result;
      }
      else {
         setError(GetLastError());
         return NULL;
      }
   }
   setError(0x57);
   return NULL;
}

WLANCAP_API LONG WincapDestroyContext(Context *pContext) {
   if (!pContext) {
      setError(0x57);
      return 0x57;
   }
   WincapStop(pContext);
   HeapFree(GetProcessHeap(), 0, pContext);
   setError(0);
   return 0;
}

WLANCAP_API LONG WincapGetState(Context *pContext) {
   if (!pContext) {
      setError(0x57);
      return 0x57;
   }
   setError(0);
   return pContext->state;
}

WLANCAP_API LONG WincapGetLastError() {
   return InterlockedExchange(&lastError, 0);
}

WLANCAP_API RingBuffer *WincapGetPacketBuffer(Context *pContext) {
   if (!pContext) {
      setError(0x57);
      return NULL;
   }
   setError(0);
   return pContext->pRingBuffer;
}

WLANCAP_API BOOL WincapGetPacketBufferStats(Context *pContext, StatBuff *pStatBuff) {
   if (pContext && pStatBuff && pContext->pRingBuffer) {
      memcpy(pStatBuff, pContext->pRingBuffer->stats, 12 * sizeof(int));
      pStatBuff->time = WincapGetTimeStamp();
      setError(0);
      return TRUE;
   }
   setError(0x57);
   return FALSE;
}

WLANCAP_API unsigned __int64 WincapGetTimeStamp() {
   FILETIME localFileTime;
   FILETIME fileTime;
   GetSystemTimeAsFileTime(&fileTime);
   FileTimeToLocalFileTime(&fileTime, &localFileTime);
   return *((unsigned __int64*)&localFileTime) / (unsigned __int64)10 - (unsigned __int64)0x21F84030192000;
}

WLANCAP_API BOOL WincapInitializeLibrary() {
   int b7test;
   if ((hFileMappingObject =
           CreateFileMapping(INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE, 0,
                             8, "{73AF8993-F9F6-11d3-A256-00902744B1FF}")) == 0) {
      return FALSE;
   }
   b7test = GetLastError() != 0xB7;
   if ((pRefCount = (LPLONG)MapViewOfFile(hFileMappingObject, FILE_MAP_WRITE, 0, 0, 0)) == 0) {
      setError(GetLastError());
      return FALSE;
   }
   if (b7test) {
      *((__int64*)pRefCount) = 0;
//      pRefCount[0] = pRefCount[1] = 0;
   }
   InterlockedIncrement((LPLONG)pRefCount);
   memset(&osversion, 0, sizeof(osversion));
   osversion.dwOSVersionInfoSize = 0x94;
   GetVersionEx(&osversion);
   InitializeCriticalSection(&criticalSection);
   WincapSynchronizeTimeStamps();
   if (!uidEvent) {
      uidEvent = SetTimer(0, 1, 60000, (TIMERPROC)PeekTimerFunc);
   }
   setError(0);
   return TRUE;
}

WLANCAP_API BOOL WincapIsWindowsNT() {
   return osversion.dwPlatformId == VER_PLATFORM_WIN32_NT;
}

WLANCAP_API BOOL WincapIsWindows2000() {
   return osversion.dwPlatformId == VER_PLATFORM_WIN32_NT && osversion.dwMajorVersion == 5;
}

WLANCAP_API BOOL WincapIsWindows9X() {
   return osversion.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS;
}

WLANCAP_API BOOL WincapLoadDriver(char *displayName, char *binPathName, char *alternateName) {
   BOOL result;
   char path[260];
   char dir[256];
   char drive[4];
   HANDLE hFile;
   if (WincapIsWindowsNT()) {
      result = startService(displayName, binPathName, alternateName);
      if (lastError) {
         stopService();
         Sleep(100);
         result = startService(displayName, binPathName, alternateName);
      }
      return result;
   }
   if (!WincapIsWindows9X()) {
      setError(0x78);
      return FALSE;
   }
   setError(0);
   if (checkFile("PEEK3")) {
      return TRUE;
   }
   GetModuleFileName(hModule, path, 260);
   GetShortPathName(path, path, 260);
   _splitpath(path, drive, dir, NULL, NULL);
   strcpy(path, "\\\\.\\");
   _makepath(path + 4, drive, dir, "PEEK3", ".VXD");
   if ((hFile = CreateFile(path, 0xC0000000, 0, NULL, 1, 0x80, 0)) == INVALID_HANDLE_VALUE) {
      setError(GetLastError());
   }
   else {
      CloseHandle(hFile);
   }
   return checkFile("PEEK3");
}

WLANCAP_API HANDLE WincapOpenAdapter(char *adapterName) {
   HANDLE dev = openDevice(adapterName);
   LONG error;
   if (dev == INVALID_HANDLE_VALUE) {
      WincapUnloadDriver();
      Sleep(100);
      dev = openDevice(adapterName);
      error = WincapGetLastError();
      if (dev == INVALID_HANDLE_VALUE) {
         WincapUnloadDriver();
      }
      setError(error);
   }
   return dev;
}

WLANCAP_API LONG WincapPause(Context *pContext) {
   if (pContext && (pContext->state == PAUSED || pContext->state == RUNNING)) {
      pContext->state = PAUSED;
      setError(0);
      return 0;
   }
   setError(0x57);
   return 0x57;
}

WLANCAP_API LONG WincapPrivateRequest(HANDLE hFile, WincapReqBuffer *prb, LPOVERLAPPED overlapped) {
   return doIOCTL(hFile, prb, overlapped, 1);
}

WLANCAP_API LONG WincapResume(Context *pContext) {
   if (pContext && (pContext->state == PAUSED || pContext->state == RUNNING)) {
      if (pContext->state == PAUSED) {
         pContext->state = RUNNING;
         SetEvent(pContext->hResumeEvent);
      }
      setError(0);
      return 0;
   }
   setError(0x57);
   return 0x57;
}

WLANCAP_API LONG WincapRequest(HANDLE hFile, WincapReqBuffer *prb, LPOVERLAPPED overlapped) {
   return doIOCTL(hFile, prb, overlapped, 0);
}

WLANCAP_API LONG WincapSetCaptureThreadPriority(Context *pContext, int priority) {
   LONG error;
   if (pContext == NULL || pContext->hThread == 0) {
      setError(0x57);
      return 0x57;
   }
   if (SetThreadPriority(pContext->hThread, priority)) {
      setError(0);
      return 0;
   }
   error = GetLastError();
   setError(error);
   return error;
}

typedef HANDLE (*VXDPROC)(HANDLE);

WLANCAP_API LONG WincapSetPacketBuffer(Context *pContext, RingBuffer *pRingBuffer) {
   OVERLAPPED overlapped;
   int len = 0;
   DWORD bytes;
   LONG error = 0;
   if (pContext == NULL) {
      setError(0x57);
      return 0x57;
   }
   if (pRingBuffer) {
      len = pRingBuffer->size + 0x50;
   }
   memset(&overlapped, 0, sizeof(OVERLAPPED));
   if ((overlapped.hEvent = CreateEvent(NULL, 1, 0, NULL)) == 0) {
      error = GetLastError();
      setError(error);
      return error;
   }
   if (DeviceIoControl(pContext->hFileAdapter, 0x8000001F, pRingBuffer,
                       len, NULL, 0, &bytes, &overlapped) == 0) {
      error = GetLastError();
      if (error == 0x3E5) {
         error = 0;
         if (GetOverlappedResult(pContext->hFileAdapter, &overlapped, &bytes, 1) == 0) {
            error = GetLastError();
         }
      }
   }
   CloseHandle(overlapped.hEvent);
   setError(error);
   return error;
}

LONG CaptureThreadFunc(Context *pContext);

WLANCAP_API LONG WincapStart(Context *pContext) {
   DWORD threadId;
   if (!pContext) {
      setError(0x57);
      return 0x57;
   }
   if (pContext->state == RUNNING) {
      return 0;
   }
   if ((pContext->pRingBuffer = (RingBuffer*)HeapAlloc(GetProcessHeap(), 0, pContext->ringSize + 80)) == NULL) {
      LONG error = GetLastError();
      setError(error);
      return error;
   }
   memset(pContext->pRingBuffer, 0, 80);  //check this
   pContext->pRingBuffer->size = pContext->ringSize;
   pContext->pRingBuffer->hEvent = CreateEvent(NULL, 1, 0, NULL);
   if (WincapIsWindows9X()) {
      HMODULE hMod = LoadLibrary("KERNEL32.DLL");
      VXDPROC pa;
      if (!hMod) {
         LONG error = GetLastError();
         setError(error);
         return error;
      }
      pa = (VXDPROC) GetProcAddress(hMod, "OpenVxDHandle");
      if (!pa) {
         LONG error = GetLastError();
         setError(error);
         return error;
      }
      if ((pContext->pRingBuffer->hVxD = (pa)(pContext->pRingBuffer->hEvent)) == 0) {
         LONG error = GetLastError();
         setError(error);
         return error;
      }
   }
   WincapSetPacketBuffer(pContext, pContext->pRingBuffer);
   if ((pContext->hEvent1 = CreateEvent(NULL, 0, 0, NULL)) == 0) {
      LONG error = GetLastError();
      setError(error);
      return error;
   }

   if ((pContext->hResumeEvent = CreateEvent(NULL, 0, 0, NULL)) == 0) {
      LONG error = GetLastError();
      setError(error);
      CloseHandle(pContext->hEvent1);
      pContext->hEvent1 = 0;
      return error;
   }

   if ((pContext->hThread =
         CreateThread(0, 0, (LPTHREAD_START_ROUTINE)CaptureThreadFunc, pContext,
                      CREATE_SUSPENDED, &threadId)) == 0) {
      LONG error = GetLastError();
      setError(error);
      CloseHandle(pContext->hEvent1);
      pContext->hEvent1 = 0;
      CloseHandle(pContext->hResumeEvent);
      pContext->hResumeEvent = 0;
      return error;
   }
   SetThreadPriority(pContext->hThread, 2);
   ResumeThread(pContext->hThread);
   setError(0);
   return 0;
}

WLANCAP_API LONG WincapStop(Context *pContext) {
   if (!pContext) {
      setError(0x57);
      return 0x57;
   }
   pContext->state = STOPPING;
   if (pContext->hResumeEvent) {
      SetEvent(pContext->hResumeEvent);
   }
   if (pContext->hEvent1) {
      SetEvent(pContext->hEvent1);
   }
   if (pContext->hThread) {
      WaitForSingleObject(pContext->hThread, 10000);
      CloseHandle(pContext->hThread);
      pContext->hThread = 0;
   }
   if (pContext->hResumeEvent) {
      CloseHandle(pContext->hResumeEvent);
      pContext->hResumeEvent = 0;
   }
   if (pContext->hEvent1) {
      CloseHandle(pContext->hEvent1);
      pContext->hEvent1 = 0;
   }
   if (pContext->pRingBuffer) {
      WincapSetPacketBuffer(pContext, NULL);
      if (pContext->hThread) {
         CloseHandle(pContext->hThread);
         pContext->pRingBuffer->hEvent = 0;
         pContext->pRingBuffer->hVxD = 0;
      }
      HeapFree(GetProcessHeap(), 0, pContext->pRingBuffer);
      pContext->pRingBuffer = 0;
   }
   pContext->state = 0;
   setError(0);
   return 0;
}

WLANCAP_API void WincapSynchronizeTimeStamps() {
   __int64 freq;
   __int64 perfCount;
   if (WincapIsWindowsNT()) {
      __int64 fTime = WincapGetTimeStamp();
      QueryPerformanceCounter((LARGE_INTEGER*)&perfCount);
      QueryPerformanceFrequency((LARGE_INTEGER*)&freq);
      EnterCriticalSection(&criticalSection);
         performanceCount = perfCount;
         performanceFrequency = freq;
         timeStamp = fTime;
      LeaveCriticalSection(&criticalSection);
   }
   else if (WincapIsWindows9X()) {
      __int64 fTime = WincapGetTimeStamp();
      __int64 ticks = GetTickCount() * (__int64)1000;
      EnterCriticalSection(&criticalSection);
         timeStamp = fTime - ticks;
      LeaveCriticalSection(&criticalSection);
   }
}

WLANCAP_API BOOL WincapUnloadDriver() {
   char fName[260];
   if (WincapIsWindowsNT()) {
      return stopService();
   }
   if (!WincapIsWindows9X()) {
      setError(0x78);
      return FALSE;
   }
   if (checkFile("PEEK3") == 0) {
      return TRUE;
   }
   sprintf(fName, "%s%s", "\\\\.\\", "PEEK3");
   DeleteFile(fName);
   return checkFile("PEEK3") == 0;    //?????  //***
}

typedef struct PrivatePeekBuffer_t {
   int n;
   int size;
   unsigned char buf[];
} PrivatePeekBuffer;

LONG doPrivatePeek1(HANDLE hFile, int n, int nBytes, unsigned char *pResult) {
   OVERLAPPED overlapped;
   WincapReqBuffer prb;
   PrivatePeekBuffer *ptr;
   LONG error;
   if (nBytes == 0 || pResult == NULL) {
      return 0x57;
   }
   memset(&overlapped, 0, sizeof(OVERLAPPED));
   if ((overlapped.hEvent = CreateEvent(NULL, 1, 0, NULL)) == 0) {
      return GetLastError();
   }
   if ((ptr = (PrivatePeekBuffer*) HeapAlloc(GetProcessHeap(), 0, nBytes + 8)) == NULL) {
      error = GetLastError();
      CloseHandle(overlapped.hEvent);
      return error;
   }
   ptr->size = nBytes;
   ptr->n = n;
   memset(ptr->buf, 0, n);
   memset(&prb, 0, sizeof(WincapReqBuffer));
   prb.field_14 = 1;
   prb.command = 0xF0020104;
   prb.pDest = ptr;
   prb.destLength = nBytes + 8;
   error = WincapPrivateRequest(hFile, &prb, &overlapped);
   HeapFree(GetProcessHeap(), 0, ptr);
   CloseHandle(overlapped.hEvent);
   return error;
}

LONG CaptureThreadFunc(Context *pContext) {
   int var_2A0;
   __int64 var_29C_8;
   int var_294;
   int ringBufferIndex;
   int var_28C;
   int var_288;
   LONG result;
   int var_280;
   OVERLAPPED overlapped;
   RingBuffer *pRingBufferLocal;
   WincapReqBuffer prb;
   int var_38 = 0;
   unsigned char reqResult_2[6];
   HANDLE events[2];
   Context *pContextLocal = pContext;
   unsigned int *packetPtr;
   int reqResult10104;

   if (pContext == NULL) {
      return 0x57;
   }
   if (IsBadReadPtr(pContext, 48)) {
      return 0x57;
   }
   if (pContext->state == STOPPING) {
      return 0;
   }
   pRingBufferLocal = pContext->pRingBuffer;
   if (pRingBufferLocal == NULL) {
      return 0x57;
   }
   memset(&overlapped, 0, sizeof(OVERLAPPED));
   if ((overlapped.hEvent = CreateEvent(0, 0, 0, 0)) == 0) {
      return GetLastError();
   }
   memset(&prb, 0, sizeof(prb));
   prb.command = 0x10104;
   prb.pDest = &reqResult10104;
   prb.destLength = 4;
   if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
      CloseHandle(overlapped.hEvent);
      return result;
   }
   memset(&prb, 0, sizeof(prb));
   prb.pDest = reqResult_2;
   prb.destLength = 6;
   switch (reqResult10104) {
      case 0:
         prb.command = 0x1010102;
         break;
      case 1:
         prb.command = 0x2010102;
         break;
      case 3:
         prb.command = 0x4010102;
         break;
      default:
         CloseHandle(overlapped.hEvent);
         return 0x57;
   }
   if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
      CloseHandle(overlapped.hEvent);
      return result;
   }
   if ((pContext->channelMask & 0x20) == 0) {
      int n = 8;
      switch (reqResult10104) {
         case 0: case 3:
            n = 6;
         case 1:
            doPrivatePeek1(pContext->hFileAdapter, n, 6, reqResult_2);
            break;
      }
   }
   var_280 = 0;
   if (pContext->wincapRequestResult1 == 0x2000000 && pContext->channelMask != 0) {
      memset(&prb, 0, sizeof(WincapReqBuffer));
      prb.field_14 = 1;
      prb.command = 0xFF636700;
      result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped);
      prb.pDest = &pContext->channelMask;
      prb.destLength = 4;
      var_280 = result ? 0 : 1;
   }
   if (pContext->channelMask & 0x20) {
      memset(&prb, 0, sizeof(WincapReqBuffer));
      prb.field_14 = 1;
      prb.command = 0xFF636710;
      if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
         CloseHandle(overlapped.hEvent);
         return result;
      }
   }
   else {
      memset(&prb, 0, sizeof(WincapReqBuffer));
      var_288 = 0x21;
      prb.field_14 = 1;
      prb.command = 0x1010E;
      prb.pDest = &var_288;
      prb.destLength = 4;
      if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
         memset(&prb, 0, sizeof(WincapReqBuffer));
         var_288 = 0x80;
         prb.field_14 = 1;
         prb.command = 0x1010E;
         prb.pDest = &var_288;
         prb.destLength = 4;
         if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
            memset(&prb, 0, sizeof(WincapReqBuffer));
            var_288 = 1;
            prb.field_14 = 1;
            prb.command = 0x1010E;
            prb.pDest = &var_288;
            prb.destLength = 4;
            if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
               CloseHandle(overlapped.hEvent);
               return result;
            }
         }
      }
   }

   memset(&prb, 0, sizeof(WincapReqBuffer));
   prb.command = 0x10105;
   prb.pDest = &var_28C;
   prb.destLength = 4;
   if ((result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) == 0) {
      memset(&prb, 0, sizeof(WincapReqBuffer));
      prb.field_14 = 1;
      prb.command = 0x1010F;
      prb.pDest = &var_28C;
      prb.destLength = 4;
      result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped);
   }
   events[0] = pContext->pRingBuffer->hEvent;
   events[1] = pContext->hEvent1;
   pContext->state = RUNNING;

   do {
      result = WaitForMultipleObjects(2, events, 0, 1000);
      if (result != 0 && result != WAIT_TIMEOUT) {
         if (result != 1) continue;
         if (pContext->channelMask & 0x20) {
            memset(&prb, 0, sizeof(WincapReqBuffer));
            prb.field_14 = 1;
            prb.command = 0xFF636711;
            if (result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped)) {
               CloseHandle(overlapped.hEvent);
               return result;
            }
         }
         else {
            memset(&prb, 0, sizeof(WincapReqBuffer));
            var_2A0 = 0;
            prb.field_14 = 1;
            prb.command = 0x1010E;
            prb.pDest = &var_2A0;
            prb.destLength = 4;
            result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped);
         }
         if (var_280) {
            memset(&prb, 0, sizeof(WincapReqBuffer));
            prb.field_14 = 1;
            prb.command = 0xFF636701;
            result = WincapRequest(pContext->hFileAdapter, &prb, &overlapped);
         }
         CloseHandle(overlapped.hEvent);
         return var_38;
      }
checkPaused:
      if (pContextLocal->state == PAUSED) {
         WaitForSingleObject(pContextLocal->hResumeEvent, INFINITE);
         pContextLocal->state = RUNNING;
      }
      var_294 = pContext->pRingBuffer->b;
      ringBufferIndex = pContext->pRingBuffer->currentIndex;
      if ((var_294 != ringBufferIndex) || (pContext->pRingBuffer->a & 1)) {
         if ((pContext->pRingBuffer->size - ringBufferIndex) < 0x20) {
            ringBufferIndex = 0;
         }
         packetPtr = (unsigned int*) (pContext->pRingBuffer->data + ringBufferIndex); // = pRingBuffer->data + eax = pRingBuffer->data + ringBufferIndex
         if (packetPtr[0] == -1) {
            ringBufferIndex = 0;
            packetPtr = (unsigned int *) pContext->pRingBuffer->data;   //reset to start of ring
         }
         if (pContext->cBack != NULL) {
            var_29C_8 = *((__int64*)(packetPtr + 4));
            if (osversion.dwPlatformId == VER_PLATFORM_WIN32_NT) {
               EnterCriticalSection(&criticalSection);
               if (var_29C_8 < performanceCount) {
                  var_29C_8 = performanceCount - var_29C_8;
                  var_29C_8 *= 1000000;
                  var_29C_8 = var_29C_8 / performanceFrequency;
                  var_29C_8 = timeStamp - var_29C_8;
               }
               else {
                  var_29C_8 -= performanceCount;
                  var_29C_8 *= 1000000;
                  var_29C_8 = var_29C_8 / performanceFrequency;
                  var_29C_8 += timeStamp;
               }
            }
            else {
               EnterCriticalSection(&criticalSection);
               var_29C_8 *= 1000;
               var_29C_8 += timeStamp;
            }
            LeaveCriticalSection(&criticalSection);
            if (!pContextLocal->cBack((WlanPacket*)(packetPtr + 8), packetPtr[0], packetPtr[1], var_29C_8, packetPtr[2], pContextLocal->userData)) {
               pContextLocal->state = PAUSED;
               ResetEvent(events[0]);
               continue;
            }
         }
         InterlockedExchange((LPLONG)&pContext->pRingBuffer->currentIndex, ((int*)packetPtr)[0] + ringBufferIndex + 32);
         InterlockedExchange((LPLONG)&pContext->pRingBuffer->a, pContext->pRingBuffer->a & 0xFFFFFFFE);
         goto checkPaused;
      }
      else {
         ResetEvent(events[0]);
         continue;
      }

//      ResetEvent(events[0]);
   } while (1);
   return var_38;
}



