/**********************************************************************
 ** Parse class: parses a string and places it in buckets according to
 **              the format for that verb
 **    
 ** Reviewed last: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef PARSE_C
#define PARSE_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "verb_formats.h"
#include "verb.h"
#include "verb_list.h"
#include "parse.h"
#include "objtype.h"
#include "memchk.h"

/***********************************************************************
 ** _Parse (constructor) - parses the input string and places the parts
 **                        in appropriate buckets. Marks status if what
 **                        we found fits the format  
 **
 ** Parameters: the_input - the user input to be parsed
 **             the_verb - pointer to the verb matching the input
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Parse::Parse(char *the_input, Entity *the_verb)
{
   int the_format;   /* Holds the verb format */
   Strings holder; /* holds the input string */
   int     i = 2;

   /* initialize variables */
   the_command = the_verb;
   number = 0;

   holder = the_input;

   if (the_command->get_type() == OBJ_TYPE_VERB)
   {
      the_format = ((Verb *) the_verb)->get_verb_format();

      /* if we find these formats, get preposition first */
      if (the_format == VERB_FORMAT_PREP)
      {
         preposition.assign_word(holder.str_show(), i);
         i++;
      }

      /* if we find these formats, get the target1 first */
      if ((the_format == VERB_FORMAT_TELL) ||
          (the_format == VERB_FORMAT_FROMTO) ||
          (the_format == VERB_FORMAT_GIVE) ||
          (the_format == VERB_FORMAT_SYS) ||
          (the_format == VERB_FORMAT_SET) ||
          (the_format == VERB_FORMAT_PREP) ||
          (the_format == VERB_FORMAT_GET))
      {
         target1.assign_word(holder.str_show(), i);
         i++;

         if ((the_format == VERB_FORMAT_GET) &&
             (target1.str_show() != NULL) &&
             (isdigit(*(target1.str_show()))))
         {
            number = atoi(target1.str_show());
            target1.assign_word(holder.str_show(), i);
            i++;
         }
      }

      /* if we find these formats, get speech now */
      if ((the_format == VERB_FORMAT_TELL) ||
          (the_format == VERB_FORMAT_COMM) ||
          (the_format == VERB_FORMAT_SET))
      {
         speech.assign_phrase(holder.str_show(), i-1);
         speech.remove_newline();
      }

      /* if we find these formats, get preposition now */
      if ((the_format == VERB_FORMAT_FROMTO) ||
          (the_format == VERB_FORMAT_GIVE) ||
          (the_format == VERB_FORMAT_GET))
      {
         preposition.assign_word(holder.str_show(), i);
         i++;
      }

      /* if we find these formats, get target2 now */
      if ((the_format == VERB_FORMAT_FROMTO) ||
          (the_format == VERB_FORMAT_GIVE) ||
          (the_format == VERB_FORMAT_GET))
      {
         target2.assign_word(holder.str_show(), i);
         i++;
      }
   }
   else if (the_command->get_type() == OBJ_TYPE_ACTION)
   {
      target1.assign_word(holder.str_show(), 2);

      speech.assign_phrase(holder.str_show(), 2);
      speech.remove_newline();
   }
   else if (the_command->get_type() == OBJ_TYPE_CHATLINE)
   {
      speech.assign_phrase(holder.str_show(), 1);
      speech.remove_newline();
   }

}


/***********************************************************************
 ** _Parse (destructor) - cleans up before destruction
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Parse::~Parse()
{

}



/***********************************************************************
 ** get_verb - gets the verb string
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the verb string
 **
 ***********************************************************************/
   
char *Parse::get_verb()
{
   return the_command->get_name();
}


/***********************************************************************
 ** get_target1 - returns a pointer to the target bucket string
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the target bucket
 **
 ***********************************************************************/
   
char *Parse::get_target1()
{
   // Lets make sure if it is empty it returns NULL
   if (target1.str_len() == 0)
	   target1.truncate(0);

   return target1.str_show();
}


/***********************************************************************
 ** get_target2 - returns a pointer to the target2 bucket string
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the target bucket
 **
 ***********************************************************************/

char *Parse::get_target2()
{
   // Lets make sure if it is empty it returns NULL
   if (target2.str_len() == 0)
	   target2.truncate(0);

   return target2.str_show();
}


/***********************************************************************
 ** get_preposition - returns a pointer to the preposition string bucket
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the preposition
 **
 ***********************************************************************/

char *Parse::get_preposition()
{
   // Lets make sure if it is empty it returns NULL
   if (preposition.str_len() == 0)
	   preposition.truncate(0);

   return preposition.str_show();
}


/***********************************************************************
 ** get_speech - gets a pointer to all text in the speech bucket
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the speech bucket
 **
 ***********************************************************************/

char *Parse::get_speech()
{
   // Lets make sure if it is empty it returns NULL
   if (speech.str_len() == 0)
	   speech.truncate(0);

   return speech.str_show();
}


/***********************************************************************
 ** get_number - gets the number gotten from the parse
 **
 **
 ***********************************************************************/

int Parse::get_number() 
{ 
   return number;
}

#endif








