// ---------------------------------------------------------------------------
// - UriQuery.cpp                                                            -
// - afnix:www module - uri query class implementation                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Uri.hpp"
#include "Vector.hpp"
#include "Strvec.hpp"
#include "UriQuery.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default query

  UriQuery::UriQuery (void) {
    reset ();
  }

  // create a uri query by name

  UriQuery::UriQuery (const String& query) {
    reset ();
    parse (query);
  }

  // return the class name

  String UriQuery::repr (void) const {
    return "UriQuery";
  }

  // reset an uri query information

  void UriQuery::reset (void) {
    wrlock ();
    d_query = "";
    Plist::reset ();
    unlock ();
  }

  // parse an uri query and update the uri query data structure

  void UriQuery::parse (const String& query) {
    // check for nothing first
    long len = query.length ();
    if (len == 0) {
      reset ();
      return;
    }
    // ok let's deal with it
    wrlock ();
    reset  ();
    try {
      // get the name/pair value
      Strvec npvec = Strvec::split (query, "&");
      // now split each pair
      long nplen = npvec.length ();
      for (long i = 0; i < nplen; i++) {
	String pair = npvec.get (i);
	Strvec data = Strvec::split (pair, "=");
	long len = data.length ();
	if (len == 0) continue;
	if (len == 1) {
	  String name = Uri::pdecode (data.get (0));
	  String pval = "";
	  add (name, pval);
	  continue;
	}
	if (len == 2) {
	  String name = Uri::pdecode (data.get (0));
	  String pval = Uri::pdecode (data.get (1));
	  add (name, pval);
	  continue;
	}
	throw Exception ("query-error", "invalid query string", query);
      }
      // save the qeury string
      d_query = query;
      unlock ();
    } catch (...) {
      reset  ();
      unlock ();
      throw;
    }
  }

  // return the query string

  String UriQuery::getquery (void) const {
    rdlock ();
    String result = d_query;
    unlock ();
    return result;
  }
    
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_PARSE    = zone.intern ("parse");
  static const long QUARK_GETQUERY = zone.intern ("get-query");

  // create a new object in a generic way

  Object* UriQuery::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new UriQuery;
    // check for 1 argument
    if (argc == 1) {
      String query = argv->getstring (0);
      return new UriQuery (query);
    }
    // invalid arguments
    throw Exception ("argument-error", "invalid arguments with uri query");
  }

  // return true if the given quark is defined

  bool UriQuery::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Plist::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* UriQuery::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETQUERY) return new String (getquery ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_PARSE) {
	String query = argv->getstring (0);
	parse (query);
	return nilp;
      }
    }
    // call the plist method
    return Plist::apply (robj, nset, quark, argv);
  }
}
