// ---------------------------------------------------------------------------
// - Main.cpp                                                                -
// - the afnix interpreter main program  (static version)                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Interp.hpp"
#include "System.hpp"
#include "Options.hpp"
#include "Libnet.hpp"
#include "Libnwg.hpp"
#include "Libpim.hpp"
#include "Libsio.hpp"
#include "Libsps.hpp"
#include "Libsys.hpp"
#include "Libtxt.hpp"
#include "Libxml.hpp"
#include "Library.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // get the interpreter options
  static Options* get_options (Output& os,const long argc,const char** argv) {
    // create a new option instance
    Options* opts = Interp::getopts ();
    
    // parse the options
    try {
      opts->parse (argc, argv);
      // check for verbose and version
      if (opts->getoflg ('h') == true) {
	opts->usage (os);
	delete opts;
	System::exit (0);
      }
      if (opts->getoflg ('v') == true) {
	os << "afnix cross interpreter, " << System::osname ();
	os << ", revision " << System::version () << eolc;
	delete opts;
	System::exit (0);
      }
    } catch (...) {
      opts->usage (os);
      delete opts;
      System::exit (1);
    }
    return opts;
  }

  // this procedure process the options
  static bool run_options (Options* opts) {
    // get the options arguments
    Strvec args = opts->getargs ();
    // get the optional file name
    bool   tflg = args.empty ();
    String name = tflg ? "" : args.pop ();
    
    // create a new interpreter
    Interp* interp = new Interp (tflg);

    // register the standard modules
    interp->reglib ("afnix-net", (void*) init_afnix_net);
    interp->reglib ("afnix-nwg", (void*) init_afnix_nwg);
    interp->reglib ("afnix-pim", (void*) init_afnix_pim);
    interp->reglib ("afnix-sio", (void*) init_afnix_sio);
    interp->reglib ("afnix-sps", (void*) init_afnix_sps);
    interp->reglib ("afnix-sys", (void*) init_afnix_sys);
    interp->reglib ("afnix-txt", (void*) init_afnix_txt);
    interp->reglib ("afnix-xml", (void*) init_afnix_xml);

    // eventually add the initial path
    if ((tflg == false) && (opts->getoflg ('f', "nopath") == false)) {
      interp->addpath (System::xdir (name));
    }
    // set some interpreter flags
    interp->setargs (args);
    interp->setasrt (opts->getoflg ('f', "assert"));
    interp->setpath (opts->getoptv ('i'));
    if (opts->getoflg ('e') == true) {
      interp->setemod (opts->getopts ('e'));
    }
    
    // loop or execute on the standard input or a file
    bool status  = (tflg == true) ? interp->loop () : interp->loop (name);
    // clean the interpreter and return
    delete interp;
    return status;
  }
}

// ---------------------------------------------------------------------------
// - main process section                                                    -
// ---------------------------------------------------------------------------

int main (int argc, const char** argv) {
  using namespace afnix;
  OutputTerm terr (OutputTerm::ERROR);
  
  // main processing loop
  try {
    // get the options and run them
    Options* opts = get_options (terr, argc, argv);
    // process the options
    bool status = run_options (opts);
    // clean everything and return
    delete opts;
    if (status == false) return 3;
  } catch (const Exception& e) {
    terr.errorln (e);
    return 1;
  } catch (...) {
    terr << "fatal: unknown exception trapped\n";
    return 2;
  }
  // so far, so good
  return 0;
}
