// ---------------------------------------------------------------------------
// - HttpReply.hpp                                                           -
// - afnix:wam module - http reply class definition                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_HTTPREPLY_HPP
#define  AFNIX_HTTPREPLY_HPP

#ifndef  AFNIX_PLIST_HPP
#include "Plist.hpp"
#endif

#ifndef  AFNIX_MIME_HPP
#include "Mime.hpp"
#endif

#ifndef  AFNIX_COOKIE_HPP
#include "Cookie.hpp"
#endif

namespace afnix {

  /// The HttpReply class is a simple designed to format a http reply.
  /// The class holds a property list which is used to format the header and 
  /// a buffer which is the reply content. When the write method is called,
  /// the header and the buffer is formatted to the output stream argument.
  /// The class creates initially the 'content-type' property which is 
  /// initialized to the 'text/plain' mime value. The object can also be
  /// filled with a mime document. This king of reply is usefull when
  /// responding to cgi request. In that case, the content-type is changed
  /// according to the mime document type.
  /// @author amaury darsch

  class HttpReply : public Object {
  protected:
    /// the http plist
    Plist  d_head;
    /// the http buffer
    Buffer d_hbuf;

  public:
    /// create an empty http reply
    HttpReply (void);

    /// create an empty reply with a content type
    /// @param type the content type to use
    HttpReply (const String& type);

    /// @return the class name
    String repr (void) const;

    /// set a cookie in the header
    /// @param cook the cookie to set
    virtual void sethead (const Cookie& cook);

    /// set a header property
    /// @param name the property name
    /// @param lval the property value
    virtual void sethead (const String& name, const Literal& pval);

    /// add a string to the http buffer
    /// @param lval the literal value to add
    virtual void addhbuf (const Literal& lval);

    /// add a buffer content to the http buffer
    /// @param buf the buffer to add
    virtual void addhbuf (const Buffer& buf);

    /// add a mime document to the http buffer
    /// @param mime the document to add
    virtual void addhbuf (const Mime& mime);

    /// write the http reply to an output stream
    /// @param os the output stream
    virtual void write (Output& os) const;

    /// write the http reply to a buffer
    /// @param buf the target buffer
    virtual void write (Buffer& buf) const;

    /// set the hhtp reply status code
    /// @param code the http status code
    virtual void setstatus (const long code);

    /// reply a redirect status with a url
    /// @param url the target url
    virtual void redirect (const String& url);

  private:
    // make the copy constructor private
    HttpReply (const HttpReply&);
    // make the assignment operator private
    HttpReply& operator = (const HttpReply&);

  public:
    /// create a new object in a generic object
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
