// ---------------------------------------------------------------------------
// - XmlSystem.cpp                                                           -
// - afnix:xml module - xml system class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "XmlSystem.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the default xml version id
  static const String XML_DEFAULT_XVID = "1.0";
  // the default xml ncoding mode
  static const String XML_DEFAULT_EMOD = "UTF-8";

  // -------------------------------------------------------------------------
  // - public section                                                        -
  // -------------------------------------------------------------------------

  // return the default xml system version string

  String XmlSystem::getxvid (void) {
    return XML_DEFAULT_XVID;
  }

  // return the default xml system version id 

  XmlSystem::t_xmlv XmlSystem::getxmlv (void) {
    return XmlSystem::toxmlv (XmlSystem::getxvid ());
  }

  // get the default xml system encoding mode

  String XmlSystem::getemod (void) {
    return XML_DEFAULT_EMOD;
  }

  // return true if the vid is valid

  bool XmlSystem::isxvid (const String& xvid) {
    if (xvid == "1.0") return true;
    if (xvid == "1.1") return true;
    return false;
  }

  // convert a string into a version

  XmlSystem::t_xmlv XmlSystem::toxmlv (const String& xvid) {
    if (xvid == "1.0") return XmlSystem::XML_1_0;
    if (xvid == "1.1") return XmlSystem::XML_1_1;
    throw Exception ("xml-error", "unsupported xml version", xvid);
  }

  // convert a version to a string

  String XmlSystem::toxvid (const t_xmlv xmlv) {
    String result;
    switch (xmlv) {
    case XML_1_0:
      result = "1.0";
      break;
    case XML_1_1:
      result = "1.1";
      break;
    }
    return result;
  }

  // convert a property into an attribute string

  String XmlSystem::toattr (const Property& prop) {
    // get the attribute name
    String result = prop.getname ();
    if (result.isnil ()) return result;
    // get the attribute value
    String pval = prop.getpval ();
    result += "=";
    result += pval.toliteral ();
    return result;
  }

  // convert a property list into an attribute string

  String XmlSystem::toattr (const Plist& plst) {
    // get the plist length
    long plen = plst.length ();
    // format result
    String result;
    for (long i = 0; i < plen; i++) {
      Property* prop = plst.get (i);
      if (prop == nilp) continue;
      String attr = toattr (*prop);
      if (attr.isnil () == true) continue;
      result += attr;
      if (i < plen - 1) result += ' ';
    }
    return result;
  }
}
