// ---------------------------------------------------------------------------
// - XmlDocument.cpp                                                         -
// - afnix:xml module - xml document class implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Runnable.hpp"
#include "InputFile.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "XmlReader.hpp"
#include "XmlDocument.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // this procedure reads an input stream and return a root node
  // the reference counter for the node is already increased
  static XmlRoot* get_root_node (Input* is) {
    // check the stream
    if (is == nilp) return nilp;
    // create a new reader and parse the input stream
    XmlReader* xmlr = new XmlReader;
    try {
      // parse the input stream
      xmlr->parse (is);
      // get the root node and protect it
      XmlRoot* result = xmlr-> getroot ();
      Object::iref (result);
      // clean the reader and result
      delete xmlr;
      // this is it
      return result;
    } catch (...) {
      delete xmlr;
      throw;
    }
  }

  // this procedure reads an input file and returns a root node
  // the reference counter for the node is already increased
  static XmlRoot* get_root_node (const String& name) {
    // open the file
    InputFile is (name);
    // get the root node
    return get_root_node (&is);
  }

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default document

  XmlDocument::XmlDocument (void) {
    p_root = nilp;
  }

  // create a document by name

  XmlDocument::XmlDocument (const String& name) {
    // create a reader
    d_name = name;
    p_root = get_root_node (name);
  }

  // create a document by name and input stream

  XmlDocument::XmlDocument (const String& name, Input* is) {
    d_name = name;
    p_root = get_root_node (is);
  }

  // create a document by name and root node

  XmlDocument::XmlDocument (const String& name, XmlRoot* root) {
    d_name = name;
    Object::iref (p_root = root);
  }

  // destroy this document

  XmlDocument::~XmlDocument (void) {
    Object::dref (p_root);
  }

  // return the document class name

  String XmlDocument::repr (void) const {
    return "XmlDocument";
  }

  // make this document shared

  void XmlDocument::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_root != nilp) p_root->mksho ();
  }

  // return the document name

  String XmlDocument::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // set the document name

  void XmlDocument::setname (const String& name) {
    wrlock ();
    d_name = name;
    unlock ();
  }

  // get the root node

  XmlRoot* XmlDocument::getroot (void) const {
    rdlock ();
    XmlRoot* result = p_root;
    unlock ();
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETNAME = zone.intern ("set-name");
  static const long QUARK_GETROOT = zone.intern ("get-root");

  // create a new object in a generic way

  Object* XmlDocument::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default document object
    if (argc == 0) return new XmlDocument;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new XmlDocument (name);
    }
    // check for 2 arguments
    if (argc == 2) {
      // get the document name
      String name = argv->getstring (0);
      // get the object and check
      Object* obj = argv->get (1);
      // check for an input stream
      Input* is = dynamic_cast <Input*> (obj);
      if (is != nilp) return new XmlDocument (name, is);
      // check for a root node
      XmlRoot* root = dynamic_cast <XmlRoot*> (obj);
      if (root != nilp) return new XmlDocument (name, root);
      throw Exception ("type-error", 
		       "invalid object with document constructor",
		       Object::repr (obj));	
    }
    throw Exception ("argument-error",
                     "too many argument with xml document constructor");
  }

  // return true if the given quark is defined

  bool XmlDocument::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Nameable::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* XmlDocument::apply (Runnable* robj, Nameset* nset, const long quark,
			      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETROOT) {
	rdlock ();
	Object* result = getroot ();
	robj->post (result);
	unlock ();
	return result;
      }
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
    }
    // call the nameable method
    return Nameable::apply (robj, nset, quark, argv);
  }
}
