// ---------------------------------------------------------------------------
// - XmlDecl.cpp                                                             -
// - afnix:xml module - xml declaration node class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "XmlDecl.hpp"
#include "XmlSystem.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the pi node name
  static const String XML_DECL_NAME = "xml";
  // the version attribute
  static const String XML_VERS_ATTR = "version";
  // the encoding attribute
  static const String XML_EMOD_ATTR = "encoding";
  // the standalone attribute
  static const String XML_STND_ATTR = "standalone";

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default xml declaration

  XmlDecl::XmlDecl (void) : XmlPi (XML_DECL_NAME) {
    setattr (XML_VERS_ATTR, XmlSystem::getxvid ());
    setattr (XML_EMOD_ATTR, XmlSystem::getemod ());
  }

  // create a xml declaration node with a version

  XmlDecl::XmlDecl (const String& xvid) : XmlPi (XML_DECL_NAME) {
    if (XmlSystem::isxvid (xvid) == false) {
      throw Exception ("xml-errpr", "invalid xml version", xvid);
    }
    setattr (XML_VERS_ATTR, xvid);
    setattr (XML_EMOD_ATTR, XmlSystem::getemod ());
  }

  // create a xml declaration node with a version

  XmlDecl::XmlDecl (const String& xvid, const String& emod) : 
    XmlPi (XML_DECL_NAME) {
    if (XmlSystem::isxvid (xvid) == false) {
      throw Exception ("xml-errpr", "invalid xml version", xvid);
    }
    setattr (XML_VERS_ATTR, xvid);
    setattr (XML_EMOD_ATTR, emod);
  }

  // create a xml declaration node with a version

  XmlDecl::XmlDecl (const String& xvid, const String& emod,
		    const String& stnd) : XmlPi (XML_DECL_NAME) {
    if (XmlSystem::isxvid (xvid) == false) {
      throw Exception ("xml-errpr", "invalid xml version", xvid);
    }
    setattr (XML_VERS_ATTR, xvid);
    setattr (XML_EMOD_ATTR, emod);
    setattr (XML_STND_ATTR, stnd);
  }

  // return the document class name

  String XmlDecl::repr (void) const {
    return "XmlDecl";
  }

  // get the xml version

  String XmlDecl::getxmlv (void) const {
    rdlock ();
    try {
      String result = getpval (XML_VERS_ATTR);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the xml encoding mode

  String XmlDecl::getemod (void) const {
    rdlock ();
    try {
      String result = getpval (XML_EMOD_ATTR);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a declaration node to a buffer

  void XmlDecl::write (Buffer& buf) const {
    rdlock ();
    try {
      // add the node name
      buf.add ("<?");
      buf.add (d_name);
      // get the attribute list
      String attr = XmlSystem::toattr (d_alst);
      if (attr.isnil () == false) {
	buf.add (' ');
	buf.add (attr);
      }
      // add the terminaison
      buf.add ("?>");
      buf.add (eolq);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a node to an output stream

  void XmlDecl::write (Output& os) const {
    rdlock ();
    try {
      // add the node name
      os.write ("<?");
      os.write (d_name);
      // get the attribute list
      String attr = XmlSystem::toattr (d_alst);
      if (attr.isnil () == false) {
	os.write (' ');
	os.write (attr);
      }
      // add the terminaison
      os.write   ("?>");
      os.newline ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // create a new object in a generic way

  Object* XmlDecl::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new XmlDecl;
    // check for 1 argument
    if (argc == 1) {
      String xvid = argv->getstring (0);
      return new XmlDecl (xvid);
    }
    // check for 2 arguments
    if (argc == 2) {
      String xvid = argv->getstring (0);
      String emod = argv->getstring (1);
      return new XmlDecl (xvid, emod);
    }
    // check for 3 arguments
    if (argc == 3) {
      String xvid = argv->getstring (0);
      String emod = argv->getstring (1);
      String stnd = argv->getstring (2);
      return new XmlDecl (xvid, emod, stnd);
    }
    throw Exception ("argument-error",
                     "too many argument with xml declaration constructor");
  }
}
