// ---------------------------------------------------------------------------
// - Strvec.hpp                                                              -
// - standard object library - string vector class definition                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_STRVEC_HPP
#define  AFNIX_STRVEC_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The Strvec class implements a string vector facility which is faster 
  /// than the standard vector implementation with objects. The string vector
  /// provides the same funcionality than the Vector class except that it 
  /// works with string only. The class provides also a facility tocheck if a 
  /// string exists in this vector. The class provides a also a facility to
  /// split a string into multiple substrings
  /// @author amaury darsch

  class Strvec : public Serial {
  public:
    /// split a string and return a vector of string
    /// @param name the string to split
    /// @param sbrk the break character sequence
    /// @return a vector of strings
    static Strvec split (const String& name, const String& sbrk);

  private:
    /// the current size of this vector
    long d_length;
    /// the size of the allocated vector
    long d_size;
    /// the allocated vector
    String* p_vector;

  public:
    /// create an empty string vector.
    Strvec (void);

    /// create a string vector with a predefined size
    /// @param size the requested size  
    Strvec (const long size);
  
    /// copy constructor for this string vector
    /// @param that the string vector to copy
    Strvec (const Strvec& that);

    /// destroy this string vector
    ~Strvec (void);

    /// @return the class name
    String repr (void) const;

    /// @return the string vector serial code
    t_byte serialid (void) const;

    /// serialize this string vector to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a string vector from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// assign a string vector to this one
    /// @param that the string vector to assign
    Strvec& operator = (const Strvec& that);

    /// add a string in this string vector
    /// @param str the string to add
    void add (const String& str);

    /// set a string at a given position
    /// @param index the vector index
    /// @param str   the string to set
    void set (const long index, const String& str);

    /// @return the first string from this vector
    String pop (void);

    /// get a string at a certain index
    /// @param index the vector index
    /// @return the string at this position
    String get (const long index) const;

    /// @return the first string in this vector
    String first (void) const;

    /// @return the last string in this vector
    String last (void) const;

    /// @return the number of elements in this string vector
    long length (void) const;

    /// @return true if the vector is empty
    bool empty (void) const;

    /// @return true if the string exists
    bool exists (const String& name) const;

    /// search for a key in the string vector or throw an exception
    /// @param key the key to search for
    /// @return the index of the first string which match the key
    long index (const String& key) const;
    
    /// search for a key in the string vector
    /// @param key the key to search for
    /// @return the index of the key or -1 if not found
    long lookup (const String& key) const;

    /// reset this string vector
    void reset (void);

    /// @return the maximum string length in this vector
    long maxlen (void) const;

    /// @return the minimum string length in this vector
    long minlen (void) const;

    /// @return an array of quarks mapping this string vector
    long* toquarks (void) const;
  };
}

#endif
