// ---------------------------------------------------------------------------
// - Plist.hpp                                                               -
// - standard object library - property list class definition                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 2005-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_PLIST_HPP
#define  AFNIX_PLIST_HPP

#ifndef  AFNIX_VECTOR_HPP
#include "Vector.hpp"
#endif

#ifndef  AFNIX_ITERABLE_HPP
#include "Iterable.hpp"
#endif

#ifndef  AFNIX_PROPERTY_HPP
#include "Property.hpp"
#endif

#ifndef  AFNIX_HASHTABLE_HPP
#include "HashTable.hpp"
#endif

namespace afnix {

  /// The Plist class is a base container class used to manage property 
  /// object in an ordered way. The property list operates by maintaining a
  /// vector of property object along with a hash table that permits
  /// to find the object quickly.
  /// @author amaury darsch

  class Plist : public Iterable, public Serial {
  private:
    /// the plist vector
    Vector    d_plist;
    /// the plist hash
    HashTable d_phash;
    
  public:
    /// create an empty plist
    Plist (void);

    /// destroy this plist
    ~Plist (void);

    /// @return the class name
    String repr (void) const;

    /// make this plist a shared object
    void mksho (void);

    /// @return the plist serial code
    t_byte serialid (void) const;

    /// serialize this plist to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a plist from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// reset this plist
    virtual void reset (void);

    /// @return the plist length
    virtual long length (void) const;

    /// @return a property by index
    virtual Property* get (const long index) const;

    /// @return a property by name
    virtual Property* find (const String& name) const;

    /// @return a property by name or throw an exception
    virtual Property* lookup (const String& name) const;

    /// @return a property value by name
    virtual String getpval (const String& name) const;

    /// @return true if the property exists
    virtual bool exists (const String& name) const;

    /// add a property to this plist
    /// @param prop the property to add
    virtual void add (Property* prop);

    /// add a property by name and value
    /// @param name the property name to add
    /// @param lval the property value to add
    virtual void add (const String& name, const Literal& lval);

    /// set a property by name and value
    /// @param name the property name to set
    /// @param lval the property value to set
    virtual void set (const String& name, const Literal& lval);

    /// @return a new iterator for this plist
    Iterator* makeit (void);

  private:
    // make the copy constructor private
    Plist (const Plist&);
    // make the assignment operator private
    Plist& operator = (const Plist&);
    // make the plist iterator a friend
    friend class Plistit;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a plist of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
  
  /// The Plistit class is the iterator for the property list class. Such 
  /// iterator is constructed with the "makeit" plist method. The iterator is
  /// reset to the beginning of the list.
  /// @author amaury darsch
  
  class Plistit : public Iterator {
  private:
    /// the plist to iterate
    Plist* p_plist;
    /// the current index
    long   d_idx;
    
  public:
    /// create a new iterator from a plist
    /// @param plist the plist to iterate
    Plistit (Plist* plist);

    /// destroy this plist iterator
    ~Plistit (void);

    /// @return the class name
    String repr (void) const;

    /// make this plist iterator a shared object
    void mksho (void);

    /// reset the iterator to the begining
    void begin (void);

    /// reset the iterator to the end
    void end (void);

    /// move the iterator to the next position
    void next (void);

    /// move the iterator to the previous position
    void prev (void);

    /// @return the object at the current position
    Object* getobj (void) const;

    /// @return true if the iterator is at the end
    bool isend (void);

  private:
    // make the copy constructor private
    Plistit (const Plistit&);
    // make the assignment operator private
    Plistit& operator = (const Plistit&);
  };
}

#endif
