// ---------------------------------------------------------------------------
// - Output.cpp                                                              -
// - standard object library - output stream class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Byte.hpp"
#include "Output.hpp"
#include "Vector.hpp"
#include "Unicode.hpp"
#include "Runnable.hpp"
#include "Character.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "ccnv.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // return the stream descriptor

  int Output::getsid (void) const {
    return -1;
  }

  // return true if we have a tty - by default is false

  bool Output::istty (void) const {
    return false;
  }

  // write an unicode character
  
  void Output::write (const t_quad value) {
    wrlock ();
    try {
      // check for byte mode
      if (d_emod == System::BYTE) {
	// encode from the transcoder
	write (decode (value));
	unlock ();
	return;
      }
      // check for utf-8 mode
      if (d_emod == System::UTF8) {
	// encode in character form
	char* sbuf = Unicode::encode (value);
	write (sbuf);
	delete [] sbuf;
	unlock ();
	return;
      }
      throw Exception ("write-error", "invalid unicode character to write");
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write an unicode string to the output stream

  void Output::write (const t_quad* value) {
    wrlock ();
    long size = Unicode::strlen (value);
    try {
      write  (value, size);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write an array of character to the output stream

  void Output::write (const char* value, const long size) {
    wrlock ();
    try {
      for (long i = 0; i < size; i++) write (value[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write an array of unicode character to the output stream

  void Output::write (const t_quad* value, const long size) {
    wrlock ();
    try {
      // check for byte mode
      if (d_emod == System::BYTE) {
	// transcode the quad buffer
	char* cbuf = decode (value, size);
	write (cbuf);
	delete [] cbuf;
	unlock ();
	return;
      }
      // check for utf-8 mode
      if (d_emod == System::UTF8) {
	// encode the unicode buffer in character form
	char* sbuf = Unicode::encode (value, size);
	write (sbuf);
	delete [] sbuf;
	unlock ();
	return;
      }
      throw Exception ("write-error", "invalid unicode buffer to write");
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a string to the output stream

  void Output::write (const String& value) {
    wrlock ();
    try {
      // check for byte mode
      if (d_emod == System::BYTE) {
	// get the quad buffer
	t_quad* sbuf = value.toquad ();
	if (sbuf == nilp) {
	  unlock ();
	  return;
	}
	// transcode to the character buffer
	char* cbuf = decode (sbuf);
	delete [] sbuf;
	// write the buffer and clean
	write (cbuf);
	delete [] cbuf;
	unlock ();
	return;
      }
      // check for utf-8 mode
      if (d_emod == System::UTF8) {
	char* sbuf = value.encode ();
	if (sbuf == nilp) {
	  unlock ();
	  return;
	}
	// write string content
	write (sbuf);
	delete [] sbuf;
	unlock ();
	return;
      }
      throw Exception ("write-error", "invalid unicode string to write");
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a string as an error string

  void Output::error (const String& value) {
    write (value);
  }

  // write a newline to this output stream

  void Output::newline (void) {
    write (eolq);
  }

  // write a string followed by a new line. This is a convenient function

  void Output::writeln (const String& line) {
    write (line + eolc);
  }

  // write a string followed by a new line in error mode

  void Output::errorln (const String& line) {
    error (line + eolq);
  }

  // write an exception string on this output stream

  void Output::errorln (const Exception& e) {
    wrlock ();
    try {
      String what = "exception : ";
      String file = "in file   : "; 
      String resm = "reason    : ";
      String resv = e.getval ();
      if (e.getnlf () == true) newline ();
      error (what + e.geteid ());
      newline ();
      long   lnum = e.getlnum ();
      String name = e.getname ();
      if ((lnum != 0) && (name.length () != 0)) {
	error (file + name + " at or around line " + lnum);
	newline ();
      }
      if (resv.length () > 0) {
	error (resm + resv);
	newline ();
      }
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write a character on the output stream - no exception is thrown

  Output& Output::operator << (const char value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // write a string on the output stream - no exception is thrown
  
  Output& Output::operator << (const String& value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // write an integer on the output stream - no exception is thrown

  Output& Output::operator << (const long value) {
    char* data = c_ltoa (value);
    try {
      this->write (data);
    } catch (const Exception& e) {
      delete [] data;
    };
    delete [] data;
    return *this;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_WRITE   = zone.intern ("write");
  static const long QUARK_WRITELN = zone.intern ("writeln");
  static const long QUARK_ERRORLN = zone.intern ("errorln");
  static const long QUARK_NEWLINE = zone.intern ("newline");

  // return true if the given quark is defined

  bool Output::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Stream::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Output::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch variable length arguments
    if ((quark == QUARK_WRITE) || (quark == QUARK_WRITELN)) {
      String result;
      for (long i = 0; i < argc; i++) {
	Object*   obj = argv->get (i);
	// check for a literal
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj != nilp) {
	  result += lobj->tostring ();
	  continue;
	}
	// check for a byte
	Byte* bobj = dynamic_cast <Byte*> (obj);
	if (bobj != nilp) {
	  result += (char) bobj->tobyte ();
	  continue;
	}
	throw Exception ("type-error", "invalid object to write", 
			 Object::repr (obj));
      }
      if (quark == QUARK_WRITELN)  result = result + eolc;
      write (result);      
      return nilp;
    }

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_NEWLINE) {
	newline ();
	return nilp;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_ERRORLN) {
	String val = argv->getstring (0);
	errorln (val);
	return nilp;
      }
    }
    
    // call the stream method
    return Stream::apply (robj, nset, quark, argv);
  }
}
