// ---------------------------------------------------------------------------
// - InputOutput.cpp                                                         -
// - standard object library - input-output stream class implementation      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Ascii.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Character.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "InputOutput.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a new input-output stream

  InputOutput::InputOutput (void) {
    d_buffer.setrflg (false);
    reset ();
  }

  // create a new input-output stream with a data string

  InputOutput::InputOutput (const String& data) {
    d_buffer.setrflg (false);
    reset ();
    d_buffer.add (data);
  }
  
  // create a new input-output stream with a buffer and a size

  InputOutput::InputOutput (const char* data, const long size) {
    d_buffer.setrflg (false);
    reset ();
    d_buffer.add (data, size);
  }

  // reset this input-output stream

  void InputOutput::reset (void) {
    wrlock ();
    d_buffer.reset ();
    unlock ();
  }

  // return the class name

  String InputOutput::repr (void) const {
    return "InputOutput";
  }

  // read a character from the stream

  char InputOutput::read (void) {
    rdlock ();
    try {
      char result = d_buffer.empty () ? eofc : d_buffer.read ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the eof flag

  bool InputOutput::iseof (void) const {
    rdlock ();
    bool result = d_buffer.empty ();
    unlock ();
    return result;
  }

  // return true if we can read a valid character

  bool InputOutput::valid (const long tout) const {
    rdlock ();
    bool result = (d_buffer.empty () == false);
    unlock ();
    return result;
  }

  // write a character in the buffer

  void InputOutput::write (const char value) {
    wrlock ();
    d_buffer.add (value);
    unlock ();
  }

  // write a character string in the buffer

  void InputOutput::write (const char* value) {
    wrlock ();
    d_buffer.add (value, Ascii::strlen (value));
    unlock ();
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 1;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_RESET = zone.intern ("reset");

  // create a new object in a generic way

  Object* InputOutput::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new InputOutput;
    // check for 1 argument
    if (argc == 1) {
      String sval = argv->getstring (0);
      return new InputOutput (sval);
    }
    throw Exception ("argument-error", 
		     "invalid arguments with with input-output stream");
  }

  // return true if the given quark is defined

  bool InputOutput::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Input::isquark (quark, hflg) : false;
    if (result == false) {
      result = hflg ? Output::isquark (quark, hflg) : false;
    }
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* InputOutput::apply (Runnable* robj, Nameset* nset, const long quark,
			      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_RESET) {
	reset ();
	return nilp;
      }
    }
    // check with the input stream object
    if (Input::isquark (quark, true) == true) {
      return Input::apply (robj, nset, quark, argv);
    }

    // fallback with the output stream
    return Output::apply (robj, nset, quark, argv);
  }
}
