// ---------------------------------------------------------------------------
// - Cipher.cpp                                                              -
// - afnix cryptography - base cipher class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cipher.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "QuarkZone.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a cipher by name

  Cipher::Cipher (const String& name) {
    d_name = name;
    d_cbsz = 0;
    d_rflg = false;
    reset ();
  }

  // create a cipher by name and key

  Cipher::Cipher (const String& name, const Key& key) {
    d_name = name;
    d_ckey = key;
    d_cbsz = 0;
    d_rflg = false;
    reset ();
  }

  // reset this cipher
  
  void Cipher::reset (void) {
  }

  // return the class name

  String Cipher::repr (void) const {
    return "Cipher";
  }

  // return the cipher name
    
  String Cipher::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // set the cipher key

  void Cipher::setkey (const Key& key) {
    wrlock ();
    reset ();
    d_ckey = key;
    unlock ();
  }

  // get the cipher key

  Key Cipher::getkey (void) const {
    rdlock ();
    Key result = d_ckey;
    unlock ();
    return result;
  }

  // set the cipher reverse flag

  void Cipher::setrflg (const bool rflg) {
    wrlock ();
    reset  ();
    d_rflg = rflg;
    unlock ();
  }

  // return the cipher reverse flag

  bool Cipher::getrflg (void) const {
    rdlock ();
    bool result = d_rflg;
    unlock ();
    return result;
  }
   
  // return the cipher block size

  long Cipher::getcbsz (void) const {
    rdlock ();
    long result = d_cbsz;
    unlock ();
    return result;
  }

  // get the normal waist from a file size

  t_long Cipher::waist (const t_long size) const {
    rdlock ();
    t_long result = (size / d_cbsz) * d_cbsz;
    if ((size == 0) || ((size % d_cbsz) != 0)) result+= d_cbsz;
    unlock ();
    return result;
  }
  
  // process an input stream into an output stream

  t_long Cipher::stream (Output& os, Input& is) {
    // first create the input and output buffer
    if (d_cbsz == 0) return 0;
    wrlock ();
    t_long  cc = 0;
    t_byte* bi = new t_byte[d_cbsz];
    t_byte* bo = new t_byte[d_cbsz];
    // loop as long as the buffer is valid
    try {
      // initialize the data counter and input buffer
      long count = 0;
      for (long i = 0; i < d_cbsz; i++) bi[i] = nilc;
      while (is.valid (0) == true) {
	bi[count++] = is.read ();
	// check if we process the buffer
	if (count == d_cbsz) {
	  process (bo, bi);
	  os.write ((char*) bo, d_cbsz);
	  cc += count;
	  count = 0;
	  for (long i = 0; i < d_cbsz; i++) bi[i] = nilc;
	}
      }
      if (count != 0) {
	process (bo, bi);
	os.write ((char*) bo, d_cbsz);
	cc += count;
      }
      unlock ();
    } catch (...) {
      delete [] bi;
      delete [] bo;
      unlock ();
      throw;
    }
    return cc;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 8;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_RESET   = zone.intern ("reset");
  static const long QUARK_STREAM  = zone.intern ("stream");
  static const long QUARK_SETKEY  = zone.intern ("set-key");
  static const long QUARK_GETKEY  = zone.intern ("get-key");
  static const long QUARK_GETNAME = zone.intern ("get-name");
  static const long QUARK_GETSIZE = zone.intern ("get-size");
  static const long QUARK_SETRFLG = zone.intern ("set-reverse");
  static const long QUARK_GETRFLG = zone.intern ("get-reverse");

  // return true if the given quark is defined

  bool Cipher::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Cipher::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETKEY)  return new Key (getkey ());
      if (quark == QUARK_GETNAME) return new String  (getname ());
      if (quark == QUARK_GETSIZE) return new Integer (getcbsz ());
      if (quark == QUARK_GETRFLG) return new Boolean (getrflg ());
      if (quark == QUARK_RESET) {
	reset ();
	return nilp;
      }
    }
    // check for 1 argument
    if (argc == 1) {
      if (argc == QUARK_SETKEY) {
	Object* obj = argv->get (0);
	Key*    key = dynamic_cast <Key*> (obj);
	if (key != nilp) {
	  setkey (*key);
	  return nilp;
	}
      }
    }
    // check for 2 arguments
    if (argc == 2) {
      if (quark == QUARK_STREAM) {
	// check for an output stream
	Object* oobj = argv->get (0);
	Output* os = dynamic_cast <Output*> (oobj);
	if (os == nilp) {
	  throw Exception ("type-error", 
			   "invalid output object for cipher stream",
			   Object::repr (oobj));
	}
	// check for an input stream
	Object* iobj = argv->get (1);
	Input*  is = dynamic_cast <Input*> (iobj);
	if (is == nilp) {
	  throw Exception ("type-error", 
			   "invalid input object for cipher stream",
			   Object::repr (oobj));
	}
	t_long result = stream (*os, *is);
	return new Integer (result);
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
