/*
   Project: Adun

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#ifndef AD_KEYVALUE_CODING
#define AD_KEYVALUE_CODING
#include <Foundation/Foundation.h>


/**
\ingroup Protocols
This protocol defines methods that must be implemented by objects that
wish to observe the values of environment variables. All framework objects
adopt this protocol.
**/

@protocol AdEnvironmentObservation
/**
The environment sends this message to the observer when the key has changed
\param key The key that has changed
\param value The new value for the key
\param object The environment
*/
- (void) updateForKey: (NSString*) key value: (id) value object: (id) object;

/** This method calls the appropriate update method for key
**/
- (void) updateDependantsOfKey: (NSString*) key;

/**
Registers the object for observation of environment. Has no effect if
observesEnvironment is NO
*/
- (void) registerWithEnvironment;

/**
Deregisters the object for observation of environment.
*/
- (void) deregisterWithEnvironment;

/**
Turns on/off environment observation. Calling [object setObservesEnvironment: YES] causes registerWithEnvironment to be called.
synchroniseWithEnvrionment: must be called if you wish the environment variables to be
updated. Calling [object setObservesEnvironment: NO] causes the object to call deregisterWithEnvironment
aswell.
*/
- (void) setObservesEnvironment: (BOOL) value;

/**
Returns YES if the object will observe the values of certain parameters in an environment object.
The value returned is independant of whether the object actually has a environment set or not
*/
- (BOOL) observesEnvironment;

/**
Synchronises the object with the current environment values. Has no effect if observation
was turned off by setObservesEnvironment:. Raises an NSInternalInconsistency exception if
no environment is set.
*/

- (void) synchroniseWithEnvironment;

/**
Sets the environment the object is to observe. The object deregisters itself with
any previous environment and registers with the new one (if observesEnvironment indicates)
synchroniseWithEnvrionment: must be called afterwards if you want the object to use the values encapsulated by the
new environment.
*/

- (void) setEnvironment: (id) object;

@end


#endif

