# AST of C language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/symbol"
require "adlint/util"
require "adlint/c/seqp"

module AdLint #:nodoc:
module C #:nodoc:

  # == DESCRIPTION
  # === SyntaxNode class hierarchy
  #  SyntaxNode
  #    <--- Expression
  #           <--- ErrorExpression
  #           <--- PrimaryExpression
  #                  <--- ObjectSpecifier
  #                  <--- ConstantSpecifier
  #                  <--- StringLiteralSpecifier
  #                  <--- NullConstantSpecifier
  #                  <--- GroupedExpression
  #           <--- PostfixExpression
  #                  <--- ArraySubscriptExpression
  #                  <--- FunctionCallExpression
  #                  <--- MemberAccessByValueExpression
  #                  <--- MemberAccessByPointerExpression
  #                  <--- BitAccessByValueExpression
  #                  <--- BitAccessByPointerExpression
  #                  <--- PostfixIncrementExpression
  #                  <--- PostfixDecrementExpression
  #                  <--- CompoundLiteralExpression
  #           <--- UnaryExpression
  #                  <--- PrefixIncrementExpression
  #                  <--- PrefixDecrementExpression
  #                  <--- AddressExpression
  #                  <--- IndirectionExpression
  #                  <--- UnaryArithmeticExpression
  #                  <--- SizeofExpression
  #                  <--- SizeofTypeExpression
  #                  <--- AlignofExpression
  #                  <--- AlignofTypeExpression
  #           <--- CastExpression
  #           <--- BinaryExpression
  #                  <--- MultiplicativeExpression
  #                  <--- AdditiveExpression
  #                  <--- ShiftExpression
  #                  <--- RelationalExpression
  #                  <--- EqualityExpression
  #                  <--- AndExpression
  #                  <--- ExclusiveOrExpression
  #                  <--- InclusiveOrExpression
  #                  <--- LogicalAndExpression
  #                  <--- LogicalOrExpression
  #                  <--- ConditionalExpression
  #                  <--- SimpleAssignmentExpression
  #                  <--- CompoundAssignmentExpression
  #           <--- CommaSeparatedExpression
  #    <--- Declaration
  #    <--- FunctionDeclaration -------------------- SymbolicElement <<module>>
  #    <--- VariableDeclaration ----------------------+  |  |  |  |
  #    <--- Definition                                   |  |  |  |
  #           <--- VariableDefinition -------------------+  |  |  |
  #                  <--- PseudoVariableDefinition          |  |  |
  #           <--- FunctionDefinition ----------------------+  |  |
  #                  <--- KandRFunctionDefinition              |  |
  #                  <--- AnsiFunctionDefinition               |  |
  #           <--- ParameterDefinition                         |  |
  #    <--- TypeDeclaration -----------------------------------+  |
  #           <--- TypedefDeclaration                             |
  #           <--- StructTypeDeclaration                          |
  #                  <--- PseudoStructTypeDeclaration             |
  #           <--- UnionTypeDeclaration                           |
  #                  <--- PseudoUnionTypeDeclaration              |
  #           <--- EnumTypeDeclaration                            |
  #                  <--- PseudoEnumTypeDeclaration               |
  #    <--- DeclarationSpecifiers                                 |
  #    <--- InitDeclarator                                        |
  #    <--- TypeSpecifier                                         |
  #           <--- StandardTypeSpecifier                          |
  #           <--- TypedefTypeSpecifier                           |
  #           <--- StructSpecifier                                |
  #           <--- UnionSpecifier                                 |
  #           <--- TypeofTypeSpecifier                            |
  #    <--- StructDeclaration                                     |
  #    <--- MemberDeclaration                                     |
  #    <--- SpecifierQualifierList                                |
  #    <--- StructDeclarator                                      |
  #    <--- EnumSpecifier                                         |
  #    <--- Enumerator -------------------------------------------+
  #    <--- Declarator
  #           <--- IdentifierDeclarator
  #           <--- GroupedDeclarator
  #           <--- ArrayDeclarator
  #           <--- FunctionDeclarator
  #                  <--- AnsiFunctionDeclarator
  #                  <--- KandRFunctionDeclarator
  #                  <--- AbbreviatedFunctionDeclarator
  #           <--- AbstractDeclarator
  #                  <--- PointerAbstractDeclarator
  #                  <--- GroupedAbstractDeclarator
  #                  <--- ArrayAbstractDeclarator
  #                  <--- FunctionAbstractDeclarator
  #    <--- ParameterTypeList
  #    <--- ParameterDeclaration
  #    <--- Statement
  #           <--- ErrorStatement
  #           <--- LabeledStatement
  #                  <--- GenericLabeledStatement
  #                  <--- CaseLabeledStatement
  #                  <--- DefaultLabeledStatement
  #           <--- CompoundStatement
  #           <--- ExpressionStatement
  #           <--- SelectionStatement
  #                  <--- IfStatement
  #                  <--- IfElseStatement
  #                  <--- SwitchStatement
  #           <--- IterationStatement
  #                  <--- WhileStatement
  #                  <--- DoStatement
  #                  <--- ForStatement
  #                  <--- C99ForStatement
  #           <--- JumpStatement
  #                  <--- GotoStatement
  #                  <--- ContinueStatement
  #                  <--- BreakStatement
  #                  <--- ReturnStatement
  #    <--- TranslationUnit
  #    <--- TypeName
  #    <--- Initializer
  class SyntaxNode
    include Visitable

    def initialize
      @head_token = nil
      @tail_token = nil
      @subsequent_sequence_point = nil
    end

    attr_accessor :head_token
    attr_accessor :tail_token
    attr_reader :subsequent_sequence_point

    def location
      subclass_responsibility
    end

    def head_location
      @head_token ? @head_token.location : nil
    end

    def tail_location
      @tail_token ? @tail_token.location : nil
    end

    def inspect(indent = 0)
      subclass_responsibility
    end

    def short_class_name
      self.class.name.sub(/\A.*::/, "")
    end

    protected
    # === DESCRIPTION
    # Append a subsequent sequence-point of this node.
    def append_sequence_point!
      @subsequent_sequence_point = SequencePoint.new(self)
    end

    # === DESCRIPTION
    # Delete a subsequent sequence-point of this node.
    def delete_sequence_point!
      @subsequent_sequence_point = nil
    end
  end

  module SyntaxNodeCollector
    def collect_object_specifiers(node)
      if node
        ObjectSpecifierCollector.new.tap { |collector|
          node.accept(collector)
        }.object_specifiers
      else
        []
      end
    end
    module_function :collect_object_specifiers

    def collect_identifier_declarators(node)
      if node
        IdentifierDeclaratorCollector.new.tap { |collector|
          node.accept(collector)
        }.identifier_declarators
      else
        []
      end
    end
    module_function :collect_identifier_declarators

    def collect_typedef_type_specifiers(node)
      if node
        TypedefTypeSpecifierCollector.new.tap { |collector|
          node.accept(collector)
        }.typedef_type_specifiers
      else
        []
      end
    end
    module_function :collect_typedef_type_specifiers

    def collect_function_declarators(node)
      if node
        FunctionDeclaratorCollector.new.tap { |collector|
          node.accept(collector)
        }.function_declarators
      else
        []
      end
    end
    module_function :collect_function_declarators

    def collect_simple_assignment_expressions(node)
      if node
        SimpleAssignmentExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.simple_assignment_expressions
      else
        []
      end
    end
    module_function :collect_simple_assignment_expressions

    def collect_compound_assignment_expressions(node)
      if node
        CompoundAssignmentExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.compound_assignment_expressions
      else
        []
      end
    end
    module_function :collect_compound_assignment_expressions

    def collect_prefix_increment_expressions(node)
      if node
        PrefixIncrementExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.prefix_increment_expressions
      else
        []
      end
    end
    module_function :collect_prefix_increment_expressions

    def collect_prefix_decrement_expressions(node)
      if node
        PrefixDecrementExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.prefix_decrement_expressions
      else
        []
      end
    end
    module_function :collect_prefix_decrement_expressions

    def collect_postfix_increment_expressions(node)
      if node
        PostfixIncrementExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.postfix_increment_expressions
      else
        []
      end
    end
    module_function :collect_postfix_increment_expressions

    def collect_postfix_decrement_expressions(node)
      if node
        PostfixDecrementExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.postfix_decrement_expressions
      else
        []
      end
    end
    module_function :collect_postfix_decrement_expressions

    def collect_additive_expressions(node)
      if node
        AdditiveExpressionCollector.new.tap { |collector|
          node.accept(collector)
        }.additive_expressions
      else
        []
      end
    end
    module_function :collect_additive_expressions

    def collect_generic_labeled_statements(node)
      if node
        GenericLabeledStatementCollector.new.tap { |collector|
          node.accept(collector)
        }.generic_labeled_statements
      else
        []
      end
    end
    module_function :collect_generic_labeled_statements

    def collect_if_statements(node)
      if node
        IfStatementCollector.new.tap { |collector|
          node.accept(collector)
        }.if_statements
      else
        []
      end
    end
    module_function :collect_if_statements

    def collect_if_else_statements(node)
      if node
        IfElseStatementCollector.new.tap { |collector|
          node.accept(collector)
        }.if_else_statements
      else
        []
      end
    end
    module_function :collect_if_else_statements

    def collect_goto_statements(node)
      if node
        GotoStatementCollector.new.tap { |collector|
          node.accept(collector)
        }.goto_statements
      else
        []
      end
    end
    module_function :collect_goto_statements

    def collect_array_declarators(node)
      if node
        ArrayDeclaratorCollector.new.tap { |collector|
          node.accept(collector)
        }.array_declarators
      else
        []
      end
    end
    module_function :collect_array_declarators

    def collect_constant_specifiers(node)
      if node
        ConstantSpecifierCollector.new.tap { |collector|
          node.accept(collector)
        }.constant_specifiers
      else
        []
      end
    end
    module_function :collect_constant_specifiers
  end

  class Expression < SyntaxNode
    def initialize
      super
      @full = false
    end

    def full=(full_expression)
      @full = full_expression

      # NOTE: The ISO C99 standard saids;
      #
      # Annex C (informative) Sequence points
      #
      # 1 The following are the sequence points described in 5.1.2.3:
      #
      #   -- The end of a full expression: an initializer (6.7.8); the
      #      expression in an expression statement (6.8.3); the controlling
      #      expression of a while or do statement (6.8.5); each of the
      #      expressions of a for statement (6.8.5.3); the expression in a
      #      return statement (6.8.6.4).
      if full_expression
        append_sequence_point!
      else
        delete_sequence_point!
      end
    end

    def have_side_effect?
      subclass_responsibility
    end

    def constant?(enumerator_table)
      object_specifiers.all? { |object_specifier|
        enumerator_table.lookup(object_specifier.identifier.value)
      }
    end

    def object_specifiers
      ObjectSpecifierCollector.new.tap { |collector|
        self.accept(collector)
      }.object_specifiers
    end

    def logical?
      subclass_responsibility
    end

    def to_s
      subclass_responsibility
    end
  end

  class ErrorExpression < Expression
    def initialize(error_token)
      super()
      @error_token = error_token
    end

    def location
      head_location
    end

    def have_side_effect?
      true
    end

    def logical?
      false
    end

    def to_s
      @error_token.value
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@error_token.value}"
    end
  end

  class PrimaryExpression < Expression
    def have_side_effect?
      false
    end

    def logical?
      false
    end
  end

  class ObjectSpecifier < PrimaryExpression
    def initialize(identifier)
      super()
      @identifier = identifier
    end

    attr_reader :identifier

    def location
      @identifier.location
    end

    def logical?
      false
    end

    def to_s
      @identifier.value
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@identifier.value}"
    end
  end

  class ConstantSpecifier < PrimaryExpression
    def initialize(constant)
      super()
      @constant = constant
    end

    attr_reader :constant

    def location
      @constant.location
    end

    def prefix
      @constant.value.scan(/\A(?:0x|0b|L)/i).first
    end

    def suffix
      @constant.value.scan(/(?:[UL]+|[FL]+)\z/i).first
    end

    def character?
      @constant.value =~ /'.*'/
    end

    def logical?
      false
    end

    def to_s
      @constant.value
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@constant.value}"
    end
  end

  class StringLiteralSpecifier < PrimaryExpression
    def initialize(literal)
      super()
      @literal = literal
    end

    attr_reader :literal

    def location
      @literal.location
    end

    def prefix
      @literal.value.scan(/\AL/i).first
    end

    def logical?
      false
    end

    def to_s
      @literal.value
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@literal.value}"
    end
  end

  class NullConstantSpecifier < PrimaryExpression
    def initialize(token)
      super()
      @token = token
    end

    attr_reader :token

    def location
      @token.location
    end

    def logical?
      false
    end

    def to_s
      @token.value
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@token.value}"
    end
  end

  class GroupedExpression < PrimaryExpression
    def initialize(expression)
      super()
      @expression = expression
    end

    attr_reader :expression

    def location
      head_location
    end

    def have_side_effect?
      @expression.have_side_effect?
    end

    def logical?
      @expression.logical?
    end

    def to_s
      "(#{expression.to_s})"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect})\n" +
        @expression.inspect(indent + 1)
    end
  end

  class PostfixExpression < Expression
    def initialize(operator)
      @operator = operator
    end

    attr_reader :operator

    def location
      @operator.location
    end

    def logical?
      false
    end
  end

  class ArraySubscriptExpression < PostfixExpression
    def initialize(expression, array_subscript, left_bracket)
      super(left_bracket)
      @expression = expression
      @array_subscript = array_subscript
    end

    attr_reader :expression
    attr_reader :array_subscript

    def have_side_effect?
      @expression.have_side_effect? || @array_subscript.have_side_effect?
    end

    def to_s
      "#{@expression.to_s}[#{@array_subscript.to_s}]"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect})\n" +
        @expression.inspect(indent + 1) + "\n" +
        @array_subscript.inspect(indent + 1)
    end
  end

  class FunctionCallExpression < PostfixExpression
    def initialize(expression, argument_expressions, left_paren)
      super(left_paren)
      @expression = expression
      @argument_expressions = argument_expressions
    end

    attr_reader :expression
    attr_reader :argument_expressions

    def have_side_effect?
      true
    end

    def to_s
      "#{@expression.to_s}(" +
        @argument_expressions.map { |expr| expr.to_s }.join(",") + ")"
    end

    def inspect(indent = 0)
      ([" " * indent + "#{short_class_name} (#{location.inspect})"] +
       [@expression.inspect(indent + 1)] +
         @argument_expressions.map { |a| a.inspect(indent + 1) }).join("\n")
    end
  end

  class MemberAccessByValueExpression < PostfixExpression
    def initialize(expression, identifier, dot)
      super(dot)
      @expression = expression
      @identifier = identifier
    end

    attr_reader :expression
    attr_reader :identifier

    def have_side_effect?
      @expression.have_side_effect?
    end

    def to_s
      "#{@expression.to_s}.#{@identifier.value}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@expression.inspect}.#{@identifier.value}"
    end
  end

  class MemberAccessByPointerExpression < PostfixExpression
    def initialize(expression, identifier, arrow)
      super(arrow)
      @expression = expression
      @identifier = identifier
    end

    attr_reader :expression
    attr_reader :identifier

    def have_side_effect?
      @expression.have_side_effect?
    end

    def to_s
      "#{@expression.to_s}->#{@identifier.value}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@expression.inspect}->#{@identifier.value}"
    end
  end

  class BitAccessByValueExpression < PostfixExpression
    def initialize(expression, constant, dot)
      super(dot)
      @expression = expression
      @constant = constant
    end

    attr_reader :expression
    attr_reader :constant

    def have_side_effect?
      @expression.have_side_effect?
    end

    def to_s
      "#{@expression.to_s}.#{@constant.value}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@expression.inspect}.#{@constant.value}"
    end
  end

  class BitAccessByPointerExpression < PostfixExpression
    def initialize(expression, constant, arrow)
      super(arrow)
      @expression = expression
      @constant = constant
    end

    attr_reader :expression
    attr_reader :constant

    def have_side_effect?
      @expression.have_side_effect?
    end

    def to_s
      "#{@expression.to_s}->#{@constant.value}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@expression.inspect}->#{@constant.value}"
    end
  end

  class PostfixIncrementExpression < PostfixExpression
    def initialize(operand, operator)
      super(operator)
      @operand = operand
    end

    attr_reader :operand

    def have_side_effect?
      true
    end

    def to_s
      "#{@operand.to_s}++"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        @operand.inspect
    end
  end

  class PostfixDecrementExpression < PostfixExpression
    def initialize(operand, operator)
      super(operator)
      @operand = operand
    end

    attr_reader :operand

    def have_side_effect?
      true
    end

    def to_s
      "#{@operand.to_s}--"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        @operand.inspect
    end
  end

  class CompoundLiteralExpression < PostfixExpression
    def initialize(type_name, initializers, left_paren)
      super(left_paren)
      @type_name = type_name
      @initializers = initializers
    end

    attr_reader :type_name
    attr_reader :initializers

    def have_side_effect?
      false
    end

    def to_s
      "(#{@type_name.to_s}){" +
        @initializers.map { |ini| ini.to_s }.join(",") + "}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
          @initializers.map { |init| init.inspect }.join(",")
    end
  end

  class UnaryExpression < Expression
    def initialize(operator, operand)
      super()
      @operator = operator
      @operand = operand
    end

    attr_reader :operator
    attr_reader :operand

    def location
      @operator.location
    end

    def have_side_effect?
      @operand.have_side_effect?
    end

    def to_s
      @operator.value + @operand.to_s
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@operator.value} #{@operand.inspect}"
    end
  end

  class PrefixIncrementExpression < UnaryExpression
    def have_side_effect?
      true
    end

    def logical?
      false
    end
  end

  class PrefixDecrementExpression < UnaryExpression
    def have_side_effect?
      true
    end

    def logical?
      false
    end
  end

  class AddressExpression < UnaryExpression
    def logical?
      false
    end
  end

  class IndirectionExpression < UnaryExpression
    def logical?
      false
    end
  end

  class UnaryArithmeticExpression < UnaryExpression
    def logical?
      @operator.type == "!"
    end
  end

  class SizeofExpression < UnaryExpression
    def logical?
      false
    end

    def to_s
      "sizeof(#{@operand.to_s})"
    end
  end

  class SizeofTypeExpression < UnaryExpression
    def have_side_effect?
      false
    end

    def logical?
      false
    end

    def to_s
      "sizeof(#{@operand.to_s})"
    end
  end

  class AlignofExpression < UnaryExpression
    def logical?
      false
    end

    def to_s
      "alignof(#{@operand.to_s})"
    end
  end

  class AlignofTypeExpression < UnaryExpression
    def have_side_effect?
      false
    end

    def logical?
      false
    end

    def to_s
      "alignof(#{@operand.to_s})"
    end
  end

  class CastExpression < Expression
    def initialize(type_name, operand)
      super()
      @type_name = type_name
      @operand = operand
    end

    attr_reader :type_name
    attr_reader :operand

    def location
      head_location
    end

    def have_side_effect?
      @operand.have_side_effect?
    end

    def logical?
      false
    end

    def to_s
      "(#{@type_name.to_s}) #{@operand.to_s}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@type_name.inspect} #{@operand.inspect}"
    end
  end

  class BinaryExpression < Expression
    def initialize(operator, lhs_operand, rhs_operand)
      super()
      @operator = operator
      @lhs_operand = lhs_operand
      @rhs_operand = rhs_operand
    end

    attr_reader :operator
    attr_reader :lhs_operand
    attr_reader :rhs_operand

    def location
      @operator.location
    end

    def have_side_effect?
      lhs_operand.have_side_effect? || rhs_operand.have_side_effect?
    end

    def to_s
      "#{@lhs_operand.to_s} #{@operator.value} #{@rhs_operand.to_s}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{@operator.value} #{lhs_operand.inspect} #{rhs_operand.inspect}"
    end
  end

  class MultiplicativeExpression < BinaryExpression
    def logical?
      false
    end
  end

  class AdditiveExpression < BinaryExpression
    def logical?
      false
    end
  end

  class ShiftExpression < BinaryExpression
    def logical?
      false
    end
  end

  class RelationalExpression < BinaryExpression
    def logical?
      true
    end
  end

  class EqualityExpression < BinaryExpression
    def logical?
      true
    end
  end

  class AndExpression < BinaryExpression
    def logical?
      false
    end
  end

  class ExclusiveOrExpression < BinaryExpression
    def logical?
      false
    end
  end

  class InclusiveOrExpression < BinaryExpression
    def logical?
      false
    end
  end

  class LogicalAndExpression < BinaryExpression
    def initialize(operator, lhs_operand, rhs_operand)
      super

      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.13 Logical AND operator
      #
      # Semantics
      #
      # 4 Unlike the bitwise binary & operator, the && operator guarantees
      #   left-to-right evaluation; there is a sequence point after the
      #   evaluation of the first operand.  If the first operand compares equal
      #   to 0, the second operand is not evaluated.
      @lhs_operand.append_sequence_point!
    end

    def logical?
      true
    end
  end

  class LogicalOrExpression < BinaryExpression
    def initialize(operator, lhs_operand, rhs_operand)
      super

      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.14 Logical OR operator
      #
      # Semantics
      #
      # 4 Unlike the bitwise | operator, the || operator guarantees
      #   left-to-right evaluation; there is a sequence point after the
      #   evaluation of the first operand.  If the first operand compares
      #   unequal to 0, the second operand is not evaluated.
      @lhs_operand.append_sequence_point!
    end

    def logical?
      true
    end
  end

  class ConditionalExpression < Expression
    def initialize(condition, then_expression, else_expression, question_mark)
      super()
      @condition = condition
      @then_expression = then_expression
      @else_expression = else_expression
      @question_mark = question_mark

      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.15 Conditional operator
      #
      # Semantics
      #
      # 4 The first operand is evaluated; there is a sequence poit after its
      #   evaluation.  The second operand is evaluated only if the first
      #   compares unequal to 0; the third operand is evaluated only if the
      #   first compares equal to 0; thre result is the value of the second or
      #   third operand (whichever is evaluated), converted to the type
      #   described below.  If an attempt is made to modify the result of a
      #   conditional operator or to access it after the next sequence point,
      #   the behavior is undefined.
      @condition.append_sequence_point!
    end

    attr_reader :condition
    attr_reader :then_expression
    attr_reader :else_expression

    def location
      @question_mark.location
    end

    def have_side_effect?
      @condition.have_side_effect? ||
        @then_expression.have_side_effect? ||
        @else_expression.have_side_effect?
    end

    def logical?
      @then_expression.logical? || @else_expression.logical?
    end

    def to_s
      "#{@condition.to_s} ? " +
        "#{@then_expression.to_s} : #{@else_expression.to_s}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @condition.inspect(indent + 1) + "\n" +
        @then_expression.inspect(indent + 1) + "\n" +
        @else_expression.inspect(indent + 1)
    end
  end

  class SimpleAssignmentExpression < BinaryExpression
    def have_side_effect?
      true
    end

    def logical?
      @rhs_operand.logical?
    end
  end

  class CompoundAssignmentExpression < BinaryExpression
    def have_side_effect?
      true
    end

    def logical?
      false
    end
  end

  class CommaSeparatedExpression < Expression
    def initialize(first_expression)
      super()
      @expressions = []
      push(first_expression)
    end

    attr_reader :expressions

    def location
      head_location
    end

    def have_side_effect?
      @expressions.any? { |expr| expr.have_side_effect? }
    end

    def logical?
      @expressions.last.logical?
    end

    def to_s
      @expressions.map { |expr| expr.to_s }.join(",")
    end

    def push(expression)
      @expressions.push(expression)

      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.17 Comma operator
      #
      # Semantics
      #
      # 2 The left operand of a comma operator is evaluated as a void
      #   expression; there is a sequence point after its evaluation.  Then the
      #   right operand is evaluated; the result has its type and value.  If an
      #   attempt is made to modify the result of a comma operator or to access
      #   it after the next sequence point, the behavior is undefined.
      @expressions[-2].append_sequence_point! if @expressions.size > 1
      self
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @expressions.map { |expr| expr.inspect(indent + 1) }.join("\n")
    end
  end

  module DeclarationSpecifiersHolder
    def storage_class_specifier
      @declaration_specifiers ?
        @declaration_specifiers.storage_class_specifier : nil
    end

    def function_specifier
      # NOTE: A function-specifier should only be in function-definitions.
      @declaration_specifiers ?
        @declaration_specifiers.function_specifier : nil
    end

    def type_qualifiers
      @declaration_specifiers ? @declaration_specifiers.type_qualifiers : []
    end

    def type_specifiers
      @declaration_specifiers ? @declaration_specifiers.type_specifiers : []
    end

    def explicitly_typed?
      @declaration_specifiers && @declaration_specifiers.explicitly_typed?
    end

    def implicitly_typed?
      !explicitly_typed?
    end
  end

  class Declaration < SyntaxNode
    include DeclarationSpecifiersHolder

    def initialize(declaration_specifiers, init_declarators, symbol_table)
      super()
      @declaration_specifiers = declaration_specifiers
      @init_declarators = init_declarators
      @items = build_items(declaration_specifiers, init_declarators,
                           symbol_table)
    end

    attr_reader :declaration_specifiers
    attr_reader :init_declarators
    attr_reader :items

    def location
      head_location
    end

    def inspect(indent = 0)
      ([" " * indent + "#{short_class_name} (#{location.inspect})"] +
       @items.map { |item| item.inspect(indent + 1) }).join("\n")
    end

    private
    def build_items(declaration_specifiers, init_declarators, symbol_table)
      build_type_declaration(
        declaration_specifiers, init_declarators, symbol_table) +
      build_function_declaration(
        declaration_specifiers, init_declarators, symbol_table) +
      build_variable_declaration(
        declaration_specifiers, init_declarators, symbol_table) +
      build_variable_definition(
        declaration_specifiers, init_declarators, symbol_table)
    end

    def build_type_declaration(declaration_specifiers, init_declarators,
                               symbol_table)
      return [] unless declaration_specifiers

      result = []

      declaration_specifiers.type_specifiers.each do |type_specifier|
        visitor = TypeDeclarationBuilder.new(symbol_table)
        type_specifier.accept(visitor)
        result.concat(visitor.type_declarations)
      end

      if declaration_specifiers.storage_class_specifier &&
          declaration_specifiers.storage_class_specifier.type == :TYPEDEF
        init_declarators.each do |init_declarator|
          symbol = symbol_table.create_new_symbol(
            TypedefName, init_declarator.declarator.identifier)
          result.push(TypedefDeclaration.new(declaration_specifiers,
                                             init_declarator, symbol))
        end
      end

      result
    end

    def build_function_declaration(declaration_specifiers, init_declarators,
                                   symbol_table)
      if declaration_specifiers &&
          declaration_specifiers.storage_class_specifier &&
          declaration_specifiers.storage_class_specifier.type == :TYPEDEF
        return []
      end

      init_declarators.each_with_object([]) do |init_declarator, result|
        if init_declarator.declarator.function?
          symbol = symbol_table.create_new_symbol(
            ObjectName, init_declarator.declarator.identifier)
          result.push(FunctionDeclaration.new(declaration_specifiers,
                                              init_declarator, symbol))
        end
      end
    end

    def build_variable_declaration(declaration_specifiers, init_declarators,
                                   symbol_table)
      return [] unless declaration_specifiers

      unless declaration_specifiers.storage_class_specifier &&
          declaration_specifiers.storage_class_specifier.type == :EXTERN
        return []
      end

      init_declarators.each_with_object([]) do |init_declarator, result|
        if init_declarator.declarator.variable?
          declarator = init_declarator.declarator
          symbol =
            symbol_table.create_new_symbol(ObjectName, declarator.identifier)
          result.push(VariableDeclaration.new(declaration_specifiers,
                                              declarator, symbol))
        end
      end
    end

    def build_variable_definition(declaration_specifiers, init_declarators,
                                  symbol_table)
      if declaration_specifiers &&
          declaration_specifiers.storage_class_specifier &&
          (declaration_specifiers.storage_class_specifier.type == :EXTERN ||
           declaration_specifiers.storage_class_specifier.type == :TYPEDEF)
        return []
      end

      init_declarators.each_with_object([]) do |init_declarator, result|
        if init_declarator.declarator.variable?
          symbol = symbol_table.create_new_symbol(
            ObjectName, init_declarator.declarator.identifier)
          result.push(VariableDefinition.new(declaration_specifiers,
                                             init_declarator, symbol))
        end
      end
    end
  end

  class FunctionDeclaration < SyntaxNode
    include SymbolicElement
    include DeclarationSpecifiersHolder
    include SyntaxNodeCollector

    def initialize(declaration_specifiers, init_declarator, symbol)
      super()
      @declaration_specifiers = declaration_specifiers
      @init_declarator = init_declarator
      @symbol = symbol
      @type = nil
    end

    attr_reader :declaration_specifiers
    attr_reader :init_declarator
    attr_reader :symbol
    attr_accessor :type

    def identifier
      @init_declarator.declarator.identifier
    end

    def signature
      FunctionSignature.new(identifier.value, @type)
    end

    def function_declarator
      collect_function_declarators(@init_declarator).first
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        identifier.value
    end
  end

  class VariableDeclaration < SyntaxNode
    include SymbolicElement
    include DeclarationSpecifiersHolder

    def initialize(declaration_specifiers, declarator, symbol)
      super()
      @declaration_specifiers = declaration_specifiers
      @declarator = declarator
      @symbol = symbol
      @type = nil
    end

    attr_reader :declaration_specifiers
    attr_reader :declarator
    attr_reader :symbol
    attr_accessor :type

    def identifier
      @declarator.identifier
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        identifier.value
    end
  end

  class Definition < SyntaxNode
    include DeclarationSpecifiersHolder

    def initialize(declaration_specifiers)
      @declaration_specifiers = declaration_specifiers
      @type = nil
    end

    attr_reader :declaration_specifiers
    attr_accessor :type
  end

  class VariableDefinition < Definition
    include SymbolicElement

    def initialize(declaration_specifiers, init_declarator, symbol)
      super(declaration_specifiers)

      @init_declarator = init_declarator
      @symbol = symbol
    end

    attr_reader :init_declarator
    attr_reader :symbol

    def identifier
      @init_declarator.declarator.identifier
    end

    def initializer
      @init_declarator.initializer
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        identifier.value
    end
  end

  class TypeDeclaration < SyntaxNode
    include SymbolicElement

    def initialize(symbol)
      super()
      @symbol = symbol
    end

    attr_reader :symbol
  end

  class TypedefDeclaration < TypeDeclaration
    include DeclarationSpecifiersHolder

    def initialize(declaration_specifiers, init_declarator, symbol)
      super(symbol)
      @declaration_specifiers = declaration_specifiers
      @init_declarator = init_declarator
      @type = nil
    end

    attr_reader :declaration_specifiers
    attr_reader :init_declarator
    attr_accessor :type

    def identifier
      @init_declarator.declarator.identifier
    end

    def declarator
      @init_declarator.declarator
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        identifier.value
    end
  end

  class StructTypeDeclaration < TypeDeclaration
    def initialize(struct_specifier, symbol)
      super(symbol)
      @struct_specifier = struct_specifier
      @struct_declarations = struct_specifier.struct_declarations
      @type = nil
    end

    attr_reader :struct_specifier
    attr_reader :struct_declarations
    attr_accessor :type

    def identifier
      @struct_specifier.identifier
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{identifier.value}\n" +
          @struct_declarations.map { |sd| sd.inspect(indent + 1) }.join("\n")
    end
  end

  class PseudoStructTypeDeclaration < StructTypeDeclaration
    def initialize(struct_specifier)
      super(struct_specifier, nil)
    end

    def mark_as_referred_by(token) end
  end

  class UnionTypeDeclaration < TypeDeclaration
    def initialize(union_specifier, symbol)
      super(symbol)
      @union_specifier = union_specifier
      @struct_declarations = union_specifier.struct_declarations
      @type = nil
    end

    attr_reader :union_specifier
    attr_reader :struct_declarations
    attr_accessor :type

    def identifier
      @union_specifier.identifier
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{identifier.value}\n" +
          @struct_declarations.map { |sd| sd.inspect(indent + 1) }.join("\n")
    end
  end

  class PseudoUnionTypeDeclaration < UnionTypeDeclaration
    def initialize(union_specifier)
      super(union_specifier, nil)
    end

    def mark_as_referred_by(token) end
  end

  class EnumTypeDeclaration < TypeDeclaration
    def initialize(enum_specifier, symbol)
      super(symbol)
      @enum_specifier = enum_specifier
      @type = nil
    end

    attr_reader :enum_specifier
    attr_accessor :type

    def identifier
      @enum_specifier.identifier
    end

    def location
      identifier.location
    end

    def enumerators
      @enum_specifier.enumerators
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        "#{identifier.value}"
    end
  end

  class PseudoEnumTypeDeclaration < EnumTypeDeclaration
    def initialize(enum_specifier)
      super(enum_specifier, nil)
    end

    def mark_as_referred_by(token) end
  end

  class DeclarationSpecifiers < SyntaxNode
    def initialize
      super
      @storage_class_specifier = nil
      @function_specifier = nil
      @type_qualifiers = []
      @type_specifiers = []
    end

    attr_accessor :storage_class_specifier
    attr_accessor :function_specifier
    attr_reader :type_qualifiers
    attr_reader :type_specifiers

    def location
      head_location
    end

    def explicitly_typed?
      !implicitly_typed?
    end

    def implicitly_typed?
      @type_specifiers.empty?
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class InitDeclarator < SyntaxNode
    def initialize(declarator, initializer)
      super()
      @declarator = declarator
      @initializer = initializer
    end

    attr_reader :declarator
    attr_reader :initializer

    def location
      @declarator.identifier.location
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class TypeSpecifier < SyntaxNode
    def to_s
      subclass_responsibility
    end
  end

  class StandardTypeSpecifier < TypeSpecifier
    def initialize(token)
      super()
      @token = token
    end

    attr_reader :token

    def location
      head_location
    end

    def to_s
      @token.value
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class TypedefTypeSpecifier < TypeSpecifier
    def initialize(token)
      super()
      @token = token
    end

    attr_reader :token

    def location
      head_location
    end

    def identifier
      @token
    end

    def to_s
      @token.value
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        @token.value
    end
  end

  class StructSpecifier < TypeSpecifier
    def initialize(identifier, struct_declarations)
      super()
      @identifier = identifier
      @struct_declarations = struct_declarations
    end

    attr_reader :identifier
    attr_reader :struct_declarations

    def location
      @identifier.location
    end

    def to_s
      if @struct_declarations
        if @struct_declarations.empty?
          "struct #{identifier.value} {}"
        else
          "struct #{identifier.value} { " +
            @struct_declarations.map { |decl| decl.to_s }.join(" ") +
            " }"
        end
      else
        "struct #{identifier.value}"
      end
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class UnionSpecifier < TypeSpecifier
    def initialize(identifier, struct_declarations)
      super()
      @identifier = identifier
      @struct_declarations = struct_declarations
    end

    attr_reader :identifier
    attr_reader :struct_declarations

    def location
      @identifier.location
    end

    def to_s
      if @struct_declarations
        if @struct_declarations.empty?
          "union #{identifier.value} {}"
        else
          "union #{identifier.value} { " +
            @struct_declarations.map { |decl| decl.to_s }.join(" ") +
            " }"
        end
      else
        "union #{identifier.value}"
      end
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class StructDeclaration < SyntaxNode
    def initialize(specifier_qualifier_list, struct_declarators)
      super()
      @specifier_qualifier_list = specifier_qualifier_list
      @struct_declarators = struct_declarators
      @items = build_items(specifier_qualifier_list, struct_declarators)
    end

    attr_reader :specifier_qualifier_list
    attr_reader :struct_declarators
    attr_reader :items

    def location
      @specifier_qualifier_list.location
    end

    def to_s
      @items.map { |item| item.to_s + ";" }.join(" ")
    end

    def inspect(indent = 0)
      ([" " * indent + "#{short_class_name} (#{location.inspect})"] +
       @items.map { |item| item.inspect(indent + 1) }).join("\n")
    end

    private
    def build_items(specifier_qualifier_list, struct_declarators)
      struct_declarators.each_with_object([]) do |struct_declarator, items|
        # FIXME: Must support unnamed bit padding.
        next unless struct_declarator.declarator
        items.push(MemberDeclaration.new(specifier_qualifier_list,
                                         struct_declarator))
      end
    end
  end

  class MemberDeclaration < SyntaxNode
    def initialize(specifier_qualifier_list, struct_declarator)
      super()
      @specifier_qualifier_list = specifier_qualifier_list
      @struct_declarator = struct_declarator
      @type = nil
    end

    attr_reader :specifier_qualifier_list
    attr_reader :struct_declarator
    attr_accessor :type

    def identifier
      @struct_declarator.declarator.identifier
    end

    def location
      identifier.location
    end

    def to_s
      "#{type.brief_image} #{identifier.value}"
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        identifier.value
    end
  end

  class SpecifierQualifierList < SyntaxNode
    def initialize
      super
      @type_specifiers = []
      @type_qualifiers = []
    end

    attr_reader :type_specifiers
    attr_reader :type_qualifiers

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class StructDeclarator < SyntaxNode
    def initialize(declarator, expression)
      super()
      @declarator = declarator
      @expression = expression
    end

    attr_reader :declarator
    attr_reader :expression

    def location
      @declarator ? @declarator.location : @expression.location
    end

    def bitfield?
      @expression != nil
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class EnumSpecifier < TypeSpecifier
    def initialize(identifier, enumerators)
      super()
      @identifier = identifier
      @enumerators = enumerators
    end

    attr_reader :identifier
    attr_reader :enumerators

    def location
      @identifier.location
    end

    def to_s
      if @enumerators
        if @enumerators.empty?
          "enum #{@identifier.value} {}"
        else
          "enum #{@identifier.value} { " +
            @enumerators.map { |enum| enum.to_s }.join(", ") + " }"
        end
      else
        "enum #{identifier.value}"
      end
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class Enumerator < SyntaxNode
    include SymbolicElement

    def initialize(identifier, expression, symbol)
      super()
      @identifier = identifier
      @expression = expression
      @symbol = symbol
    end

    attr_reader :identifier
    attr_reader :expression
    attr_reader :symbol
    attr_accessor :value
    attr_accessor :type

    def location
      @identifier.location
    end

    def to_s
      if @expression
        "#{@identifier.value} = #{@expression.to_s}"
      else
        "#{@identifier.value}"
      end
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class TypeofTypeSpecifier < TypeSpecifier
    def initialize(expression, type_name)
      super()
      @expression = expression
      @type_name = type_name
    end

    attr_reader :expression
    attr_reader :type_name

    def location
      head_location
    end

    def to_s
      if @expression
        "__typeof__(#{@expression.to_s})"
      else
        "__typeof__(#{@type_name.to_s})"
      end
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class Declarator < SyntaxNode
    def initialize
      super
      @pointer = nil
      @full = false
    end

    attr_accessor :pointer

    def identifier
      subclass_responsibility
    end

    def abstract?
      false
    end

    def full=(full_declarator)
      @full = full_declarator

      # NOTE: The ISO C99 standard saids;
      #
      # 6.7.5 Declarators
      #
      # Semantics
      #
      # 3 A full declarator is a declarator that is not part of another
      #   declarator.  The end of a full declarator is a sequence point.  If,
      #   in the nested sequence of declarators in a full declarator, there is
      #   a declarator specifying a variable length array type, the type
      #   specified by the full declarator is said to be variably modified.
      #   Furthermore, any type derived by declarator type derivation from a
      #   variably modified type is itself variably modified.
      if full_declarator
        append_sequence_point!
      else
        delete_sequence_point!
      end
    end

    def full?
      @full
    end

    def base
      subclass_responsibility
    end

    def function?
      subclass_responsibility
    end

    def variable?
      !function?
    end
  end

  class IdentifierDeclarator < Declarator
    def initialize(identifier)
      super()
      @identifier = identifier
    end

    attr_reader :identifier

    def location
      @identifier.location
    end

    def base
      nil
    end

    def parameter_type_list
      nil
    end

    def identifier_list
      nil
    end

    def function?(stack = [])
      stack.push(:pointer) if pointer
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} #{@identifier.value}"
    end
  end

  class GroupedDeclarator < Declarator
    def initialize(declarator)
      super()
      @base = declarator
    end

    attr_reader :base

    def location
      @base.location
    end

    def identifier
      @base.identifier
    end

    def parameter_type_list
      @base.parameter_type_list
    end

    def identifier_list
      @base.identifier_list
    end

    def function?(stack = [])
      stack.push(:pointer) if pointer
      @base.function?(stack)
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" + @base.inspect(indent + 1)
    end
  end

  class ArrayDeclarator < Declarator
    def initialize(declarator, size_expression)
      super()
      @base = declarator
      @size_expression = size_expression
    end

    attr_reader :base
    attr_reader :size_expression

    def location
      @base.location
    end

    def identifier
      @base.identifier
    end

    def parameter_type_list
      @base.parameter_type_list
    end

    def identifier_list
      @base.identifier_list
    end

    def function?(stack = [])
      stack.push(:pointer) if pointer
      stack.push(:array)
      @base.function?(stack)
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" + @base.inspect(indent + 1)
    end
  end

  class FunctionDeclarator < Declarator
    def initialize(declarator)
      super()
      @base = declarator
    end

    attr_reader :base

    def location
      @base.location
    end

    def identifier
      @base.identifier
    end

    def function?(stack = [])
      stack.push(:pointer) if pointer
      stack.push(:function)
      @base.function?(stack)
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" + @base.inspect(indent + 1)
    end
  end

  class AnsiFunctionDeclarator < FunctionDeclarator
    def initialize(declarator, parameter_type_list)
      super(declarator)
      @parameter_type_list = parameter_type_list
    end

    attr_accessor :parameter_type_list

    def identifier_list
      @base.identifier_list
    end
  end

  class KandRFunctionDeclarator < FunctionDeclarator
    def initialize(declarator, identifier_list)
      super(declarator)
      @identifier_list = identifier_list
    end

    attr_reader :identifier_list

    def parameter_type_list
      @base.parameter_type_list
    end
  end

  class AbbreviatedFunctionDeclarator < FunctionDeclarator
    def parameter_type_list
      @base.parameter_type_list
    end

    def identifier_list
      @base.identifier_list
    end
  end

  class ParameterTypeList < SyntaxNode
    def initialize(parameters, have_va_list)
      super()
      @parameters = parameters
      @have_va_list = have_va_list
    end

    attr_reader :parameters

    def location
      head_location
    end

    def have_va_list?
      @have_va_list
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class ParameterDeclaration < SyntaxNode
    include DeclarationSpecifiersHolder

    def initialize(declaration_specifiers, declarator)
      super()
      @declaration_specifiers = declaration_specifiers
      @declarator = declarator
    end

    attr_reader :declaration_specifiers
    attr_reader :declarator

    def location
      @declarator ? @declarator.location : @declaration_specifiers.location
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class Statement < SyntaxNode
    def initialize
      super
      @executed = false
    end

    attr_writer :executed

    def executed?
      @executed
    end
  end

  class ErrorStatement < Statement
    def initialize(error_token)
      super()
      @error_token = error_token
    end

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}"
    end
  end

  class LabeledStatement < Statement; end

  class GenericLabeledStatement < LabeledStatement
    def initialize(label, statement)
      super()
      @label = label
      @statement = statement
      @referrers = []
    end

    attr_reader :label
    attr_reader :statement
    attr_reader :referrers

    def location
      @label.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{@label.inspect})\n" +
        @statement.inspect(indent + 1)
    end
  end

  class CaseLabeledStatement < LabeledStatement
    def initialize(expression, statement)
      super()
      @expression = expression
      @statement = statement
    end

    attr_reader :expression
    attr_reader :statement
    attr_accessor :normalized_expression

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @expression.inspect(indent + 1) + "\n" +
        @statement.inspect(indent + 1)
    end
  end

  class DefaultLabeledStatement < LabeledStatement
    def initialize(statement)
      super()
      @statement = statement
    end

    attr_reader :statement
    attr_accessor :normalized_expression

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" + @statement.inspect(indent + 1)
    end
  end

  class CompoundStatement < Statement
    def initialize(block_items)
      super()
      @block_items = block_items
    end

    attr_reader :block_items

    def location
      head_location
    end

    def inspect(indent = 0)
      ([" " * indent + short_class_name] +
       @block_items.map { |item| item.inspect(indent + 1) }).join("\n")
    end
  end

  class ExpressionStatement < Statement
    def initialize(expression)
      super()
      @expression = expression
    end

    attr_reader :expression

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}" +
        (@expression ? "\n#{@expression.inspect(indent + 1)}" : "")
    end
  end

  class SelectionStatement < Statement; end

  class IfStatement < SelectionStatement
    def initialize(expression, statement, header_terminator)
      super()
      @expression = expression
      @statement = statement
      @header_terminator = header_terminator
    end

    attr_reader :expression
    attr_reader :statement
    attr_reader :header_terminator

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @expression.inspect(indent + 1) + "\n" +
        @statement.inspect(indent + 1)
    end
  end

  class IfElseStatement < SelectionStatement
    def initialize(expression, then_statement, else_statement,
                   then_header_terminator, else_header_terminator)
      super()
      @expression = expression
      @then_statement = then_statement
      @else_statement = else_statement
      @then_header_terminator = then_header_terminator
      @else_header_terminator = else_header_terminator
    end

    attr_reader :expression
    attr_reader :then_statement
    attr_reader :else_statement
    attr_reader :then_header_terminator
    attr_reader :else_header_terminator

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @expression.inspect(indent + 1) + "\n" +
        @then_statement.inspect(indent + 1) + "\n" +
        @else_statement.inspect(indent + 1)
    end
  end

  class SwitchStatement < SelectionStatement
    def initialize(expression, statement)
      super()
      @expression = expression
      @statement = statement
      derive_clause_conditions
    end

    attr_reader :expression
    attr_reader :statement

    def location
      head_location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @expression.inspect(indent + 1) + "\n" +
        @statement.inspect(indent + 1)
    end

    private
    def derive_clause_conditions
      case_expressions = []
      default_labeled_statement = nil

      @statement.block_items.each do |block_item|
        while block_item.kind_of?(GenericLabeledStatement)
          block_item = block_item.statement
        end

        case block_item
        when CaseLabeledStatement
          block_item.normalized_expression = equal_to_expression(
            @expression, block_item.expression, block_item.location)
          case_expressions.push(block_item.normalized_expression)
        when DefaultLabeledStatement
          default_labeled_statement = block_item
        end
      end

      if default_labeled_statement
        default_labeled_statement.normalized_expression =
          derive_default_clause_condition(case_expressions,
                                          default_labeled_statement.location)
      end
    end

    def derive_default_clause_condition(case_expressions, location)
      if case_expressions.empty?
        equal_to_expression(@expression, @expression, location)
      else
        case_expressions.map { |expr|
          not_equal_to_expression(expr.lhs_operand, expr.rhs_operand, location)
        }.reduce do |result, expr|
          logical_and_expression(result, expr, location)
        end
      end
    end

    def equal_to_expression(lhs, rhs, location)
      EqualityExpression.new(equal_to_operator(location), lhs, rhs)
    end

    def not_equal_to_expression(lhs, rhs, location)
      EqualityExpression.new(not_equal_to_operator(location), lhs, rhs)
    end

    def logical_and_expression(lhs, rhs, location)
      LogicalAndExpression.new(logical_and_operator(location), lhs, rhs)
    end

    def equal_to_operator(location)
      Token.new("==", "==", location)
    end

    def not_equal_to_operator(location)
      Token.new("!=", "!=", location)
    end

    def logical_and_operator(location)
      Token.new("&&", "&&", location)
    end
  end

  class IterationStatement < Statement
    include SyntaxNodeCollector
    extend Memoizable

    def deduct_controlling_expression
      subclass_responsibility
    end

    def varying_variable_names
      collect_varying_variable_names(self).uniq
    end

    def varying_expressions
      collect_varying_binary_expressions(self) +
        collect_varying_unary_expressions(self)
    end

    private
    def deduct_controlling_expression_candidates(expressions)
      varying_var_names = varying_variable_names
      expressions.compact.select do |expr|
        collect_object_specifiers(expr).any? do |os|
          varying_var_names.include?(os.identifier.value)
        end
      end
    end

    def collect_loop_breaking_selection_statements(node)
      collect_loop_breaking_if_statements(node) +
        collect_loop_breaking_if_else_statements(node)
    end

    def collect_loop_breaking_if_statements(node)
      collect_if_statements(node).select do |stmt|
        contain_loop_breaking?(stmt.statement)
      end
    end

    def collect_loop_breaking_if_else_statements(node)
      collect_if_else_statements(node).select do |stmt|
        contain_loop_breaking?(stmt.then_statement) ||
          contain_loop_breaking?(stmt.else_statement)
      end
    end

    def contain_loop_breaking?(node)
      items = node.kind_of?(CompoundStatement) ? node.block_items : [node]
      items.any? do |item|
        item = item.statement while item.kind_of?(GenericLabeledStatement)
        case item
        when GotoStatement
          # FIXME: Must check whether the goto-statement goes outside of the
          #        loop or not.
          true
        when BreakStatement
          true
        when ReturnStatement
          true
        else
          false
        end
      end
    end

    def collect_varying_variable_names(node)
      collect_varying_variable_names_in_binary_expression(node) +
        collect_varying_variable_names_in_unary_expression(node)
    end

    def collect_varying_variable_names_in_binary_expression(node)
      collect_varying_binary_expressions(node).map do |expr|
        expr.lhs_operand.identifier.value
      end
    end

    def collect_varying_binary_expressions(node)
      all_varying_binary_exprs =
        collect_simple_assignment_expressions(node) +
        collect_compound_assignment_expressions(node)

      all_varying_binary_exprs.select do |expr|
        expr.lhs_operand.kind_of?(ObjectSpecifier)
      end
    end
    memoize :collect_varying_binary_expressions

    def collect_varying_variable_names_in_unary_expression(node)
      collect_varying_unary_expressions(node).map do |expr|
        expr.operand.identifier.value
      end
    end

    def collect_varying_unary_expressions(node)
      all_varying_unary_exprs =
        collect_prefix_increment_expressions(node) +
        collect_prefix_decrement_expressions(node) +
        collect_postfix_increment_expressions(node) +
        collect_postfix_decrement_expressions(node)

      all_varying_unary_exprs.select do |expr|
        expr.operand.kind_of?(ObjectSpecifier)
      end
    end
    memoize :collect_varying_unary_expressions
  end

  class WhileStatement < IterationStatement
    def initialize(expression, statement, header_terminator)
      super()
      @expression = expression
      @statement = statement
      @header_terminator = header_terminator
    end

    attr_reader :expression
    attr_reader :statement
    attr_reader :header_terminator

    def location
      head_location
    end

    def deduct_controlling_expression
      selection_stmts = collect_loop_breaking_selection_statements(@statement)
      expressions = [
        @expression,
        *selection_stmts.map { |stmt| stmt.expression }
      ]

      # FIXME: When many loop breaking selection-statements are found, how can
      #        I select one selection-statement?
      # FIXME: When the loop breaking selection-statement is a
      #        if-else-statement and the loop breaking is in the else branch,
      #        the controlling expression should be inverted.
      deduct_controlling_expression_candidates(expressions).first
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @expression.inspect(indent + 1) + "\n" +
        @statement.inspect(indent + 1)
    end
  end

  class DoStatement < IterationStatement
    def initialize(statement, expression, header_terminator, footer_initiator)
      super()
      @statement = statement
      @expression = expression
      @header_terminator = header_terminator
      @footer_initiator = footer_initiator
    end

    attr_reader :statement
    attr_reader :expression
    attr_reader :header_terminator
    attr_reader :footer_initiator

    def location
      head_location
    end

    def deduct_controlling_expression
      selection_stmts = collect_loop_breaking_selection_statements(@statement)
      expressions = [
        @expression,
        *selection_stmts.map { |stmt| stmt.expression }
      ]

      # FIXME: When many loop breaking selection-statements are found, how can
      #        I select one selection-statement?
      # FIXME: When the loop breaking selection-statement is a
      #        if-else-statement and the loop breaking is in the else branch,
      #        the controlling expression should be inverted.
      deduct_controlling_expression_candidates(expressions).first
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @statement.inspect(indent + 1) + "\n" +
        @expression.inspect(indent + 1)
    end
  end

  class ForStatement < IterationStatement
    def initialize(initial_statement, condition_statement, expression,
                   body_statement, header_terminator)
      super()
      @initial_statement = initial_statement
      @condition_statement = condition_statement
      @expression = expression
      @body_statement = body_statement
      @header_terminator = header_terminator
    end

    attr_reader :initial_statement
    attr_reader :condition_statement
    attr_reader :expression
    attr_reader :body_statement
    attr_reader :header_terminator

    def location
      head_location
    end

    def deduct_controlling_expression
      selection_stmts =
        collect_loop_breaking_selection_statements(@body_statement)
      expressions = [
        @condition_statement.expression,
        *selection_stmts.map { |stmt| stmt.expression }
      ]

      # FIXME: When many loop breaking selection-statements are found, how can
      #        I select one selection-statement?
      # FIXME: When the loop breaking selection-statement is a
      #        if-else-statement and the loop breaking is in the else branch,
      #        the controlling expression should be inverted.
      deduct_controlling_expression_candidates(expressions).first
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @initial_statement.inspect(indent + 1) + "\n" +
        @condition_statement.inspect(indent + 1) +
        (@expression ? "\n#{@expression.inspect(indent + 1)}" : "") +
        "\n" + @body_statement.inspect(indent + 1)
    end
  end

  class C99ForStatement < IterationStatement
    def initialize(declaration, condition_statement, expression,
                   body_statement, header_terminator)
      super()
      @declaration = declaration
      @condition_statement = condition_statement
      @expression = expression
      @body_statement = body_statement
      @header_terminator = header_terminator
    end

    attr_reader :declaration
    attr_reader :condition_statement
    attr_reader :expression
    attr_reader :body_statement
    attr_reader :header_terminator

    def location
      head_location
    end

    def deduct_controlling_expression
      selection_stmts =
        collect_loop_breaking_selection_statements(@body_statement)
      expressions = [
        @condition_statement.expression,
        *selection_stmts.map { |stmt| stmt.expression }
      ]

      # FIXME: When many loop breaking selection-statements are found, how can
      #        I select one selection-statement?
      # FIXME: When the loop breaking selection-statement is a
      #        if-else-statement and the loop breaking is in the else branch,
      #        the controlling expression should be inverted.
      deduct_controlling_expression_candidates(expressions).first
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name}\n" +
        @declaration.inspect(indent + 1) + "\n" +
        @condition_statement.inspect(indent + 1) +
        (@expression ? "\n#{@expression.inspect(indent + 1)}" : "") +
        "\n" + @body_statement.inspect(indent + 1)
    end
  end

  class JumpStatement < Statement
    def location
      head_location
    end
  end

  class GotoStatement < JumpStatement
    def initialize(identifier)
      super()
      @identifier = identifier
    end

    attr_reader :identifier

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{@identifier.inspect})"
    end
  end

  class ContinueStatement < JumpStatement
    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect})"
    end
  end

  class BreakStatement < JumpStatement
    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect})"
    end
  end

  class ReturnStatement < JumpStatement
    def initialize(expression)
      super()
      @expression = expression
    end

    attr_reader :expression

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect})" +
        (@expression ? "\n#{@expression.inspect(indent + 1)}" : "")
    end
  end

  class TranslationUnit < SyntaxNode
    def initialize
      super()
      @external_declarations = []
    end

    attr_reader :external_declarations

    def push(external_declaration)
      @external_declarations.push(external_declaration)
      self
    end

    def location
      head_location ? head_location : Location.new
    end

    def inspect(indent = 0)
      ([" " * indent + short_class_name] +
       @external_declarations.map { |d| d.inspect(indent + 1) }).join("\n")
    end
  end

  class FunctionDefinition < Definition
    include SymbolicElement
    include SyntaxNodeCollector

    def initialize(declaration_specifiers, declarator, parameter_definitions,
                   compound_statement, symbol_table)
      super(declaration_specifiers)

      @declarator = declarator
      @parameter_definitions = parameter_definitions
      @function_body = compound_statement
      @symbol = symbol_table.create_new_symbol(ObjectName, identifier)
      @type_declaration = build_type_declaration(declaration_specifiers,
                                                 symbol_table)
      build_label_references(compound_statement)
    end

    attr_reader :declarator
    attr_reader :parameter_definitions
    attr_reader :function_body
    attr_reader :symbol
    attr_reader :type_declaration

    def identifier
      @declarator.identifier
    end

    def signature
      FunctionSignature.new(identifier.value, @type)
    end

    def function_declarator
      collect_function_declarators(@declarator).first
    end

    def lines
      start_line = identifier.location.line_no
      end_line = @function_body.tail_location.line_no
      end_line - start_line + 1
    end

    def location
      identifier.location
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        (storage_class_specifier ? "#{storage_class_specifier.value} " : "") +
        (function_specifier ? "#{function_specifier.value} " : "") +
        "#{identifier.value}\n" +
        @parameter_definitions.map { |p| p.inspect(indent + 1) }.join("\n") +
        "\n#{@function_body.inspect(indent + 1)}"
    end

    private
    def build_type_declaration(declaration_specifiers, symbol_table)
      return nil unless declaration_specifiers
      declaration_specifiers.type_specifiers.each do |type_specifier|
        visitor = TypeDeclarationBuilder.new(symbol_table)
        type_specifier.accept(visitor)
        unless visitor.type_declarations.empty?
          return visitor.type_declarations.first
        end
      end
      nil
    end

    def build_label_references(compound_statement)
      labels = collect_generic_labeled_statements(compound_statement)

      gotos = collect_goto_statements(compound_statement)

      labels.each do |generic_labeled_statement|
        label_name = generic_labeled_statement.label.value
        gotos.select { |goto_statement|
          goto_statement.identifier.value == label_name
        }.each do |goto_statement|
          generic_labeled_statement.referrers.push(goto_statement)
        end
      end
    end
  end

  class KandRFunctionDefinition < FunctionDefinition
    def initialize(declaration_specifiers, declarator, declarations,
                   compound_statement, symbol_table)
      parameter_definitions = create_parameters(declarator.identifier_list,
                                                declarations, symbol_table)
      super(declaration_specifiers, declarator, parameter_definitions,
            compound_statement, symbol_table)
    end

    def identifier_list
      declarator.identifier_list
    end

    private
    def create_parameters(parameter_names, declarations, symbol_table)
      return [] unless parameter_names

      parameter_names.each_with_object([]) do |name, result|
        variable_definition =
          find_variable_definition(declarations, name, symbol_table)
        result.push(
          variable_definition_to_parameter_definition(variable_definition))
      end
    end

    def find_variable_definition(declarations, name, symbol_table)
      declarations.each do |declaration|
        declaration.items.select { |item|
          item.kind_of?(VariableDefinition)
        }.each do |definition|
          return definition if definition.identifier.value == name.value
        end
      end

      declaration = implicit_parameter_definition(name, symbol_table)
      declarations.push(declaration)
      declaration.items.first
    end

    def variable_definition_to_parameter_definition(variable_definition)
      declaration_specifiers = variable_definition.declaration_specifiers
      declarator = variable_definition.init_declarator.declarator
      pdef = ParameterDefinition.new(declaration_specifiers, declarator)

      unless declaration_specifiers
        pdef.head_token = declarator.head_token
        pdef.tail_token = declarator.tail_token
      end

      unless declarator
        pdef.head_token = declaration_specifiers.head_token
        pdef.tail_token = declaration_specifiers.tail_token
      end

      pdef
    end

    def implicit_parameter_definition(identifier, symbol_table)
      init_declarator =
        InitDeclarator.new(IdentifierDeclarator.new(identifier), nil)
      Declaration.new(nil, [init_declarator], symbol_table)
    end
  end

  class AnsiFunctionDefinition < FunctionDefinition
    def initialize(declaration_specifiers, declarator,
                   compound_statement, symbol_table)
      super(declaration_specifiers, declarator,
            create_parameters(declarator.parameter_type_list),
            compound_statement, symbol_table)
    end

    def parameter_type_list
      declarator.parameter_type_list
    end

    private
    def create_parameters(parameter_type_list)
      return [] unless parameter_type_list
      parameter_type_list.parameters.map do |parameter_declaration|
        declaration_specifiers = parameter_declaration.declaration_specifiers
        declarator = parameter_declaration.declarator
        pdef = ParameterDefinition.new(declaration_specifiers, declarator)

        unless declaration_specifiers
          pdef.head_token = declarator.head_token
          pdef.tail_token = declarator.tail_token
        end

        unless declarator
          pdef.head_token = declaration_specifiers.head_token
          pdef.tail_token = declaration_specifiers.tail_token
        end

        pdef
      end
    end
  end

  class ParameterDefinition < Definition
    def initialize(declaration_specifiers, declarator)
      super(declaration_specifiers)

      @declarator = declarator
    end

    attr_reader :declarator

    def identifier
      if @declarator
        if @declarator.abstract?
          nil
        else
          @declarator.identifier
        end
      else
        nil
      end
    end

    def location
      identifier ? identifier.location : nil
    end

    def to_variable_definition
      PseudoVariableDefinition.new(declaration_specifiers,
                                   InitDeclarator.new(@declarator, nil),
                                   type)
    end

    def inspect(indent = 0)
      " " * indent + "#{short_class_name} (#{location.inspect}) " +
        (storage_class_specifier ? storage_class_specifier.inspect : "") +
        (identifier ? identifier.value : "")
    end
  end

  class PseudoVariableDefinition < VariableDefinition
    def initialize(declaration_specifiers, init_declarator, type)
      super(declaration_specifiers, init_declarator, nil)

      self.type = type
    end

    def mark_as_referred_by(token) end
  end

  class TypeName < SyntaxNode
    def initialize(specifier_qualifier_list, abstract_declarator, symbol_table)
      super()
      @specifier_qualifier_list = specifier_qualifier_list
      @abstract_declarator = abstract_declarator
      @type = nil
      @type_declaration = build_type_declaration(specifier_qualifier_list,
                                                 symbol_table)
    end

    attr_reader :specifier_qualifier_list
    attr_reader :abstract_declarator
    attr_accessor :type
    attr_reader :type_declaration

    def location
      @specifier_qualifier_list.location
    end

    def to_s
      @type.image
    end

    def inspect(indent = 0)
      " " * indent + short_class_name + " (#{@type ? @type.image : "nil"})"
    end

    private
    def build_type_declaration(specifier_qualifier_list, symbol_table)
      specifier_qualifier_list.type_specifiers.each do |type_specifier|
        visitor = TypeDeclarationBuilder.new(symbol_table)
        type_specifier.accept(visitor)
        unless visitor.type_declarations.empty?
          return visitor.type_declarations.first
        end
      end
      nil
    end
  end

  class AbstractDeclarator < Declarator
    def identifier
      nil
    end

    def abstract?
      true
    end
  end

  class PointerAbstractDeclarator < AbstractDeclarator
    def initialize(abstract_declarator, pointer)
      super()
      @base = abstract_declarator
      @pointer = pointer
    end

    attr_reader :base

    def location
      head_location
    end

    def function?(stack = [])
      stack.push(:pointer)
      @base.function?(stack) if @base
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class GroupedAbstractDeclarator < AbstractDeclarator
    def initialize(abstract_declarator)
      super()
      @base = abstract_declarator
    end

    attr_reader :base

    def location
      @base.location
    end

    def function?
      @base.function?
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class ArrayAbstractDeclarator < AbstractDeclarator
    def initialize(abstract_declarator, size_expression)
      super()
      @base = abstract_declarator
      @size_expression = size_expression
    end

    attr_reader :base
    attr_reader :size_expression

    def location
      return @base.location if @base
      return @size_expression.location if @size_expression
      nil
    end

    def function?(stack = [])
      stack.push(:array)
      @base.function?(stack) if @base
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class FunctionAbstractDeclarator < AbstractDeclarator
    def initialize(abstract_declarator, parameter_type_list)
      super()
      @base = abstract_declarator
      @parameter_type_list = parameter_type_list
    end

    attr_reader :base
    attr_reader :parameter_type_list

    def location
      return @base.location if @base
      return @parameter_type_list.location if @parameter_type_list
      nil
    end

    def function?(stack = [])
      stack.push(:function)
      @base.function?(stack) if @base
      stack.last == :function
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class Initializer < SyntaxNode
    def initialize(expression, initializers)
      super()
      @expression = expression
      @initializers = initializers
    end

    attr_reader :expression
    attr_reader :initializers

    def location
      head_location
    end

    def to_s
      case
      when @expression
        @expression.to_s
      when @initializers
        "{#{@initializers.map { |ini| ini.to_s }.join(",")}}"
      else
        "{}"
      end
    end

    def inspect(indent = 0)
      " " * indent + short_class_name
    end
  end

  class FunctionSignature
    def initialize(name, type)
      @name = name
      @type = type
    end

    attr_reader :name

    def ==(rhs)
      if @type.parameter_types.empty? || rhs.type.parameter_types.empty?
        @name == rhs.name
      else
        @name == rhs.name && @type == rhs.type
      end
    end

    def to_s
      "#{@type.return_type.brief_image} #{@name}(" +
        @type.parameter_types.map { |t| t.brief_image }.join(",") +
        (@type.have_va_list? ? ",...)" : ")")
    end

    protected
    attr_reader :type
  end

  class TypeDeclarationBuilder
    def initialize(symbol_table)
      @symbol_table = symbol_table
      @type_declarations = []
    end

    attr_reader :type_declarations

    def visit_standard_type_specifier(node)
    end

    def visit_typedef_type_specifier(node)
    end

    def visit_struct_specifier(node)
      if node.struct_declarations
        node.struct_declarations.each { |child| child.accept(self) }
        symbol = @symbol_table.create_new_symbol(StructTag, node.identifier)
        @type_declarations.push(StructTypeDeclaration.new(node, symbol))
      end
    end

    def visit_union_specifier(node)
      if node.struct_declarations
        node.struct_declarations.each { |child| child.accept(self) }
        symbol = @symbol_table.create_new_symbol(UnionTag, node.identifier)
        @type_declarations.push(UnionTypeDeclaration.new(node, symbol))
      end
    end

    def visit_enum_specifier(node)
      if node.enumerators
        symbol = @symbol_table.create_new_symbol(EnumTag, node.identifier)
        @type_declarations.push(EnumTypeDeclaration.new(node, symbol))
      end
    end

    def visit_typeof_type_specifier(node)
    end

    def visit_struct_declaration(node)
      node.specifier_qualifier_list.accept(self)
    end

    def visit_specifier_qualifier_list(node)
      node.type_specifiers.each { |child| child.accept(self) }
    end
  end

  class SyntaxTreeVisitor
    def visit_error_expression(node)
    end

    def visit_object_specifier(node)
    end

    def visit_constant_specifier(node)
    end

    def visit_string_literal_specifier(node)
    end

    def visit_null_constant_specifier(node)
    end

    def visit_grouped_expression(node)
      node.expression.accept(self)
    end

    def visit_array_subscript_expression(node)
      node.expression.accept(self)
      node.array_subscript.accept(self)
    end

    def visit_function_call_expression(node)
      node.expression.accept(self)
      node.argument_expressions.each { |expr| expr.accept(self) }
    end

    def visit_member_access_by_value_expression(node)
      node.expression.accept(self)
    end

    def visit_member_access_by_pointer_expression(node)
      node.expression.accept(self)
    end

    def visit_bit_access_by_value_expression(node)
      node.expression.accept(self)
    end

    def visit_bit_access_by_pointer_expression(node)
      node.expression.accept(self)
    end

    def visit_postfix_increment_expression(node)
      node.operand.accept(self)
    end

    def visit_postfix_decrement_expression(node)
      node.operand.accept(self)
    end

    def visit_compound_literal_expression(node)
      node.type_name.accept(self) if node.type_name
      node.initializers.each { |ini| ini.accept(self) }
    end

    def visit_prefix_increment_expression(node)
      node.operand.accept(self)
    end

    def visit_prefix_decrement_expression(node)
      node.operand.accept(self)
    end

    def visit_address_expression(node)
      node.operand.accept(self)
    end

    def visit_indirection_expression(node)
      node.operand.accept(self)
    end

    def visit_unary_arithmetic_expression(node)
      node.operand.accept(self)
    end

    def visit_sizeof_expression(node)
      node.operand.accept(self)
    end

    def visit_sizeof_type_expression(node)
      node.operand.accept(self)
    end

    def visit_alignof_expression(node)
      node.operand.accept(self)
    end

    def visit_alignof_type_expression(node)
      node.operand.accept(self)
    end

    def visit_cast_expression(node)
      node.type_name.accept(self)
      node.operand.accept(self)
    end

    def visit_multiplicative_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_additive_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_shift_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_relational_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_equality_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_and_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_exclusive_or_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_inclusive_or_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_logical_and_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_logical_or_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_conditional_expression(node)
      node.condition.accept(self)
      node.then_expression.accept(self)
      node.else_expression.accept(self)
    end

    def visit_simple_assignment_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_compound_assignment_expression(node)
      node.lhs_operand.accept(self)
      node.rhs_operand.accept(self)
    end

    def visit_comma_separated_expression(node)
      node.expressions.each { |expr| expr.accept(self) }
    end

    def visit_declaration(node)
      node.declaration_specifiers.accept(self) if node.declaration_specifiers
      node.init_declarators.each { |decl| decl.accept(self) }
      node.items.each { |item| item.accept(self) }
    end

    def visit_function_declaration(node)
    end

    def visit_variable_declaration(node)
    end

    def visit_variable_definition(node)
    end

    def visit_typedef_declaration(node)
    end

    def visit_struct_type_declaration(node)
    end

    def visit_union_type_declaration(node)
    end

    def visit_enum_type_declaration(node)
    end

    def visit_declaration_specifiers(node)
      node.type_specifiers.each { |ts| ts.accept(self) }
    end

    def visit_init_declarator(node)
      node.declarator.accept(self)
      node.initializer.accept(self) if node.initializer
    end

    def visit_standard_type_specifier(node)
    end

    def visit_typedef_type_specifier(node)
    end

    def visit_struct_specifier(node)
      if node.struct_declarations
        node.struct_declarations.each { |decl| decl.accept(self) }
      end
    end

    def visit_union_specifier(node)
      if node.struct_declarations
        node.struct_declarations.each { |decl| decl.accept(self) }
      end
    end

    def visit_struct_declaration(node)
      node.specifier_qualifier_list.accept(self)
      node.struct_declarators.each { |decl| decl.accept(self) }
      node.items.each { |item| item.accept(self) }
    end

    def visit_member_declaration(node)
    end

    def visit_specifier_qualifier_list(node)
      node.type_specifiers.each { |ts| ts.accept(self) }
    end

    def visit_struct_declarator(node)
      node.declarator.accept(self) if node.declarator
      node.expression.accept(self) if node.expression
    end

    def visit_enum_specifier(node)
      if node.enumerators
        node.enumerators.each { |enum| enum.accept(self) }
      end
    end

    def visit_enumerator(node)
      node.expression.accept(self) if node.expression
    end

    def visit_typeof_type_specifier(node)
      node.expression.accept(self) if node.expression
      node.type_name.accept(self) if node.type_name
    end

    def visit_identifier_declarator(node)
    end

    def visit_grouped_declarator(node)
      node.base.accept(self)
    end

    def visit_array_declarator(node)
      node.base.accept(self)
      node.size_expression.accept(self) if node.size_expression
    end

    def visit_ansi_function_declarator(node)
      node.base.accept(self)
      node.parameter_type_list.accept(self)
    end

    def visit_kandr_function_declarator(node)
      node.base.accept(self)
    end

    def visit_abbreviated_function_declarator(node)
      node.base.accept(self)
    end

    def visit_parameter_type_list(node)
      node.parameters.each { |param| param.accept(self) }
    end

    def visit_parameter_declaration(node)
      node.declaration_specifiers.accept(self)
      node.declarator.accept(self) if node.declarator
    end

    def visit_error_statement(node)
    end

    def visit_generic_labeled_statement(node)
      node.statement.accept(self)
    end

    def visit_case_labeled_statement(node)
      node.expression.accept(self)
      node.statement.accept(self)
    end

    def visit_default_labeled_statement(node)
      node.statement.accept(self)
    end

    def visit_compound_statement(node)
      node.block_items.each { |item| item.accept(self) }
    end

    def visit_expression_statement(node)
      node.expression.accept(self) if node.expression
    end

    def visit_if_statement(node)
      node.expression.accept(self)
      node.statement.accept(self)
    end

    def visit_if_else_statement(node)
      node.expression.accept(self)
      node.then_statement.accept(self)
      node.else_statement.accept(self)
    end

    def visit_switch_statement(node)
      node.expression.accept(self)
      node.statement.accept(self)
    end

    def visit_while_statement(node)
      node.expression.accept(self)
      node.statement.accept(self)
    end

    def visit_do_statement(node)
      node.statement.accept(self)
      node.expression.accept(self)
    end

    def visit_for_statement(node)
      node.initial_statement.accept(self)
      node.condition_statement.accept(self)
      node.expression.accept(self) if node.expression
      node.body_statement.accept(self)
    end

    def visit_c99_for_statement(node)
      node.declaration.accept(self)
      node.condition_statement.accept(self)
      node.expression.accept(self) if node.expression
      node.body_statement.accept(self)
    end

    def visit_goto_statement(node)
    end

    def visit_continue_statement(node)
    end

    def visit_break_statement(node)
    end

    def visit_return_statement(node)
      node.expression.accept(self) if node.expression
    end

    def visit_translation_unit(node)
      node.external_declarations.each { |decl| decl.accept(self) }
    end

    def visit_kandr_function_definition(node)
      node.declaration_specifiers.accept(self) if node.declaration_specifiers
      node.declarator.accept(self)
      node.parameter_definitions.each { |pdef| pdef.accept(self) }
      node.function_body.accept(self)
      node.type_declaration.accept(self) if node.type_declaration
    end

    def visit_ansi_function_definition(node)
      node.declaration_specifiers.accept(self) if node.declaration_specifiers
      node.declarator.accept(self)
      node.parameter_definitions.each { |pdef| pdef.accept(self) }
      node.function_body.accept(self)
      node.type_declaration.accept(self) if node.type_declaration
    end

    def visit_parameter_definition(node)
    end

    def visit_type_name(node)
      node.specifier_qualifier_list.accept(self)
      node.abstract_declarator.accept(self) if node.abstract_declarator
      node.type_declaration.accept(self) if node.type_declaration
    end

    def visit_pointer_abstract_declarator(node)
      node.base.accept(self) if node.base
    end

    def visit_grouped_abstract_declarator(node)
      node.base.accept(self)
    end

    def visit_array_abstract_declarator(node)
      node.base.accept(self) if node.base
      node.size_expression.accept(self) if node.size_expression
    end

    def visit_function_abstract_declarator(node)
      node.base.accept(self) if node.base
      node.parameter_type_list.accept(self) if node.parameter_type_list
    end

    def visit_initializer(node)
      case
      when node.expression
        node.expression.accept(self)
      when node.initializers
        node.initializers.each { |ini| ini.accept(self) }
      end
    end
  end

  class SyntaxTreeMulticastVisitor < SyntaxTreeVisitor
    extend Pluggable

    def_plugin :enter_error_expression
    def_plugin :leave_error_expression
    def_plugin :enter_object_specifier
    def_plugin :leave_object_specifier
    def_plugin :enter_constant_specifier
    def_plugin :leave_constant_specifier
    def_plugin :enter_string_literal_specifier
    def_plugin :leave_string_literal_specifier
    def_plugin :enter_null_constant_specifier
    def_plugin :leave_null_constant_specifier
    def_plugin :enter_grouped_expression
    def_plugin :leave_grouped_expression
    def_plugin :enter_array_subscript_expression
    def_plugin :leave_array_subscript_expression
    def_plugin :enter_function_call_expression
    def_plugin :leave_function_call_expression
    def_plugin :enter_member_access_by_value_expression
    def_plugin :leave_member_access_by_value_expression
    def_plugin :enter_member_access_by_pointer_expression
    def_plugin :leave_member_access_by_pointer_expression
    def_plugin :enter_bit_access_by_value_expression
    def_plugin :leave_bit_access_by_value_expression
    def_plugin :enter_bit_access_by_pointer_expression
    def_plugin :leave_bit_access_by_pointer_expression
    def_plugin :enter_postfix_increment_expression
    def_plugin :leave_postfix_increment_expression
    def_plugin :enter_postfix_decrement_expression
    def_plugin :leave_postfix_decrement_expression
    def_plugin :enter_compound_literal_expression
    def_plugin :leave_compound_literal_expression
    def_plugin :enter_prefix_increment_expression
    def_plugin :leave_prefix_increment_expression
    def_plugin :enter_prefix_decrement_expression
    def_plugin :leave_prefix_decrement_expression
    def_plugin :enter_address_expression
    def_plugin :leave_address_expression
    def_plugin :enter_indirection_expression
    def_plugin :leave_indirection_expression
    def_plugin :enter_unary_arithmetic_expression
    def_plugin :leave_unary_arithmetic_expression
    def_plugin :enter_sizeof_expression
    def_plugin :leave_sizeof_expression
    def_plugin :enter_sizeof_type_expression
    def_plugin :leave_sizeof_type_expression
    def_plugin :enter_alignof_expression
    def_plugin :leave_alignof_expression
    def_plugin :enter_alignof_type_expression
    def_plugin :leave_alignof_type_expression
    def_plugin :enter_cast_expression
    def_plugin :leave_cast_expression
    def_plugin :enter_multiplicative_expression
    def_plugin :leave_multiplicative_expression
    def_plugin :enter_additive_expression
    def_plugin :leave_additive_expression
    def_plugin :enter_shift_expression
    def_plugin :leave_shift_expression
    def_plugin :enter_relational_expression
    def_plugin :leave_relational_expression
    def_plugin :enter_equality_expression
    def_plugin :leave_equality_expression
    def_plugin :enter_and_expression
    def_plugin :leave_and_expression
    def_plugin :enter_exclusive_or_expression
    def_plugin :leave_exclusive_or_expression
    def_plugin :enter_inclusive_or_expression
    def_plugin :leave_inclusive_or_expression
    def_plugin :enter_logical_and_expression
    def_plugin :leave_logical_and_expression
    def_plugin :enter_logical_or_expression
    def_plugin :leave_logical_or_expression
    def_plugin :enter_conditional_expression
    def_plugin :leave_conditional_expression
    def_plugin :enter_simple_assignment_expression
    def_plugin :leave_simple_assignment_expression
    def_plugin :enter_compound_assignment_expression
    def_plugin :leave_compound_assignment_expression
    def_plugin :enter_comma_separated_expression
    def_plugin :leave_comma_separated_expression
    def_plugin :enter_declaration
    def_plugin :leave_declaration
    def_plugin :enter_function_declaration
    def_plugin :leave_function_declaration
    def_plugin :enter_variable_declaration
    def_plugin :leave_variable_declaration
    def_plugin :enter_variable_definition
    def_plugin :leave_variable_definition
    def_plugin :enter_typedef_declaration
    def_plugin :leave_typedef_declaration
    def_plugin :enter_struct_type_declaration
    def_plugin :leave_struct_type_declaration
    def_plugin :enter_union_type_declaration
    def_plugin :leave_union_type_declaration
    def_plugin :enter_enum_type_declaration
    def_plugin :leave_enum_type_declaration
    def_plugin :enter_declaration_specifiers
    def_plugin :leave_declaration_specifiers
    def_plugin :enter_init_declarator
    def_plugin :leave_init_declarator
    def_plugin :enter_standard_type_specifier
    def_plugin :leave_standard_type_specifier
    def_plugin :enter_typedef_type_specifier
    def_plugin :leave_typedef_type_specifier
    def_plugin :enter_struct_specifier
    def_plugin :leave_struct_specifier
    def_plugin :enter_union_specifier
    def_plugin :leave_union_specifier
    def_plugin :enter_struct_declaration
    def_plugin :leave_struct_declaration
    def_plugin :enter_member_declaration
    def_plugin :leave_member_declaration
    def_plugin :enter_specifier_qualifier_list
    def_plugin :leave_specifier_qualifier_list
    def_plugin :enter_struct_declarator
    def_plugin :leave_struct_declarator
    def_plugin :enter_enum_specifier
    def_plugin :leave_enum_specifier
    def_plugin :enter_enumerator
    def_plugin :leave_enumerator
    def_plugin :enter_typeof_type_specifier
    def_plugin :leave_typeof_type_specifier
    def_plugin :enter_identifier_declarator
    def_plugin :leave_identifier_declarator
    def_plugin :enter_grouped_declarator
    def_plugin :leave_grouped_declarator
    def_plugin :enter_array_declarator
    def_plugin :leave_array_declarator
    def_plugin :enter_ansi_function_declarator
    def_plugin :leave_ansi_function_declarator
    def_plugin :enter_kandr_function_declarator
    def_plugin :leave_kandr_function_declarator
    def_plugin :enter_abbreviated_function_declarator
    def_plugin :leave_abbreviated_function_declarator
    def_plugin :enter_parameter_type_list
    def_plugin :leave_parameter_type_list
    def_plugin :enter_parameter_declaration
    def_plugin :leave_parameter_declaration
    def_plugin :enter_error_statement
    def_plugin :leave_error_statement
    def_plugin :enter_generic_labeled_statement
    def_plugin :leave_generic_labeled_statement
    def_plugin :enter_case_labeled_statement
    def_plugin :leave_case_labeled_statement
    def_plugin :enter_default_labeled_statement
    def_plugin :leave_default_labeled_statement
    def_plugin :enter_compound_statement
    def_plugin :leave_compound_statement
    def_plugin :enter_expression_statement
    def_plugin :leave_expression_statement
    def_plugin :enter_if_statement
    def_plugin :leave_if_statement
    def_plugin :enter_if_else_statement
    def_plugin :leave_if_else_statement
    def_plugin :enter_switch_statement
    def_plugin :leave_switch_statement
    def_plugin :enter_while_statement
    def_plugin :leave_while_statement
    def_plugin :enter_do_statement
    def_plugin :leave_do_statement
    def_plugin :enter_for_statement
    def_plugin :leave_for_statement
    def_plugin :enter_c99_for_statement
    def_plugin :leave_c99_for_statement
    def_plugin :enter_goto_statement
    def_plugin :leave_goto_statement
    def_plugin :enter_continue_statement
    def_plugin :leave_continue_statement
    def_plugin :enter_break_statement
    def_plugin :leave_break_statement
    def_plugin :enter_return_statement
    def_plugin :leave_return_statement
    def_plugin :enter_translation_unit
    def_plugin :leave_translation_unit
    def_plugin :enter_kandr_function_definition
    def_plugin :leave_kandr_function_definition
    def_plugin :enter_ansi_function_definition
    def_plugin :leave_ansi_function_definition
    def_plugin :enter_parameter_definition
    def_plugin :leave_parameter_definition
    def_plugin :enter_type_name
    def_plugin :leave_type_name
    def_plugin :enter_pointer_abstract_declarator
    def_plugin :leave_pointer_abstract_declarator
    def_plugin :enter_grouped_abstract_declarator
    def_plugin :leave_grouped_abstract_declarator
    def_plugin :enter_array_abstract_declarator
    def_plugin :leave_array_abstract_declarator
    def_plugin :enter_function_abstract_declarator
    def_plugin :leave_function_abstract_declarator
    def_plugin :enter_initializer
    def_plugin :leave_initializer

    def visit_error_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_object_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_constant_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_string_literal_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_null_constant_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_grouped_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_array_subscript_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_function_call_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_member_access_by_value_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_member_access_by_pointer_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_bit_access_by_value_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_bit_access_by_pointer_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_postfix_increment_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_postfix_decrement_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_compound_literal_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_prefix_increment_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_prefix_decrement_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_address_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_indirection_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_unary_arithmetic_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_sizeof_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_sizeof_type_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_alignof_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_alignof_type_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_cast_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_multiplicative_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_additive_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_shift_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_relational_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_equality_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_and_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_exclusive_or_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_inclusive_or_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_logical_and_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_logical_or_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_conditional_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_simple_assignment_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_compound_assignment_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_comma_separated_expression(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_function_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_variable_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_variable_definition(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_typedef_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_struct_type_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_union_type_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_enum_type_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_declaration_specifiers(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_init_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_standard_type_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_typedef_type_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_struct_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_union_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_struct_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_member_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_specifier_qualifier_list(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_struct_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_enum_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_enumerator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_typeof_type_specifier(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_identifier_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_grouped_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_array_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_ansi_function_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_kandr_function_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_abbreviated_function_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_parameter_type_list(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_parameter_declaration(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_error_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_generic_labeled_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_case_labeled_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_default_labeled_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_compound_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_expression_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_if_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_if_else_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_switch_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_while_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_do_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_for_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_c99_for_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_goto_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_continue_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_break_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_return_statement(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_translation_unit(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_kandr_function_definition(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_ansi_function_definition(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_parameter_definition(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_type_name(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_pointer_abstract_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_grouped_abstract_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_array_abstract_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_function_abstract_declarator(node)
      visit_with_notifying(__method__, node) { super }
    end

    def visit_initializer(node)
      visit_with_notifying(__method__, node) { super }
    end

    private
    def visit_with_notifying(caller_method, node, &block)
      suffix = caller_method.to_s.sub(/\Avisit_/, "")
      __send__("enter_#{suffix}").invoke(node)
      yield
      __send__("leave_#{suffix}").invoke(node)
    end
  end

  class ObjectSpecifierCollector < SyntaxTreeVisitor
    def initialize
      @object_specifiers = []
    end

    attr_reader :object_specifiers

    def visit_object_specifier(node)
      super
      @object_specifiers.push(node)
    end
  end

  class IdentifierDeclaratorCollector < SyntaxTreeVisitor
    def initialize
      @identifier_declarators = []
    end

    attr_reader :identifier_declarators

    def visit_identifier_declarator(node)
      super
      @identifier_declarators.push(node)
    end
  end

  class TypedefTypeSpecifierCollector < SyntaxTreeVisitor
    def initialize
      @typedef_type_specifiers = []
    end

    attr_reader :typedef_type_specifiers

    def visit_variable_definition(node)
    end

    def visit_typedef_type_specifier(node)
      super
      @typedef_type_specifiers.push(node)
    end
  end

  class FunctionDeclaratorCollector < SyntaxTreeVisitor
    def initialize
      @function_declarators = []
    end

    attr_reader :function_declarators

    def visit_ansi_function_declarator(node)
      @function_declarators.push(node)
      super
    end

    def visit_kandr_function_declarator(node)
      @function_declarators.push(node)
      super
    end
  end

  class SimpleAssignmentExpressionCollector < SyntaxTreeVisitor
    def initialize
      @simple_assignment_expressions = []
    end

    attr_reader :simple_assignment_expressions

    def visit_simple_assignment_expression(node)
      super
      @simple_assignment_expressions.push(node)
    end
  end

  class CompoundAssignmentExpressionCollector < SyntaxTreeVisitor
    def initialize
      @compound_assignment_expressions = []
    end

    attr_reader :compound_assignment_expressions

    def visit_compound_assignment_expression(node)
      super
      @compound_assignment_expressions.push(node)
    end
  end

  class PrefixIncrementExpressionCollector < SyntaxTreeVisitor
    def initialize
      @prefix_increment_expressions = []
    end

    attr_reader :prefix_increment_expressions

    def visit_prefix_increment_expression(node)
      super
      @prefix_increment_expressions.push(node)
    end
  end

  class PrefixDecrementExpressionCollector < SyntaxTreeVisitor
    def initialize
      @prefix_decrement_expressions = []
    end

    attr_reader :prefix_decrement_expressions

    def visit_prefix_decrement_expression(node)
      super
      @prefix_decrement_expressions.push(node)
    end
  end

  class PostfixIncrementExpressionCollector < SyntaxTreeVisitor
    def initialize
      @postfix_increment_expressions = []
    end

    attr_reader :postfix_increment_expressions

    def visit_postfix_increment_expression(node)
      super
      @postfix_increment_expressions.push(node)
    end
  end

  class PostfixDecrementExpressionCollector < SyntaxTreeVisitor
    def initialize
      @postfix_decrement_expressions = []
    end

    attr_reader :postfix_decrement_expressions

    def visit_postfix_decrement_expression(node)
      super
      @postfix_decrement_expressions.push(node)
    end
  end

  class AdditiveExpressionCollector < SyntaxTreeVisitor
    def initialize
      @additive_expressions = []
    end

    attr_reader :additive_expressions

    def visit_additive_expression(node)
      super
      @additive_expressions.push(node)
    end
  end

  class GenericLabeledStatementCollector < SyntaxTreeVisitor
    def initialize
      @generic_labeled_statements = []
    end

    attr_reader :generic_labeled_statements

    def visit_generic_labeled_statement(node)
      super
      @generic_labeled_statements.push(node)
    end
  end

  class IfStatementCollector < SyntaxTreeVisitor
    def initialize
      @if_statements = []
    end

    attr_reader :if_statements

    def visit_if_statement(node)
      super
      @if_statements.push(node)
    end
  end

  class IfElseStatementCollector < SyntaxTreeVisitor
    def initialize
      @if_else_statements = []
    end

    attr_reader :if_else_statements

    def visit_if_else_statement(node)
      super
      @if_else_statements.push(node)
    end
  end

  class GotoStatementCollector < SyntaxTreeVisitor
    def initialize
      @goto_statements = []
    end

    attr_reader :goto_statements

    def visit_goto_statement(node)
      @goto_statements.push(node)
    end
  end

  class ArrayDeclaratorCollector < SyntaxTreeVisitor
    def initialize
      @array_declarators = []
    end

    attr_reader :array_declarators

    def visit_array_declarator(node)
      @array_declarators.push(node)
      super
    end
  end

  class ConstantSpecifierCollector < SyntaxTreeVisitor
    def initialize
      @constant_specifiers = []
    end

    attr_reader :constant_specifiers

    def visit_constant_specifier(node)
      @constant_specifiers.push(node)
    end
  end

  class ExpressionExtractor < SyntaxTreeVisitor
    def initialize
      @expressions = []
    end

    attr_reader :expressions

    def push_expression(expression)
      @expressions.push(expression)
    end

    alias :visit_grouped_expression :push_expression
    alias :visit_array_subscript_expression :push_expression
    alias :visit_function_call_expression :push_expression
    alias :visit_member_access_by_value_expression :push_expression
    alias :visit_member_access_by_pointer_expression :push_expression
    alias :visit_bit_access_by_value_expression :push_expression
    alias :visit_bit_access_by_pointer_expression :push_expression
    alias :visit_postfix_increment_expression :push_expression
    alias :visit_postfix_decrement_expression :push_expression
    alias :visit_compound_literal_expression :push_expression
    alias :visit_prefix_increment_expression :push_expression
    alias :visit_prefix_decrement_expression :push_expression
    alias :visit_address_expression :push_expression
    alias :visit_indirection_expression :push_expression
    alias :visit_unary_arithmetic_expression :push_expression
    alias :visit_sizeof_expression :push_expression
    alias :visit_alignof_expression :push_expression
    alias :visit_cast_expression :push_expression
    alias :visit_multiplicative_expression :push_expression
    alias :visit_additive_expression :push_expression
    alias :visit_shift_expression :push_expression
    alias :visit_relational_expression :push_expression
    alias :visit_equality_expression :push_expression
    alias :visit_and_expression :push_expression
    alias :visit_exclusive_or_expression :push_expression
    alias :visit_inclusive_or_expression :push_expression
    alias :visit_logical_and_expression :push_expression
    alias :visit_logical_or_expression :push_expression
    alias :visit_conditional_expression :push_expression
    alias :visit_simple_assignment_expression :push_expression
    alias :visit_compound_assignment_expression :push_expression

    private :push_expression
  end

  class ConditionalExpressionExtractor < SyntaxTreeVisitor
    def initialize
      @expressions = []
    end

    attr_reader :expressions

    def visit_conditional_expression(node)
      @expressions.push(node)
    end
  end

end
end
