# Quality metric measurement classes for C language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/metric"

module AdLint #:nodoc:
module C #:nodoc:

  class FL_STMT < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_error_statement += method(:count_statement)
      visitor.enter_generic_labeled_statement += method(:count_statement)
      visitor.enter_case_labeled_statement += method(:count_statement)
      visitor.enter_default_labeled_statement += method(:count_statement)
      visitor.enter_expression_statement += method(:count_statement)
      visitor.enter_if_statement += method(:count_statement)
      visitor.enter_if_else_statement += method(:count_statement)
      visitor.enter_switch_statement += method(:count_statement)
      visitor.enter_while_statement += method(:count_statement)
      visitor.enter_do_statement += method(:count_statement)
      visitor.enter_for_statement += method(:count_statement)
      visitor.enter_c99_for_statement += method(:count_statement)
      visitor.enter_goto_statement += method(:count_statement)
      visitor.enter_continue_statement += method(:count_statement)
      visitor.enter_break_statement += method(:count_statement)
      visitor.enter_return_statement += method(:count_statement)
      visitor.leave_translation_unit += method(:measure)
      @statement_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def count_statement(statement)
      if @fpath == statement.location.fpath
        @statement_count += 1
      end
    end

    def measure(node)
      FL_STMT(@fpath, @statement_count)
    end
  end

  class FL_FUNC < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:count_function)
      visitor.enter_kandr_function_definition += method(:count_function)
      visitor.leave_translation_unit += method(:measure)
      @function_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def count_function(function_definition)
      if @fpath == function_definition.location.fpath
        @function_count += 1
      end
    end

    def measure(node)
      FL_FUNC(@fpath, @function_count)
    end
  end

  class FN_STMT < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_error_statement += method(:count_statement)
      visitor.enter_generic_labeled_statement += method(:count_statement)
      visitor.enter_case_labeled_statement += method(:count_statement)
      visitor.enter_default_labeled_statement += method(:count_statement)
      visitor.enter_expression_statement += method(:count_statement)
      visitor.enter_if_statement += method(:count_statement)
      visitor.enter_if_else_statement += method(:count_statement)
      visitor.enter_switch_statement += method(:count_statement)
      visitor.enter_while_statement += method(:count_statement)
      visitor.enter_do_statement += method(:count_statement)
      visitor.enter_for_statement += method(:count_statement)
      visitor.enter_c99_for_statement += method(:count_statement)
      visitor.enter_goto_statement += method(:count_statement)
      visitor.enter_continue_statement += method(:count_statement)
      visitor.enter_break_statement += method(:count_statement)
      visitor.enter_return_statement += method(:count_statement)
      @current_function = nil
      @statement_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @statement_count = 0
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_STMT(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @statement_count)

      @current_function = nil
      @statement_count = 0
    end

    def count_statement(statement)
      if @current_function
        @statement_count += 1
      end
    end
  end

  class FN_UNRC < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_error_statement += method(:count_statement)
      visitor.enter_generic_labeled_statement += method(:count_statement)
      visitor.enter_case_labeled_statement += method(:count_statement)
      visitor.enter_default_labeled_statement += method(:count_statement)
      visitor.enter_expression_statement += method(:count_statement)
      visitor.enter_if_statement += method(:count_statement)
      visitor.enter_if_else_statement += method(:count_statement)
      visitor.enter_switch_statement += method(:count_statement)
      visitor.enter_while_statement += method(:count_statement)
      visitor.enter_do_statement += method(:count_statement)
      visitor.enter_for_statement += method(:count_statement)
      visitor.enter_c99_for_statement += method(:count_statement)
      visitor.enter_goto_statement += method(:count_statement)
      visitor.enter_continue_statement += method(:count_statement)
      visitor.enter_break_statement += method(:count_statement)
      visitor.enter_return_statement += method(:count_statement)
      @current_function = nil
      @unreached_statement_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @unreached_statement_count = 0
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_UNRC(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @unreached_statement_count)

      @current_function = nil
      @unreached_statement_count = 0
    end

    def count_statement(statement)
      return unless @current_function

      unless statement.executed?
        @unreached_statement_count += 1
      end
    end
  end

  class FN_LINE < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:measure)
      visitor.enter_kandr_function_definition += method(:measure)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def measure(function_definition)
      if @fpath == function_definition.location.fpath
        FN_LINE(FunctionIdentifier.new(function_definition.identifier.value,
                                       function_definition.signature.to_s),
                function_definition.location, function_definition.lines)
      end
    end
  end

  class FN_PARA < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:measure_ansi_func)
      visitor.enter_kandr_function_definition += method(:measure_kandr_func)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def measure_ansi_func(function_definition)
      if @fpath == function_definition.location.fpath
        # TODO: Determine how many parameters if function has va_list.
        if function_definition.parameter_type_list
          parameters = function_definition.parameter_definitions
          FN_PARA(FunctionIdentifier.new(function_definition.identifier.value,
                                         function_definition.signature.to_s),
                  function_definition.location,
                  parameters.count { |param| !param.type.void? })
        else
          # TODO: Determine how many parameters if signature is abbreviated.
          FN_PARA(FunctionIdentifier.new(function_definition.identifier.value,
                                         function_definition.signature.to_s),
                  function_definition.location, 0)
        end
      end
    end

    def measure_kandr_func(function_definition)
      if @fpath == function_definition.location.fpath
        FN_PARA(FunctionIdentifier.new(function_definition.identifier.value,
                                       function_definition.signature.to_s),
                function_definition.location,
                function_definition.identifier_list.size)
      end
    end
  end

  class FN_UNUV < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      interp = context[:c_interpreter]
      interp.on_function_started += method(:enter_function)
      interp.on_function_ended += method(:leave_function)
      interp.on_variable_defined += method(:define_variable)
      interp.on_parameter_defined += method(:define_variable)
      interp.on_variable_referred += method(:refer_variable)
      interp.on_variable_value_referred += method(:read_variable)
      interp.on_variable_value_updated += method(:write_variable)
      @current_function = nil
      @variables = nil
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition, function)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @variables = {}
      end
    end

    def leave_function(function_definition, function)
      return unless @current_function

      useless_count = @variables.each_value.reduce(0) { |count, read_count|
        read_count == 0 ? count + 1 : count
      }

      FN_UNUV(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, useless_count)

      @current_function = nil
      @variables = nil
    end

    def define_variable(definition, variable)
      if @current_function
        @variables[variable.name] = 0
      end
    end

    def refer_variable(expression, variable)
      return unless @current_function

      if variable.named? && @variables[variable.name]
        @variables[variable.name] += 1
      end
    end

    def read_variable(expression, variable)
      return unless @current_function

      if variable.named? && @variables[variable.name]
        @variables[variable.name] += 1
      end
    end

    def write_variable(expression, variable)
      return unless @current_function

      if variable.named? && @variables[variable.name]
        @variables[variable.name] = 0
      end
    end
  end

  class FN_CSUB < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_function_call_expression += method(:count_function_call)
      @current_function = nil
      @function_call_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @function_call_count = 0
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_CSUB(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @function_call_count)

      @current_function = nil
      @function_call_count = 0
    end

    def count_function_call(node)
      if @current_function
        @function_call_count += 1
      end
    end
  end

  class FN_GOTO < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_goto_statement += method(:count_goto)
      @current_function = nil
      @goto_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @goto_count = 0
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_GOTO(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @goto_count)

      @current_function = nil
      @goto_count = 0
    end

    def count_goto(node)
      if @current_function
        @goto_count += 1
      end
    end
  end

  class FN_RETN < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_error_statement += method(:enter_statement)
      visitor.enter_generic_labeled_statement += method(:enter_statement)
      visitor.enter_case_labeled_statement += method(:enter_statement)
      visitor.enter_default_labeled_statement += method(:enter_statement)
      visitor.enter_expression_statement += method(:enter_statement)
      visitor.enter_if_statement += method(:enter_statement)
      visitor.enter_if_else_statement += method(:enter_statement)
      visitor.enter_switch_statement += method(:enter_statement)
      visitor.enter_while_statement += method(:enter_statement)
      visitor.enter_do_statement += method(:enter_statement)
      visitor.enter_for_statement += method(:enter_statement)
      visitor.enter_c99_for_statement += method(:enter_statement)
      visitor.enter_goto_statement += method(:enter_statement)
      visitor.enter_continue_statement += method(:enter_statement)
      visitor.enter_break_statement += method(:enter_statement)
      visitor.enter_return_statement += method(:count_return)
      @current_function = nil
      @return_count = 0
      @last_statement = nil
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @return_count = 0
        @last_statement = nil
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      if @current_function.type.return_type.void? &&
          !(@last_statement.kind_of?(ReturnStatement))
        FN_RETN(FunctionIdentifier.new(@current_function.identifier.value,
                                       @current_function.signature.to_s),
                @current_function.location, @return_count + 1)
      else
        FN_RETN(FunctionIdentifier.new(@current_function.identifier.value,
                                       @current_function.signature.to_s),
                @current_function.location, @return_count)
      end

      @current_function = nil
      @return_count = 0
      @last_statement = nil
    end

    def enter_statement(node)
      return unless @current_function

      if node.executed?
        @last_statement = node
      end
    end

    def count_return(node)
      if @current_function
        @return_count += 1
        @last_statement = node
      end
    end
  end

  class FN_UELS < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_if_else_statement += method(:enter_if_else_statement)
      visitor.leave_if_else_statement += method(:leave_if_else_statement)
      @current_function = nil
      @if_else_statement_chain = 0
      @incomplete_if_else_statement_count = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @if_else_statement_chain = 0
        @incomplete_if_else_statement_count = 0
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_UELS(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @incomplete_if_else_statement_count)

      @current_function = nil
      @if_else_statement_chain = 0
      @incomplete_if_else_statement_count = 0
    end

    def enter_if_else_statement(node)
      @if_else_statement_chain += 1

      if @current_function && @if_else_statement_chain > 0
        if node.else_statement.kind_of?(IfStatement)
          @incomplete_if_else_statement_count += 1
        end
      end
    end

    def leave_if_else_statement(node)
      @if_else_statement_chain -= 1
    end
  end

  class FN_NEST < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_compound_statement += method(:enter_block)
      visitor.leave_compound_statement += method(:leave_block)
      visitor.enter_if_statement += method(:check_statement)
      visitor.enter_if_else_statement += method(:check_statement)
      visitor.enter_while_statement += method(:check_statement)
      visitor.enter_do_statement += method(:check_statement)
      visitor.enter_for_statement += method(:check_statement)
      visitor.enter_c99_for_statement += method(:check_statement)
      @current_function = nil
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        # NOTE: Nest level of the top of the function is 0.
        #       Function definition must have a compound-statement as the
        #       function body.
        @max_nest_level = @current_nest_level = -1
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_NEST(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @max_nest_level)

      @current_function = nil
    end

    def enter_block(compound_statement)
      if @current_function
        @current_nest_level += 1
        @max_nest_level = [@max_nest_level, @current_nest_level].max
      end
    end

    def leave_block(compound_statement)
      if @current_function
        @current_nest_level -= 1
      end
    end

    def check_statement(statement)
      return unless @current_function

      case statement
      when IfStatement
        sub_statement = statement.statement
      when IfElseStatement
        if statement.then_statement.kind_of?(CompoundStatement)
          sub_statement = statement.else_statement
        else
          sub_statement = statement.then_statement
        end
      when WhileStatement, DoStatement
        sub_statement = statement.statement
      when ForStatement, C99ForStatement
        sub_statement = statement.body_statement
      end

      case sub_statement
      when CompoundStatement, IfStatement, IfElseStatement
      else
        @current_nest_level += 1
        @max_nest_level = [@max_nest_level, @current_nest_level].max
      end
    end
  end

  class FN_PATH < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      interp = context[:c_interpreter]
      interp.on_function_started += method(:enter_function)
      interp.on_function_ended += method(:leave_function)
      interp.on_branch_started += method(:enter_branch)
      interp.on_branch_ended += method(:leave_branch)
      @current_function = nil
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition, function)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition

        # NOTE: Number of paths in the current function.
        @paths_in_function = 1
        # NOTE: Stack of the number of paths to enter the current branch group.
        @paths_to_enter_branch_group = [@paths_in_function]
        # NOTE: Stack of the number of paths in the current branch.
        @paths_in_branch = [@paths_in_function]
        # NOTE: Stack of the number of paths in the current branch group.
        @paths_in_branch_group = [@paths_in_function]
      end
    end

    def leave_function(function_definition, function)
      return unless @current_function

      FN_PATH(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @paths_in_function)

      @current_function = nil
    end

    def enter_branch(branch)
      return unless @current_function

      # NOTE: Entering into new branch group.
      if branch.first?
        @paths_in_function -= @paths_in_branch.last
        @paths_to_enter_branch_group.push(@paths_in_branch.last)
        @paths_in_branch_group.push(0)
      end

      # NOTE: Entering into new branch.
      @paths_in_branch.push(@paths_to_enter_branch_group.last)
      @paths_in_function += @paths_to_enter_branch_group.last
    end

    def leave_branch(branch)
      return unless @current_function

      paths_in_this_branch = @paths_in_branch.pop

      # NOTE: Leaving from the current branch whose paths are not terminated.
      unless branch.break_with_return?
        @paths_in_branch_group[-1] += paths_in_this_branch
      end

      # NOTE: Leaving from the current branch group.
      if branch.final?
        paths_to_enter_this_branch_group = @paths_to_enter_branch_group.pop
        paths_in_this_branch_group = @paths_in_branch_group.pop

        @paths_in_branch[-1] = paths_in_this_branch_group

        # NOTE: The current branch group is an incomplete branch group.
        unless branch.group.complete?
          @paths_in_function += paths_to_enter_this_branch_group
          @paths_in_branch[-1] += paths_to_enter_this_branch_group
        end
      end
    end
  end

  class FN_CYCM < MetricMeasurement
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.enter_if_statement += method(:enter_selection)
      visitor.enter_if_else_statement += method(:enter_selection)
      visitor.enter_case_labeled_statement += method(:enter_selection)
      @current_function = nil
      @complexity = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def enter_function(function_definition)
      if @fpath == function_definition.location.fpath
        @current_function = function_definition
        @complexity = 0
      end
    end

    def leave_function(function_definition)
      return unless @current_function

      FN_CYCM(FunctionIdentifier.new(@current_function.identifier.value,
                                     @current_function.signature.to_s),
              @current_function.location, @complexity + 1)

      @current_function = nil
      @total_branch_count = 0
    end

    def enter_selection(if_statement)
      @complexity += 1 if @current_function
    end
  end

end
end
