#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <QtCore/QObject>
#include <QtWidgets/QTreeWidgetItem>
#include <QDebug>
#include "voicedata.h"
#include "model.h"

Model::Model(int p_poly, QObject *parent) : QObject(parent) {

  int i1, i2;

  uid = 1;
  rate = 44100;
  currentGroupIndex = 0;
  scalaBaseFreq = freqConst;
  scalaBaseNote = 0;
  scalaMode = false;
  max = 100.0;
  max_noise_step = 4;
  editStart = 0;
  editEnd = 0;
  interpolate = false;
  frame = 0;
  harmonicSplit[0] = 0;
  harmonicSplit[1] = 64;
  harmonicSplit[2] = 96;
  harmonicSplit[3] = 128;
  harmonicSplit[4] = 136;
  harmonicSplit[5] = 200;
  harmonicSplit[6] = 232;
  harmonicSplit[7] = 264;
  harmonicSplit[8] = 272;
  delta[0] = 1.5;
  delta[1] = 0.7;
  delta[2] = 1.5;
  delta[3] = 0.7;
  delta[4] = 1.0;
  delta[5] = 1.0;
  numHarmonics = harmonicSplit[MAX_SPLIT - 1];
  morphing = 0.5;
  linearLevel = 0.5;
  transitionWidth = 0.5;
  compression = 0.5;
  antialiasing = 4.0;
  poly = p_poly;
  masterVolume = 4e4;
  pitchBend = 0;
  noiseBandwidth = 0.25;
  modeToggleState = false;
  sceneToggleState = false;
  sourceSwitchState = 0;
  filterSwitchState = 0;
  filterCurveMode = false;
  groupSwitchState[0] = 0;
  groupSwitchState[1] = 0;
  voices = (voiceType *)malloc(poly * sizeof(voiceType));
  for (i1 = 0; i1 < 2; i1++) {
    buf[i1] = (double *)malloc(MAX_FRAMES * sizeof(double));
    memset(buf[i1], 0, MAX_FRAMES * sizeof(double));
  }                                     
  currentVoiceDataIndex = 0;
  for (i1 = 0; i1 < poly; i1++) {
    voiceData[i1] = new VoiceData();
    voiceData[i1]->setEditRange(0, poly);
    voiceData[i1]->setRate(rate);
    voiceData[i1]->setVoice(&voices[i1]);
    voices[i1].note = 0;
    voices[i1].velocity = 0;
    voices[i1].gate = false;
    voices[i1].active = false;
    voices[i1].samples = 1e9;
    voices[i1].cut = 20000.0;
    for (i2 = 0; i2 < MAX_SOURCES; i2++) {
      voices[i1].filterMixEnv[i2] = 0;
      voices[i1].filterMixEnvState[i2] = 0;
      voices[i1].filterMixSampleStack[i2] = 0;
    }  
    for (i2 = 0; i2 < 2; i2++) {
      voices[i1].filterModEnv[i2] = 0;
      voices[i1].filterModEnvState[i2] = 0;
      voices[i1].filterModLfo[i2] = 0;
      voices[i1].filterModLfoState[i2] = 0;
      voices[i1].filterModSpacingLfo[i2] = 0;
      voices[i1].filterModSpacingLfoState[i2] = 0;
    }
    envelopes[i1] = (envelopeType *)malloc(numHarmonics * sizeof(envelopeType));
    for (i2 = 0; i2 < numHarmonics; i2++) {
      envelopes[i1][i2].val = 0;
      envelopes[i1][i2].state = 0;
      envelopes[i1][i2].lfo = 0;
      envelopes[i1][i2].lfo_state = 0;
      envelopes[i1][i2].a = 0;
      envelopes[i1][i2].y = 0;
      envelopes[i1][i2].dy = 0;
      envelopes[i1][i2].ddy = 0;
      envelopes[i1][i2].freqSh = 0;
    }  
  }      
  bufSize = 1024;
  scaleLin[0] = 1.0;
  scaleExp[0] = 2.0;
  minScale[0] = 0.1;
  maxScale[0] = 10.0;
  scaleLin[1] = 0.000001;
  scaleExp[1] = 3.0;
  minScale[1] = 0.1;
  maxScale[1] = 10.0;
  scaleLin[2] = 0.000001;
  scaleExp[2] = 3.0;
  minScale[2] = 0.1;
  maxScale[2] = 10.0;
  scaleLin[3] = 0.000001;
  scaleExp[3] = 3.0;
  minScale[3] = 0.1;
  maxScale[3] = 10.0;
  scaleLin[4] = 0.00001;
  scaleExp[4] = 3.0;
  minScale[4] = 0.1;
  maxScale[4] = 10.0;
  scaleLin[5] = 0.001;
  scaleExp[5] = 0.0;
  minScale[5] = 0.1;
  maxScale[5] = 10.0;
  scaleLin[6] = 0.000001;
  scaleExp[6] = 3.0;
  minScale[6] = 0.1;
  maxScale[6] = 10.0;
  scaleLin[7] = 0.0001;
  scaleExp[7] = 2.0;
  minScale[7] = 0.1;
  maxScale[7] = 10.0;
  scaleLin[8] = 0.01;
  scaleExp[8] = 2.0;
  minScale[8] = 0.1;
  maxScale[8] = 10.0;
  scaleLin[9] = 0.00001;
  scaleExp[9] = 2.0;
  minScale[9] = 0.1;
  maxScale[9] = 10.0;
  for (i1 = 0; i1 < MAX_SCALES; i1++) {
    scale[i1] = 1.0;
  }
  for (i1 = 0; i1 < MAX_PARAMS; i1++) {
    harmonics[i1] = (double*)malloc(numHarmonics * sizeof(double));
    scaledHarmonics[i1] = (double*)malloc(numHarmonics * sizeof(double));
  }
  for (i1 = 0; i1 < MAX_PARAMS; i1++) {  
    for (i2 = 0; i2 < numHarmonics; i2++) {
      setHarmonic(i1, i2, 0);
    }
  }   
  synthSemaphore = new QSemaphore(poly);
  synthesisFlag = true;
  lfo = 0;
  lfoState = 1;
  lfoFreq = 12;
  lfoDepth = 0;
  dlfo = 0;
  editGroup = 0;
  clearScalaData();
}

Model::~Model() {
  
  int i;

  for (i = 0; i < MAX_PARAMS; i++) {
    free(harmonics[i]);
  }    
  for (i = 0; i < poly; i++) {
    free(envelopes[i]);
  }    
  free(voices);
  free(synthSemaphore);
}

long Model::getUID() {

  return(uid++);
}

int Model::getRate() {

  return(rate);
}

void Model::setRate(int p_rate) {

  int i1;

  rate = p_rate;
  for (i1 = 0; i1 < poly; i1++) {
    voiceData[i1]->setRate(rate);
  }
}

int Model::getBufSize() {

  return(bufSize);
}

void Model::setBufSize(int p_bufSize) {

  bufSize = p_bufSize;
}

double Model::getPitchBend() {

  return(pitchBend);
}

void Model::setPitchBend(double p_pitchBend) {

  pitchBend = p_pitchBend;
}

bool Model::doSynthesis() {

  return(synthesisFlag);
}

void Model::setSynthesis(bool on) {

  synthesisFlag = on;
}

double Model::getMax() {

  return(max);
}

void Model::setMax(double p_max) {

  max = p_max;
}

int Model::getPoly() {

  return(poly);
}

void Model::setMaxNoiseStep(int value) {

  max_noise_step = value;
}

int Model::getMaxNoiseStep() {

  return(max_noise_step);
}

double Model::getMinScale(int p_index) {

  return(minScale[p_index]);
}

double Model::getMaxScale(int p_index) {

  return(maxScale[p_index]);
}

void Model::setPoly(int p_poly) {

  int i1, i2, oldPoly;

  oldPoly = poly;
  poly = p_poly;
  for (i1 = poly; i1 < oldPoly; i1++) {
    free(envelopes[i1]);
  }
  for (i1 = oldPoly; i1 < poly; i1++) {
    envelopes[i1] = (envelopeType *)malloc(numHarmonics * sizeof(double));
    for (i2 = 0; i2 < numHarmonics; i2++) {
      envelopes[i1][i2].val = 0;
      envelopes[i1][i2].state = 0;
      envelopes[i1][i2].lfo = 0;
      envelopes[i1][i2].lfo_state = 0;
      envelopes[i1][i2].a = 0;
      envelopes[i1][i2].y = 0;
      envelopes[i1][i2].dy = 0;
      envelopes[i1][i2].ddy = 0;
      envelopes[i1][i2].freqSh = 0;
    }
  }                  
  voices = (voiceType *)realloc(voices, poly * sizeof(voiceType));
  for (i1 = 0; i1 < poly; i1++) {
    voices[i1].note = 0;
    voices[i1].velocity = 0;
    voices[i1].gate = false;
    voices[i1].active = false;
    voices[i1].samples = 0;
  }
}

voiceType* Model::getVoices() {

  return(voices);
}

envelopeType* Model::getEnvelopes(int voiceIndex) {

  return(envelopes[voiceIndex]);
}

int Model::getNumHarmonics() {

  return(numHarmonics);
}

void Model::setNumHarmonics(int p_numHarmonics) {

  int i1, i2;
  
  numHarmonics = p_numHarmonics;
  for (i1 = 0; i1 < MAX_PARAMS; i1++) {
    harmonics[i1] = (double*)realloc(harmonics[i1], numHarmonics * sizeof(double));
  }
  for (i1 = 0; i1 < MAX_PARAMS; i1++) {
    for (i2 = 0; i2 < numHarmonics; i2++) {
      setHarmonic(i1, i2, 0);
    }
  }                
}

int Model::getHarmonicSplit(int p_group) {

  return(harmonicSplit[p_group]);
}

void Model::setHarmonicSplit(int p_group, int p_harmonicSplit) {

  harmonicSplit[p_group] = p_harmonicSplit;
}

double* Model::getHarmonics(int p_index) { 

  return(harmonics[p_index]);
}

double Model::getHarmonic(int p_index, int p_num) {
 
  if ((p_num >= 0) && (p_num < numHarmonics)) {
    return(harmonics[p_index][p_num]);
  } else {
    return(0);
  }
}

double Model::getScaledHarmonic(int p_index, int p_num) {
 
  if ((p_num >= 0) && (p_num < numHarmonics)) {
    return(scaledHarmonics[p_index][p_num]);
  } else {
    return(0);
  }
}

int Model::calcEditGroup(int p_index) {

  int e, i1;
  
  e = 0;
  for (i1 = 0; i1 < MAX_SPLIT; i1++) {
    if (p_index >= harmonicSplit[i1]) {
      e = i1;
    }
  }
  return(e);
}

void Model::setHarmonic(int p_index, int p_num, double p_value) {
 
  double val;
  int e, c;
  
  if ((p_num >= 0) && (p_num < numHarmonics)) {
    val = (p_value < 0) ? 0 : p_value;
    if (val > max) val = max;
    harmonics[p_index][p_num] = val;
    e = calcEditGroup(p_num);
    c = MAX_SCENES * MAX_SOURCES;
    switch (p_index) {
      case 0:
        scaledHarmonics[p_index][p_num] = (1.0 / sqrt((double)poly)) * scale[c * p_index + e] * scaleLin[p_index] * pow(10, val/20.0);
        break;
      case 1:
        scaledHarmonics[p_index][p_num] = (double)rate * scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index]);
        break;
      case 2:
        scaledHarmonics[p_index][p_num] = (double)STEP/((double)rate * (1e-5 + scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index])));
        break;
      case 3:
        scaledHarmonics[p_index][p_num] = exp(-0.693147 * (double)STEP/(1e-5 + (double)rate * scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index])));
        break;
      case 4:
        scaledHarmonics[p_index][p_num] = exp(-0.693147 * (double)STEP/(1e-5 + (double)rate * scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index])));
        break;
      case 5:
        scaledHarmonics[p_index][p_num] = scale[c * p_index + e] * 1e-6 * pow(10, val / 20.0);
        break;
      case 6:
        scaledHarmonics[p_index][p_num] = exp(-0.693147 * (double)STEP/(1e-5 + (double)rate * scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index])));
        break;
      case 7:
        scaledHarmonics[p_index][p_num] = scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index]);
        break;
      case 8:
        scaledHarmonics[p_index][p_num] = 2.0 * scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index]);
        break;
      case 9:
        scaledHarmonics[p_index][p_num] = 2.0 * scale[c * p_index + e] * scaleLin[p_index] * pow(val, scaleExp[p_index]);
        break;
    }    
  }
}

void Model::setGroupScale(int p_group, int p_index, int value) {

  int i1;
  
  if (p_index) {
    scale[MAX_SCENES * MAX_SOURCES * p_index + p_group] = pow(10.0, (double)value / 1000.0);
  } else {
    scale[MAX_SCENES * MAX_SOURCES * p_index + p_group] = pow(10.0, (double)value / 1000.0) - 0.1;
  }
  for (i1 = 0; i1 < numHarmonics; i1++) {
    setHarmonic(p_index, i1, harmonics[p_index][i1]);
  }
}

double Model::getLFO() {

  return(lfo);
}

void Model::lfoStep() {

  switch (lfoState) {
  case 1:
    if (lfo < lfoDepth) {
      lfo += dlfo;
      if (lfo > lfoDepth) {
        lfo = lfoDepth;
      }
    } else {
      lfoState = 2;
    }
    break;
  case 2:
    if (lfo > -lfoDepth) {
      lfo -= dlfo;
      if (lfo < -lfoDepth) {
        lfo = -lfoDepth;
      }
    } else {
      lfoState = 1;
    }
    break;
  }    
}
   
void Model::setLfoFreq(double value) {

  lfoFreq = value;
  dlfo = 2.0 * lfoDepth * lfoFreq * (double)STEP / rate;
}          

double Model::getLfoFreq() {

  return(lfoFreq);
}          

void Model::setLfoDepth(double value) {

  lfoDepth = value;
  dlfo = 2.0 * lfoDepth * lfoFreq * (double)STEP / rate;
}          

double Model::getLfoDepth() {

  return(lfoDepth);
}          

void Model::resetLFO() {

  lfo = 0;
  lfoState = 1;
}

int Model::getEditGroup() {

  return(editGroup);
}   
        
void Model::setEditGroup(int p_group) {     

  editGroup = p_group;
}

double Model::getMorphing() {

  return(morphing);          
} 

void Model::setMorphing(double value) {

  morphing = value;
}

double Model::getDelta(int index) {

  return(delta[index]);
}

void Model::setDelta(int index, double value) {

  delta[index] = value;
}

void Model::addControlObject(QString name, int objectType, int objectIndex, QTreeWidgetItem *item) {

  controlObject co;
  
  co.uid = getUID();
  co.name = name;
  co.objectType = objectType;
  co.objectIndex = objectIndex;
  co.groupIndex = 0;
  if ((objectType == 1) || (objectType == 4)) {
    co.groupIndex = 18;
  }
  if (objectType == 8) {
    co.groupIndex = 1;
  }
  if ((objectType == 5) || (objectType == 6) || (objectType == 13)) {
    co.groupIndex = 19;
  }
  co.modDepth = 0;
  co.full = true;
  co.inverse = false;
  co.treeWidgetItem = item;
  controlObjectList.append(co);
}

void Model::addGroupSwitch(int radioGroupID, int radioGroupIndex) {

  groupSwitch g;
  
  g.eventChannel = 0;
  g.eventParam = 0;
  g.value = 0;
  g.type = 0;
  g.radioGroupID = radioGroupID;
  g.radioGroupIndex = radioGroupIndex;
  groupSwitchList.append(g);
}

int Model::hasMidiController(int type, unsigned char eventChannel,
                             unsigned int eventParam) {

  int l1;
  bool found;
  
  found = false;
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if ((midiControllerList.at(l1).eventChannel == eventChannel) 
        && (midiControllerList.at(l1).eventParam == eventParam)
        && (midiControllerList.at(l1).type == type)) {
      found = true;
      break;  
    }
  }
  return(found ? l1 : -1);                    
}                      
 
void Model::addMidiController(int type, unsigned char eventChannel,
                              unsigned int eventParam, QTreeWidgetItem *item) {
  midiController m;
  
  m.uid = getUID();
  m.type = type;
  m.eventChannel = eventChannel;
  m.eventParam = eventParam;                        
  m.treeWidgetItem = item;    
  midiControllerList.append(m);                            
} 
    
bool Model::connectController(QTreeWidgetItem *midiControllerItem, 
                              QTreeWidgetItem *controlObjectItem) {

  int l1, l2, l3;
  bool notInList;
  midiController m;
  controlObject co1, co2;
  groupSwitch g;
  int groupSwitchIndex;
  
  notInList = true;
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if (midiControllerList.at(l1).treeWidgetItem == midiControllerItem) {
      for (l2 = 0; l2 < controlObjectList.count(); l2++) {
        if (controlObjectList.at(l2).treeWidgetItem == controlObjectItem) {
          co1 = controlObjectList.at(l2);
          for (l3 = 0; l3 < midiControllerList.at(l1).connectedObjects.count(); l3++) {
            co2 = midiControllerList.at(l1).connectedObjects.at(l3);
            if ((co1.objectType == co2.objectType) && (co1.objectIndex == co2.objectIndex)) {
              notInList = false;
              break;
            }
          }
          if (notInList)  {
            m = midiControllerList.at(l1);
            if ((co1.groupIndex != 1) && (co1.groupIndex != 18) && (co1.groupIndex != 19)) {
              co1.groupIndex = currentGroupIndex;
            }  
            m.connectedObjects.append(co1);
            updateMidiControllerList(l1, m);
            updateGroupSwitchList(co1.groupIndex, m, true);
            if (co1.objectType > 8) {
              groupSwitchIndex = 0;
              switch (co1.objectType) {
                case 9:
                   groupSwitchIndex = co1.objectIndex;
                  break;
                case 10:
                  groupSwitchIndex = 2 + co1.objectIndex;
                  break;
                case 11:
                  groupSwitchIndex = 6 + co1.objectIndex;
                  break;
                case 12:
                  groupSwitchIndex = 8 + co1.objectIndex;
                  break;
              } 
              g = groupSwitchList.at(groupSwitchIndex);
              g.eventChannel = m.eventChannel;
              g.eventParam = m.eventParam;
              g.value = m.value;
              g.type = m.type;
              groupSwitchList.replace(groupSwitchIndex, g);
            }
          }  
          break;        
        }
      } 
      break;  
    }
  }
  return(notInList);
}

void Model::setCurrentGroupIndex(int value) {

  currentGroupIndex = value;
}    

int Model::getCurrentGroupIndex() {

  return(currentGroupIndex);
}

void Model::updateMidiControllerList(int index, midiController mc) {

  int l1, l2;

  midiControllerList.replace(index, mc);
  for (l1 = 0; l1 < groupSwitchList.count(); l1++) {
    for (l2 = 0; l2 < groupSwitchList.at(l1).connectedControllers.count(); l2++) {
      if (groupSwitchList.at(l1).connectedControllers.at(l2).uid == mc.uid) {
        groupSwitch g = groupSwitchList.at(l1);
        g.connectedControllers.replace(l2, mc);      
        groupSwitchList.replace(l1, g);
      }
    }
  }
}

void Model::updateGroupSwitchList(int groupIndex, midiController mc, bool addController) {

  int l1, l2, l3;
  bool removeFlag;

  if (addController) {
    if (groupIndex == 1) {
      for (l1 = 0; l1 < groupSwitchList.count(); l1++) {
        if (groupSwitchList.at(l1).radioGroupID == 3) {
          groupSwitch g = groupSwitchList.at(l1);
          g.connectedControllers.append(mc);
          groupSwitchList.replace(l1, g);
        }
      }    
    }
    if (groupIndex == 18) {
      for (l1 = 0; l1 < groupSwitchList.count(); l1++) {
        if (groupSwitchList.at(l1).radioGroupID == 1) {
          groupSwitch g = groupSwitchList.at(l1);
          g.connectedControllers.append(mc);
          groupSwitchList.replace(l1, g);
        }
      }    
    }
    if (groupIndex == 19) {
      for (l1 = 0; l1 < groupSwitchList.count(); l1++) {
        if (groupSwitchList.at(l1).radioGroupID == 2) {
          groupSwitch g = groupSwitchList.at(l1);
          g.connectedControllers.append(mc);
          groupSwitchList.replace(l1, g);
        }
      }    
    }
  } else {
    for (l1 = 0; l1 < groupSwitchList.count(); l1++) {
      for (l2 = 0; l2 < groupSwitchList.at(l1).connectedControllers.count(); l2++) {
        if (groupSwitchList.at(l1).connectedControllers.at(l2).uid == mc.uid) {
          removeFlag = true;
          for (l3 = 0; l3 < mc.connectedObjects.count(); l3++) {
            if (mc.connectedObjects.at(l3).groupIndex == groupIndex) { 
              removeFlag = false; // Nicht entfernen, falls noch ein controlObject aus der Group in der Liste ist
            }
          }
          if (removeFlag) {
            groupSwitch g = groupSwitchList.at(l1);
            g.connectedControllers.removeAt(l2);      
            groupSwitchList.replace(l1, g);
          }  
        }
      }  
    }
  }
}

MidiEventList Model::fetchEvents(int nframes) {

  MidiEventList returnList;
  
  returnList = currentEvents;
  frame = 0;
  QMutexLocker midiOutMutexLocker(&midiOutMutex);
  currentEvents.clear();
  return(returnList);
}

void Model::addMidiEvent(int frame, int type, int ch, int index, int val) {

  MidiEvent ev;
  ev.frame = frame;
  ev. type = type;
  ev.ch = ch;
  ev.index = index;
  ev.val = val;
  QMutexLocker midiOutMutexLocker(&midiOutMutex);
  currentEvents.append(ev);
}

bool Model::connectController(QTreeWidgetItem *midiControllerItem,
                              int objectType, int objectIndex, int groupIndex, int modDepth, bool full, bool inverse) { 

  int l1, l2, l3;
  bool notInList;
  midiController m;
  controlObject co1, co2;
  groupSwitch g;
  int groupSwitchIndex; 
  
  notInList = true;
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if (midiControllerList.at(l1).treeWidgetItem == midiControllerItem) {
      for (l2 = 0; l2 < controlObjectList.count(); l2++) {
        co1 = controlObjectList.at(l2);
        if ((co1.objectType == objectType) && (co1.objectIndex == objectIndex)) {
          break;
        } 
      }  
      for (l3 = 0; l3 < midiControllerList.at(l1).connectedObjects.count(); l3++) {
        co2 = midiControllerList.at(l1).connectedObjects.at(l3);
        if ((co2.objectType == objectType) && (co2.objectIndex == objectIndex)) {
          notInList = false;
          break;
        }
      }  
      if (notInList)  {
        m = midiControllerList.at(l1);
        co2.name = co1.name;
        co2.objectType = objectType;
        co2.objectIndex = objectIndex;
        co2.groupIndex = groupIndex;
        co2.modDepth = modDepth;
        co2.full = full;
        co2.inverse = inverse;
        m.connectedObjects.append(co2);
        updateMidiControllerList(l1, m);
        updateGroupSwitchList(co2.groupIndex, m, true);

        if (co2.objectType > 8) {
          groupSwitchIndex = 0;
          switch (co2.objectType) {
            case 9:
               groupSwitchIndex = co2.objectIndex;
              break;
            case 10:
              groupSwitchIndex = 2 + co2.objectIndex;
              break;
            case 11:
              groupSwitchIndex = 6 + co2.objectIndex;
              break;
            case 12:
              groupSwitchIndex = 8 + co2.objectIndex;
              break;
          } 
          g = groupSwitchList.at(groupSwitchIndex);
          g.eventChannel = m.eventChannel;
          g.eventParam = m.eventParam;
          g.value = m.value;
          g.type = m.type;
          groupSwitchList.replace(groupSwitchIndex, g);
        }

      } 
      break;  
    }
  }
  return(notInList);
}

bool Model::disconnectController(QTreeWidgetItem *midiControllerItem, QTreeWidgetItem *controlObjectItem) {
  
  int l1, l2;
  midiController m;
  bool success;
  int objectGroupIndex;
  
  success = false;
  objectGroupIndex = 0;
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if (midiControllerList.at(l1).treeWidgetItem == midiControllerItem) {
      for (l2 = 0; l2 < midiControllerList.at(l1).connectedObjects.count(); l2++) {
        if (midiControllerList.at(l1).connectedObjects.at(l2).treeWidgetItem == controlObjectItem) {
          m = midiControllerList.at(l1);
          objectGroupIndex = m.connectedObjects.at(l2).groupIndex;
          m.connectedObjects.removeAt(l2);
          updateMidiControllerList(l1, m);
          updateGroupSwitchList(objectGroupIndex, m, false);
          success = true;
          break;        
        }
      } 
      break;  
    }
  }
  return(success);
}

midiController Model::getMidiController(int index) {

  return(midiControllerList.at(index));
}

controlObject Model::getControlObject(int index) {

  return(controlObjectList.at(index));
}

int Model::getGroupSwitchCount() {

  return(groupSwitchList.count());
}

groupSwitch Model::getGroupSwitch(int index) {

  return(groupSwitchList.at(index));
}

controlObject Model::getConnectedObject(int midiIndex, int controllerIndex) {

  return(midiControllerList.at(midiIndex).connectedObjects.at(controllerIndex));
}

void Model::setControllerItem(QTreeWidgetItem *midiControllerItem, 
                              QTreeWidgetItem *controlObjectItem, 
                              QTreeWidgetItem *connectedObjectItem) {

  int l1, l2, l3;
  midiController m;
  controlObject co1, co2;
  
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if (midiControllerList.at(l1).treeWidgetItem == midiControllerItem) {
      for (l2 = 0; l2 < controlObjectList.count(); l2++) {
        if (controlObjectList.at(l2).treeWidgetItem == controlObjectItem) {
          co1 = controlObjectList.at(l2);
          for (l3 = 0; l3 < midiControllerList.at(l1).connectedObjects.count(); l3++) {
            co2 = midiControllerList.at(l1).connectedObjects.at(l3);
            if ((co1.objectType == co2.objectType) && (co1.objectIndex == co2.objectIndex)) {
              m = midiControllerList.at(l1);
              co2.treeWidgetItem = connectedObjectItem;
              m.connectedObjects.replace(l3, co2);
              updateMidiControllerList(l1, m);
              break;
            }
          }
          break;            
        }
      } 
      break;  
    }
  }
}

void Model::setControllerItem(QTreeWidgetItem *midiControllerItem, 
                              int objectType, int objectIndex,  
                              QTreeWidgetItem *connectedObjectItem) {

  int l1, l2;
  midiController m;
  controlObject co;
  
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if (midiControllerList.at(l1).treeWidgetItem == midiControllerItem) {
      for (l2 = 0; l2 < midiControllerList.at(l1).connectedObjects.count(); l2++) {
        co = midiControllerList.at(l1).connectedObjects.at(l2);
        if ((co.objectType == objectType) && (co.objectIndex == objectIndex)) {
          m = midiControllerList.at(l1);
          co.treeWidgetItem = connectedObjectItem;
          m.connectedObjects.replace(l2, co);
          updateMidiControllerList(l1, m);
          break;
        }          
      } 
      break;  
    }
  }  
}

void Model::clearMidiControllerList() {

  midiControllerList.clear();
}      

void Model::clearControlObjectList() {

  controlObjectList.clear();
}
 
bool Model::isMidiControllerItem(QTreeWidgetItem *midiControllerItem) {

  int l1;
  
  for (l1 = 0; l1 < midiControllerList.count(); l1++) {
    if (midiControllerList.at(l1).treeWidgetItem == midiControllerItem) {
      return(true);
      break;
    }
  }    
  return(false);
}

QString Model::getControlObjectName(int objectType, int objectIndex) {

  int l1;
  controlObject co;
  QString qs;

  qs = "NOT FOUND";
  for (l1 = 0; l1 < controlObjectList.count(); l1++) {
    co = controlObjectList.at(l1);
    if ((co.objectType == objectType) && (co.objectIndex == objectIndex)) {
      qs = co.name;
      break;      
    }
  }    
  return(qs);
}

int Model::midiControllerCount() {

  return(midiControllerList.count());
}

int Model::connectedObjectCount(int midiControllerIndex) {

  return(midiControllerList.at(midiControllerIndex).connectedObjects.count());
}             

void Model::setEvenOdd(int index, double value) {

  evenOdd[index] = value;
}

double Model::getEvenOdd(int index) {

  return(evenOdd[index]);
}

void Model::setMasterVolume(double value) {

  masterVolume = value;
}
        
double Model::getMasterVolume() {

  return(masterVolume);
}

void Model::setNoiseBandwidth(double value) {

  noiseBandwidth = value;
}

double Model::getNoiseBandwidth() {

  return(noiseBandwidth);       
}

void Model::setControllerModulation(QTreeWidgetItem *midiControllerItem, QTreeWidgetItem *controlObjectItem, int modDepth, bool full, bool inverse) {

  int i1, i2;
  controlObject co;
  midiController m;

  for (i1 = 0; i1 < midiControllerList.count(); i1++) {
    if (midiControllerList.at(i1).treeWidgetItem == midiControllerItem) {
      for (i2 = 0; i2 < midiControllerList.at(i1).connectedObjects.count(); i2++) {
        co = midiControllerList.at(i1).connectedObjects.at(i2);
        if (co.treeWidgetItem == controlObjectItem) {
          co.modDepth = modDepth;
          co.full = full;
          co.inverse = inverse;
          m = midiControllerList.at(i1);
          m.connectedObjects.replace(i2, co);
          updateMidiControllerList(i1, m);
          break;
        }
      } 
      break;  
    }
  }
}

controlObject Model::getControlObject(QTreeWidgetItem *midiControllerItem, QTreeWidgetItem *controlObjectItem, bool& success) {

  int i1, i2;
  controlObject co;
  midiController m;

  success = false;
  for (i1 = 0; i1 < midiControllerList.count(); i1++) {
    if (midiControllerList.at(i1).treeWidgetItem == midiControllerItem) {
      for (i2 = 0; i2 < midiControllerList.at(i1).connectedObjects.count(); i2++) {
        co = midiControllerList.at(i1).connectedObjects.at(i2);
        if (co.treeWidgetItem == controlObjectItem) {
          success = true;
          break;
        }
      } 
      break;  
    }
  }
  return(co);
}

void Model::setCompressor(int index, double value) {

  switch (index) {
    case 0:
      linearLevel = 0.008 * pow(10, value / 500.0);
      break;
    case 1:
      transitionWidth = 0.03 * pow(10, value / 500.0);
      break;
    case 2:
      compression = 0.005 * pow(10, value / 500.0);
      break;    
    case 3:
      antialiasing = 1.0 + (1000.0 - value) / 66.6666667;
      break;    
  }
}

void Model::getCompressorCoeff(double& x0, double& x1, double& a, double& b, double& c, double& d, double& aa) {

  x0 = (1.0 / (double)poly) * (double)MAX_LEVEL * linearLevel;
  x1 = x0 + (1.0 / (double)poly) * (double)MAX_LEVEL * transitionWidth;
  d = compression;
  c = x0*(1.0 - d) + (x0*x1)/(x0-x1) - x0*x0/(2.0 * (x0-x1));
  b = 1.0/(2.0 * (x0-x1));
  a = d - x1/(x0-x1);
  aa = antialiasing;
}

void Model::setInterpolate(bool on) {

  interpolate = on;
}

bool Model::getInterpolate() {

  return(interpolate);
}

void Model::setEditRange(int start, int end) {

  editStart = start;
  editEnd = end;
}

void Model::getEditRange(int& start, int& end) {

  start = editStart;
  end = editEnd;
}

VoiceData* Model::getVoiceData(int index) {

  return(voiceData[index]);
}

VoiceData* Model::currentVoiceData() {

  return(voiceData[currentVoiceDataIndex]);
}

void Model::setCurrentVoiceDataIndex(int index) {

  currentVoiceDataIndex = index;
}

void Model::sendXmlData(tokenCodeEnum code, int index1, int index2, int index3, int voiceIndex, double value) {

  emit xmlData(code, index1, index2, index3, voiceIndex, value);
}

void Model::setScalaBaseFreq(double value) {

  scalaBaseFreq = value;
}

double Model::getScalaBaseFreq() {

  return(scalaBaseFreq);
}

void Model::setScalaBaseNote(int value) {

  scalaBaseNote = value;
}

int Model::getScalaBaseNote() {

  return(scalaBaseNote);
}

void Model::setScalaMode(bool on) {

  scalaMode = on;
}

bool Model::getScalaMode() {

  return(scalaMode);
}

void Model::updateScala(int &scalaLen, bool &octavePeriod) {

  int i, index;
  double bf, bn;

  QMutexLocker jackMutexLocker(&jackMutex);
  bf = scalaBaseFreq;
  bn = scalaBaseNote;
  index = 0;
  if (scalaMode) {
    for (i = 0; i < 128; i++) {
      if ((i <= scalaBaseNote) || !scalaData.count()) {
        scalaFreq[i] = scalaBaseFreq;
        scalaNote[i] = scalaBaseNote;
      } else {
        if (scalaData.at(index).isCent) {
          scalaNote[i] = bn + scalaData.at(index).value / 100.0;
          if (scalaNote[i] > 256) scalaNote[i] = 256;
          scalaFreq[i] = freqConst * exp(scalaNote[i] * log2div12);
          if (scalaFreq[i] > rate / 2) scalaFreq[i] = rate / 2;
        } else {
          scalaFreq[i] = bf * scalaData.at(index).value;
          if (scalaFreq[i] < 1) scalaFreq[i] = 1;
          if (scalaFreq[i] > rate / 2) scalaFreq[i] = rate / 2;
          scalaNote[i] = log(scalaFreq[i] / freqConst) / log2div12;
          if (scalaNote[i] > 256) scalaNote[i] = 256;
        }
        if (index < scalaData.count() - 1) {
          index++;
        } else {
          index = 0;
          bf = scalaFreq[i];
          bn = scalaNote[i];
        }
      }
    }
  } else {
    for (i = 0; i < 128; i++) {
      scalaNote[i] = i;
      scalaFreq[i] = freqConst * exp(scalaNote[i] * log2div12);
    }
  }
  scalaLen = scalaData.count();
  octavePeriod = scalaLen && (((scalaData.at(scalaLen - 1).isCent) && (scalaData.at(scalaLen - 1).value == 1200))
                          ||  ((!scalaData.at(scalaLen - 1).isCent) && (scalaData.at(scalaLen - 1).value == 2)));
}

void Model::clearScalaData() {

  int i;

  QMutexLocker jackMutexLocker(&jackMutex);
  for (i = 0; i < 128; i++) {
    scalaFreq[i] = freqConst * exp((double)i * log2div12);
    scalaNote[i] = i;
  }
  scalaData.clear();
}

void Model::appendScalaItem(double value, bool isCent) {

  scalaItem item;

  item.value = value;
  item.isCent = isCent;
  scalaData.append(item);
}

double Model::getScalaNote(int index) {

  if ((index >= 0) && index < 128) {
    return(scalaNote[index]);
  } else {
    return(-1);
  }
}

double Model::getScalaFreq(int index) {

  if ((index >= 0) && index < 128) {
    return(scalaFreq[index]);
  } else {
    return(-1);
  }
}

void Model::setModeToggleState(bool on) {

  int index;

  modeToggleState = on;
  index = (modeToggleState) ? 1 : 0;
  updateRadioGroup(3, groupSwitchState[index]);
}

bool Model::getModeToggleState() {

  return(modeToggleState);
}

void Model::setSceneToggleState(bool on) {

  sceneToggleState = on;
}

bool Model::getSceneToggleState() {

  return(sceneToggleState);
}

void Model::updateRadioGroup(int radioGroupID, int radioGroupIndex) {
  
  int i1, sendValue, index;

  index = 0;
  if (!modeToggleState) {
    for (i1 = 0; i1 < groupSwitchList.count(); i1++) {
      if (groupSwitchList.at(i1).radioGroupID == radioGroupID) {
        index = i1;
        break;
      }
    }
    i1 = index;
    while ((i1 < groupSwitchList.count()) && (groupSwitchList.at(i1).radioGroupID == radioGroupID)) {
      sendValue = (radioGroupIndex == (i1 - index)) ? 127 : 0;
      addMidiEvent(0, 0xB0, groupSwitchList.at(i1).eventChannel, groupSwitchList.at(i1).eventParam, sendValue);
      i1++;
    }
  }  
}


void Model::setSourceSwitchState(int val) {
  
  sourceSwitchState = val;
  updateRadioGroup(1, sourceSwitchState);
  filterCurveMode = false;
}

int Model::getSourceSwitchState() {

  return(sourceSwitchState);
}

void Model::setFilterSwitchState(int val) {

  filterSwitchState = val;
  updateRadioGroup(2, filterSwitchState);
  filterCurveMode = true;
}

int Model::getFilterSwitchState() {

  return(filterSwitchState);
}

void Model::setGroupSwitchState(bool mode, int val) {

  int index;

  index = (mode) ? 1 : 0;
  groupSwitchState[index] = val;
  filterCurveMode = false;
  updateRadioGroup(3, groupSwitchState[index]);
}

int Model::getGroupSwitchState(bool mode) {

  int index;

  index = (mode) ? 1 : 0;
  return(groupSwitchState[index]);
}

void Model::setFilterCurveMode(bool on) {

  filterCurveMode = on;
}

bool Model::getFilterCurveMode() {

  return(filterCurveMode);
}
                                                
