;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;Advanced Micro Devices, Inc.
;;;june, 2001
;;;***************************************************************

(in-package "ACL2")

(local (include-book "../support/top"))

(include-book "reps")


;;;**********************************************************************
;;;                            TRUNC
;;;**********************************************************************

(defun trunc (x n)
  (* (sgn x) (fl (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))

(in-theory (disable trunc))

(defthm trunc-rewrite
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (trunc x n)
		    (* (sgn x) 
		       (fl (* (expt 2 (- (1- n) (expo x))) (abs x))) 
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable trunc-rewrite))

(defthm trunc-pos
    (implies (and (> x 0)
                  (rationalp x)
		  (integerp n)
		  (> n 0))
	     (> (trunc x n) 0))
  :rule-classes :linear)

(in-theory (disable trunc-pos))

(defthm trunc-neg
    (implies (and (< x 0)
                  (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (trunc x n) 0))
  :rule-classes :linear)

(in-theory (disable trunc-neg))

(defthm trunc-0
  (equal (trunc 0 n) 0))

(defthm sgn-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (trunc x n))
		    (sgn x))))

(in-theory (disable sgn-trunc))

(defthm trunc-minus
  (= (trunc (* -1 x) n) (* -1 (trunc x n))))

(in-theory (disable trunc-minus))

(defthm abs-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (abs (trunc x n))
		    (* (fl (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable abs-trunc))

(defthm trunc-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (trunc x n))
		  (exactp x n)))
  :rule-classes ())

(defthm trunc-exactp-b
    (implies (and (rationalp x)
		  (integerp n))
	     (exactp (trunc x n) n)))

(defthm trunc-exactp-c
    (implies (and (rationalp x)
		  (integerp n)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (trunc x n))))

(in-theory (disable trunc-exactp-c))

(defthm int-trunc
  (implies (and (rationalp x)
                (integerp n)
                (>= (expo x) n))
           (integerp (trunc x n)))
  :rule-classes :type-prescription)

(in-theory (disable int-trunc))

(defthm trunc-upper-bound
    (implies (and (rationalp x)
		  (integerp n))
	     (<= (abs (trunc x n)) (abs x)))
  :rule-classes :linear)

(in-theory (disable trunc-upper-bound))

(defthm trunc-upper-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n))
	     (<= (trunc x n) x))
  :rule-classes :linear)

(in-theory (disable trunc-upper-pos))

(defthm trunc-lower-1
    (implies (and (rationalp x)
		  (integerp n))
	     (> (abs (trunc x n)) (- (abs x) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes :linear)

(in-theory (disable trunc-lower-1))

(defthm trunc-lower-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (> (trunc x n) (* x (- 1 (expt 2 (- 1 n))))))
  :rule-classes :linear)

(in-theory (disable trunc-lower-pos))

(defthm trunc-lower-3
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (* (abs x) (- 1 (expt 2 (- 1 n))))))
  :rule-classes :linear)

(in-theory (disable trunc-lower-3))

(defthm trunc-lower-4
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (trunc x n) (- x (* (abs x) (expt 2 (- 1 n))))))
  :rule-classes :linear)

(in-theory (disable trunc-lower-4))

(defthm trunc-diff
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (abs (- x (trunc x n))) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes ())

(defthm trunc-diff-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (< (- x (trunc x n)) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes ())

(defthm trunc-diff-expo
    (implies (and (rationalp x)
		  (not (exactp x n))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- x (trunc x n))) (- (expo x) n)))
  :rule-classes ())

(defthm trunc-monotone
  (implies (and (rationalp x)
                (rationalp y)
                (integerp n)
                (<= x y))
           (<= (trunc x n) (trunc y n)))
  :rule-classes :linear)

(in-theory (disable trunc-monotone))

(defthm trunc-shift
  (implies (integerp n)
           (= (trunc (* x (expt 2 k)) n)
              (* (trunc x n) (expt 2 k)))))

(in-theory (disable trunc-shift))

(defthm expo-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (expo (trunc x n)) (expo x))))

(defthm trunc-trunc
    (implies (and (integerp n)
		  (integerp m)
		  (>= n m))
	     (equal (trunc (trunc x n) m)
		    (trunc x m))))

(in-theory (disable trunc-trunc))

(defthm plus-trunc
    (implies (and (rationalp x)
		  (>= x 0)
		  (rationalp y)
		  (>= y 0)
		  (integerp k)
		  (exactp x (+ k (- (expo x) (expo y)))))
	     (= (+ x (trunc y k))
		(trunc (+ x y) (+ k (- (expo (+ x y)) (expo y))))))
  :rule-classes ())

#|
;alternate form of plus-trunc
(defthm plus-trunc-alt
  (implies (and (rationalp x)
                (>= x 0)
                (rationalp y)
                (>= y 0)
                (integerp j)
                (exactp x (+ j (expo x) (- (expo (+ x y))))))
           (= (trunc (+ x y) j)
              (+ x (trunc y (+ j (- (expo (+ x y))) (expo y))))))
  :rule-classes ())
|#

;new
(DEFTHM PLUS-TRUNC-ALT
  (IMPLIES (AND (EXACTP X (+ J (EXPO X) (- (EXPO (+ X Y)))))
                (RATIONALP X)
                (>= X 0)
                (RATIONALP Y)
                (>= Y 0)
                (INTEGERP J))
           (= (TRUNC (+ X Y) J)
              (+ X
                 (TRUNC Y (+ J (- (EXPO (+ X Y))) (EXPO Y))))))
  :RULE-CLASSES NIL)

#|
(defthm plus-trunc-corollary
  (implies (and (rationalp x)
                (> x 0)
                (rationalp y)
                (>= y 0)
                (integerp n)
                (exactp x n)
                (< y (expt 2 (- (1+ (expo x)) n))))
           (= (trunc (+ x y) n)
              x)))
|#

;new
(defthm plus-trunc-corollary
  (IMPLIES (AND (< Y (EXPT 2 (- (1+ (EXPO X)) N)))
                (EXACTP X N)
                (RATIONALP X)
                (> X 0)
                (RATIONALP Y)
                (>= Y 0)
                (INTEGERP N))
           (= (TRUNC (+ X Y) N) X)))

(in-theory (disable plus-trunc-corollary))

(defthm trunc-plus
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e))
		  (integerp m)
		  (> m 0)
		  (integerp k)
		  (> k 0)
		  (<= m (1+ k)))
	     (= (trunc (+ (expt 2 e) (trunc y k)) m)
		(trunc (+ (expt 2 e) y) m)))
  :rule-classes ())

(defthm trunc-n+k
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  ;;this isn't really needed, but it won't hurt me.
		  (not (exactp x n))          
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		 ; (= y (- x (trunc x n)))
                  )
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (1- (sig (trunc (+ (expt 2 e) z) (1+ k))))
		   (expt 2 e))))
  :rule-classes ())

(defthm bits-trunc
    (implies (and (integerp x) (> x 0)
		  (integerp m) (>= m n)
		  (integerp n) (> n k)
		  (integerp k) (> k 0)
		  (>= x (expt 2 (1- n)))
		  (< x (expt 2 n)))
	     (= (trunc x k)
		(logand x (- (expt 2 m) (expt 2 (- n k))))))
  :rule-classes ())

(defthm bits-trunc-2
    (implies (and (integerp x) (> x 0)
		  (integerp n) (> n k)
		  (integerp k) (> k 0)
		  (= n (1+ (expo x))))
	     (= (trunc x k)
		(* (expt 2 (- n k))
		   (bits x (1- n) (- n k)))))
  :rule-classes ())

(defthm trunc-away-a
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (= (- x (expt 2 (- (expo x) n)))
		(trunc x n)))
  :rule-classes ())


;;;**********************************************************************
;;;                            AWAY
;;;**********************************************************************

(defun away (x n)
  (* (sgn x) (cg (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))

(in-theory (disable away))

(defthm away-rewrite
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (away x n)
		    (* (sgn x) 
		       (cg (* (expt 2 (- (1- n) (expo x))) (abs x))) 
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable away-rewrite))

(defthm away-pos
    (implies (and (> x 0)
                  (rationalp x)
		  (integerp n))
	     (> (away x n) 0))
  :rule-classes :linear)

(in-theory (disable away-pos))

(defthm away-neg
    (implies (and (< x 0)
                  (rationalp x)
		  (integerp n))
	     (< (away x n) 0))
  :rule-classes :linear)

(in-theory (disable away-neg))

(defthm away-0
  (equal (away 0 n) 0))

(defthm sgn-away
    (implies (and (rationalp x)
		  (integerp n))
	     (equal (sgn (away x n))
		    (sgn x))))

(in-theory (disable sgn-away))

(defthm away-minus
  (= (away (* -1 x) n) (* -1 (away x n))))

(in-theory (disable away-minus))

(defthm abs-away
    (implies (and (rationalp x)
		  (integerp n))
	     (equal (abs (away x n)) 
		    (* (cg (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable abs-away))

(defthm away-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (away x n))
		  (exactp x n)))
  :rule-classes ())

(defthm away-exactp-b
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (exactp (away x n) n)))

(defthm away-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x))
	     (>= a (away x n))))

(in-theory (disable away-exactp-c))

(defthm away-lower-bound
    (implies (and (rationalp x)
		  (integerp n))
	     (>= (abs (away x n)) (abs x)))
  :rule-classes :linear)

(in-theory (disable away-lower-bound))

(defthm away-lower-pos
    (implies (and (>= x 0)
                  (rationalp x)
		  (integerp n))
	     (>= (away x n) x))
  :rule-classes :linear)

(in-theory (disable away-lower-pos))

(defthm expo-away-lower-bound
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (expo (away x n)) (expo x)))
  :rule-classes :linear)

(in-theory (disable expo-away-lower-bound))

(defthm away-upper-1
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (abs (away x n)) (+ (abs x) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes :linear)

(in-theory (disable away-upper-1))

(defthm away-upper-2
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (< (abs (away x n)) (* (abs x) (+ 1 (expt 2 (- 1 n))))))
  :rule-classes :linear)

(in-theory (disable away-upper-2))

(defthm away-upper-3
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (abs (away x n)) (* (abs x) (+ 1 (expt 2 (- 1 n))))))
  :rule-classes :linear)

(in-theory (disable away-upper-3))

(defthm away-diff
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (abs (- (away x n) x)) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes :linear)

(in-theory (disable away-diff))

(defthm away-diff-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (< (- (away x n) x) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes :linear)

(in-theory (disable away-diff-pos))

(defthm away-diff-expo
    (implies (and (rationalp x)
		  (not (exactp x n))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- (away x n) x)) (- (expo x) n)))
  :rule-classes :linear)

(in-theory (disable away-diff-expo))

(defthm away-exactp-d
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (<= (abs (away x n)) (expt 2 (1+ (expo x)))))
  :rule-classes ())

(defthm expo-away
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0)
		  (not (= (abs (away x n)) (expt 2 (1+ (expo x))))))
	     (= (expo (away x n)) (expo x)))
  :rule-classes ())

(defthm away-monotone
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (<= x y))
	     (<= (away x n) (away y n)))
  :rule-classes :linear)

(in-theory (disable away-monotone))

(defthm away-shift
    (implies (integerp n)
	     (= (away (* x (expt 2 k)) n)
		(* (away x n) (expt 2 k)))))

(in-theory (disable away-shift))

(defthm away-away
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (equal (away (away x n) m)
		    (away x m))))

(in-theory (disable away-away))

(defthm away-imp
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m))
	     (= (away x n)
		(trunc (+ x
			  (expt 2 (- (1+ (expo x)) n))
			  (- (expt 2 (- (1+ (expo x)) m))))
		       n)))
  :rule-classes ())

#|
(defthm plus-away
    (implies (and (rationalp x)
		  (>= x 0)
		  (rationalp y)
		  (>= y 0)
		  (integerp k)
		  (exactp x (+ k (- (expo x) (expo y)))))
	     (= (+ x (away y k))
		(away (+ x y) (+ k (- (expo (+ x y)) (expo y))))))
  :rule-classes ())
|#

(DEFTHM PLUS-AWAY
  (IMPLIES (AND (EXACTP X (+ K (- (EXPO X) (EXPO Y))))
                (RATIONALP X)
                (>= X 0)
                (RATIONALP Y)
                (>= Y 0)
                (INTEGERP K))
           (= (+ X (AWAY Y K))
              (AWAY (+ X Y)
                    (+ K (- (EXPO (+ X Y)) (EXPO Y))))))
  :RULE-CLASSES NIL)

#|
;alternate form of plus-away
(defthm plus-away-alt
  (implies (and (rationalp x)
                (>= x 0)
                (rationalp y)
                (>= y 0)
                (integerp j)
                (exactp x (+ j (expo x) (- (expo (+ x y))))))
           (= (away (+ x y) j)
              (+ x (away y (+ j (- (expo (+ x y))) (expo y))))))
  :rule-classes ())
|#

(DEFTHM PLUS-AWAY-ALT
  (IMPLIES (AND (EXACTP X (+ J (EXPO X) (- (EXPO (+ X Y)))))
                (RATIONALP X)
                (>= X 0)
                (RATIONALP Y)
                (>= Y 0)
                (INTEGERP J))
           (= (AWAY (+ X Y) J)
              (+ X
                 (AWAY Y (+ J (- (EXPO (+ X Y))) (EXPO Y))))))
  :RULE-CLASSES NIL)

#|
; isn't nice for y=0
(defthm plus-away-corollary
  (implies (and (rationalp x)
                (> x 0)
                (rationalp y)
                (> y 0)
                (integerp n)
                (exactp x n)
                (< y (expt 2 (- (1+ (expo x)) n))))
           (= (away (+ x y) n)
              (fp+ x n))))
|#

(DEFTHM PLUS-AWAY-COROLLARY
  (IMPLIES (AND (< Y (EXPT 2 (- (1+ (EXPO X)) N)))
                (RATIONALP X)
                (> X 0)
                (RATIONALP Y)
                (> Y 0)
                (INTEGERP N)
                (EXACTP X N))
           (= (AWAY (+ X Y) N)
              (FP+ X N))))

(in-theory (disable plus-away-corollary)) 

(defthm trunc-away-b
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (= (away x n)
		(+ x (expt 2 (- (expo x) n)))))
  :rule-classes ())

(defthm trunc-away
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 0)
		  (not (exactp x n)))
	     (= (away x n)
		(+ (trunc x n)
		   (expt 2 (+ (expo x) 1 (- n))))))		
  :rule-classes ())

(defthm minus-trunc-4
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (< y x)
		  (integerp k)
		  (> k 0)
		  (> (+ k (- (expo (- x y)) (expo y))) 0)
		  (= n (+ k (- (expo x) (expo y))))
		  (exactp x n))
	     (equal (- x (trunc y k))
		    (away (- x y) (+ k (- (expo (- x y)) (expo y))))))
  :rule-classes ())

(defthm minus-trunc-5
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (< x y)
		  (integerp k)
		  (> k 0)
		  (> (+ k (- (expo (- x y)) (expo y))) 0)
		  (= n (+ k (- (expo x) (expo y))))
		  (exactp x n))
	     (equal (- x (trunc y k))
		    (- (trunc (- y x) (+ k (- (expo (- x y)) (expo y)))))))
  :rule-classes ())


;;;**********************************************************************
;;;                            NEAR
;;;**********************************************************************

(defun re (x)
  (- x (fl x)))

(defun near (x n)
  (let ((z (fl (* (expt 2 (1- n)) (sig x))))
	(f (re (* (expt 2 (1- n)) (sig x)))))
    (if (< f 1/2)
	(trunc x n)
      (if (> f 1/2)
	  (away x n)
	(if (evenp z)
	    (trunc x n)
	  (away x n))))))

(in-theory (disable near))

(defthm near1-a
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0)
		  (< (- x (trunc x n)) (- (away x n) x)))
	     (= (near x n) (trunc x n)))
  :rule-classes ())

(defthm near1-b
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0)
		  (> (- x (trunc x n)) (- (away x n) x)))
	     (= (near x n) (away x n)))
  :rule-classes ())

(defthm near-choice
    (or (= (near x n) (trunc x n))
	(= (near x n) (away x n)))
  :rule-classes ())

(defthm near-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (> (near x n) 0))
  :rule-classes :linear)

(in-theory (disable near-pos))

(defthm near-neg
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0))
	     (< (near x n) 0))
  :rule-classes :linear)

(in-theory (disable near-neg))

(defthm near-0
  (equal (near 0 n) 0))

(defthm sgn-near-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (near x n))
		    (sgn x))))

(in-theory (disable sgn-near-2))

(defthm near-minus
  (= (near (* -1 x) n) (* -1 (near x n))))

(in-theory (disable near-minus))

(defthm near-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (near x n))
		  (exactp x n)))
  :rule-classes ())

(defthm near-exactp-b
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (exactp (near x n) n)))

(defthm near-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x))
	     (>= a (near x n))))

(in-theory (disable near-exactp-c))

(defthm near-exactp-d
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (near x n))))

(in-theory (disable near-exactp-d))

(defthm near<=away
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (<= (near x n) (away x n)))
  :rule-classes ())

(defthm near>=trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (>= (near x n) (trunc x n)))
  :rule-classes ())

(defthm monotone-near
    (implies (and (rationalp x)
		  (rationalp y)
		  (< 0 x)
		  (<= x y)
		  (integerp n)
		  (> n 0))
	     (<= (near x n) (near y n))))

(defthm near-shift
    (implies (and (rationalp x)
                  (integerp n)
		  (integerp k))
	     (= (near (* x (expt 2 k)) n)
		(* (near x n) (expt 2 k)))))

(in-theory (disable near-shift))

(defthm near2
    (implies (and (rationalp x)
		  (rationalp y)
		  (> x 0)
		  (> y 0)
		  (integerp n)
		  (> n 0)
		  (exactp y n))
	     (>= (abs (- x y)) (abs (- x (near x n)))))
  :rule-classes ())

(defun near-witness (x y n)
  (if (= (expo x) (expo y))
      (/ (+ (near x n) (near y n)) 2)
    (expt 2 (expo y))))

(in-theory (disable near-witness))

(defthm near-near-lemma
    (implies (and (rationalp x)
		  (rationalp y)
		  (< 0 x)
		  (< x y)
		  (integerp n)
		  (> n 0)
		  (not (= (near x n) (near y n))))
	     (and (<= x (near-witness x y n))
		  (<= (near-witness x y n) y)
		  (exactp (near-witness x y n) (1+ n))))
  :rule-classes ())

(defthm near-near
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp a)
		  (integerp n)
		  (integerp k)
		  (> k 0)
		  (>= n k)		  
		  (< 0 a)
		  (< a x)
		  (< 0 y)
		  (< y (fp+ a (1+ n)))
		  (exactp a (1+ n)))
	     (<= (near y k) (near x k)))
  :rule-classes ())

(defthm near-est
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0))
	     (<= (abs (- x (near x n)))
		 (expt 2 (- (expo x) n))))
  :rule-classes ())

(defthm near-a-a
    (implies (and (rationalp x) (> x 0)
		  (rationalp a) (> a 0)
		  (integerp n) (> n 0)
		  (exactp a n)
		  (> x (+ a (expt 2 (- (expo a) n)))))
	     (>= (near x n) (+ a (expt 2 (- (1+ (expo a)) n)))))
  :rule-classes ())

(defthm near-a-b
    (implies (and (rationalp x) (> x 0)
		  (rationalp a) (> a 0)
		  (integerp n) (> n 0)
		  (exactp a n)
		  (< x (+ a (expt 2 (- (expo a) n)))))
	     (<= (near x n) a))
  :rule-classes ())

(defthm near-a-c
    (implies (and (rationalp x) (> x 0)
		  (rationalp a) (> a 0)
		  (integerp n) (> n 0)
		  (exactp a n)
		  (< x a)
		  (> x (- a (expt 2 (- (expo x) n)))))
	     (>= (near x n) a))
  :rule-classes ())

(defthm near-exact
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 1)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (exactp (near x n) (1- n)))
  :rule-classes ())

(defthm near-power-a
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 1)
		  (>= (+ x (expt 2 (- (expo x) n)))
		      (expt 2 (1+ (expo x)))))
	     (= (near x n)
		(expt 2 (1+ (expo x)))))
  :rule-classes ())

(defthm near-power-b
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 1)
		  (>= (+ x (expt 2 (- (expo x) n)))
		      (expt 2 (1+ (expo x)))))
	     (= (trunc (+ x (expt 2 (- (expo x) n))) n)
		(expt 2 (1+ (expo x)))))
  :rule-classes ())

(defthm near-trunc
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 1))
	     (= (near x n)
		(if (and (exactp x (1+ n)) (not (exactp x n)))
		    (trunc (+ x (expt 2 (- (expo x) n))) (1- n))
		  (trunc (+ x (expt 2 (- (expo x) n))) n))))
  :rule-classes ())


;;;**********************************************************************
;;;                            NEAR+
;;;**********************************************************************

(defun near+ (x n)
  (if (< (re (* (expt 2 (1- n)) (sig x)))
	 1/2)
      (trunc x n)
    (away x n)))

(in-theory (disable near+))

(defthm near+trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (= (near+ x n)
		(trunc (+ x (expt 2 (- (expo x) n))) n)))		
  :rule-classes ())

(defthm sgn-near+
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (= (near+ x n)
		(* (sgn x) (near+ (abs x) n))))
  :rule-classes ())

(defthm sgn-near+-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (near+ x n))
		    (sgn x))))

(defthm near+-pos
  (implies (and (rationalp x)
                (> x 0)
                (integerp n)
                (> n 0))
           (> (near+ x n) 0))
  :rule-classes :linear)

(defthm near+-neg
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0))
	     (< (near+ x n) 0))
  :rule-classes :linear)

(defthm near+-0-0
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (iff (= (near+ x n) 0)
		  (= x 0)))
  :rule-classes ())

(defthm near+-0
  (equal (near+ 0 n) 0))

(defthm near+-minus
  (= (near+ (* -1 x) n) (* -1 (near+ x n))))

(in-theory (disable near+-minus))

(defthm near+-shift
    (implies (and (rationalp x)
		  (integerp n)
		  (integerp k))
	     (= (near+ (* x (expt 2 k)) n)
		(* (near+ x n) (expt 2 k)))))

(in-theory (disable near+-shift))

(defthm near+-choice
    (or (= (near+ x n) (trunc x n))
	(= (near+ x n) (away x n)))
  :rule-classes ())

(defthm near+1-a
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0)
		  (< (- x (trunc x n)) (- (away x n) x)))
	     (= (near+ x n) (trunc x n)))
  :rule-classes ())

(defthm near+1-b
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0)
		  (> (- x (trunc x n)) (- (away x n) x)))
	     (= (near+ x n) (away x n)))
  :rule-classes ())

(defthm near+<=away
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (<= (near+ x n) (away x n)))
  :rule-classes ())

(defthm near+>=trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (>= (near+ x n) (trunc x n)))
  :rule-classes ())

(defthm near+2
    (implies (and (rationalp x)
		  (rationalp y)
		  (> x 0)
		  (> y 0)
		  (integerp n)
		  (> n 0)
		  (exactp y n))
	     (>= (abs (- x y)) (abs (- x (near+ x n)))))
  :rule-classes ())

(defthm near+-est
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0))
	     (<= (abs (- x (near+ x n)))
		 (expt 2 (- (expo x) n))))
  :rule-classes ())

(defthm monotone-near+
  (implies (and (rationalp x)
                (rationalp y)
                (< 0 x)
                (<= x y)
                (integerp n)
                (> n 0))
           (<= (near+ x n) (near+ y n))))

(defthm near+-power
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 1)
		  (>= (+ x (expt 2 (- (expo x) n)))
		      (expt 2 (1+ (expo x)))))
	     (= (near+ x n)
		(expt 2 (1+ (expo x)))))
  :rule-classes ())

(defthm near+-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (near+ x n))
		  (exactp x n)))
  :rule-classes ())

(defthm near+-exactp-b
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (exactp (near+ x n) n)))

(defthm near+-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x))
	     (>= a (near+ x n))))

(defthm near+-exactp-d
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (near+ x n))))


;;;**********************************************************************
;;;                           STICKY
;;;**********************************************************************

(defun sticky (x n)
  (cond ((exactp x (1- n)) x)
	(t (+ (trunc x (1- n))
              (* (sgn x) (expt 2 (1+ (- (expo x) n))))))))

(defthm sticky-1
  (implies (rationalp x)
           (equal (sticky x 1)
                  (* (sgn x) (expt 2 (expo x))))))

(in-theory (disable sticky))

(defthm sticky-pos
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 0))
	     (> (sticky x n) 0))
  :rule-classes :linear)

(in-theory (disable sticky-pos))

(defthm sticky-0
  (equal (sticky 0 n) 0))

(defthm sticky-minus
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (= (sticky (* -1 x) n) (* -1 (sticky x n)))))

(in-theory (disable sticky-minus))

(defthm sticky-shift
    (implies (and (rationalp x)
		  (integerp n) (> n 0)
		  (integerp k))
	     (= (sticky (* (expt 2 k) x) n)
		(* (expt 2 k) (sticky x n))))		
  :rule-classes ())

(defthm sticky-exactp
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 0))
	     (exactp (sticky x n) n))
  :rule-classes ())

(defthm sticky-exactp-n-1
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 1))
	     (iff (exactp (sticky x n) (1- n))
		  (exactp x (1- n))))
  :rule-classes ())

(defthm expo-sticky
    (implies (and (rationalp x) (> x 0)
		  (integerp n) (> n 0))
	     (= (expo (sticky x n))
		(expo x)))
  :rule-classes ())

(defthm trunc-sticky
    (implies (and (rationalp x) (> x 0)
		  (integerp m) (> m 0)
		  (integerp n) (> n m))
	     (= (trunc (sticky x n) m)
		(trunc x m)))
  :rule-classes ())

(defthm away-sticky
    (implies (and (rationalp x) (> x 0)
		  (integerp m) (> m 0)
		  (integerp n) (> n m))
	     (= (away (sticky x n) m)
		(away x m)))
  :rule-classes ())

(defthm near-sticky
    (implies (and (rationalp x) (> x 0)
		  (integerp m) (> m 0)
		  (integerp n) (> n (1+ m)))
	     (= (near (sticky x n) m)
		(near x m)))
  :rule-classes ())

(defthm sticky-sticky
    (implies (and (rationalp x)
		  (integerp m)
		  (> m 1)
		  (integerp n)
		  (>= n m))
	     (= (sticky (sticky x n) m)
		(sticky x m)))
  :rule-classes ())

(defthm sticky-plus
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (integerp k)
		  (= k1 (+ k (- (expo x) (expo y))))
		  (= k2 (+ k (- (expo (+ x y)) (expo y))))
		  (> k 1)
		  (> k1 1)
		  (> k2 1)
		  (exactp x (1- k1)))
	     (= (+ x (sticky y k))
		(sticky (+ x y) k2)))
  :rule-classes ())

(defthm minus-sticky
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (integerp k)
		  (= k1 (+ k (- (expo x) (expo y))))
		  (= k2 (+ k (- (expo (- x y)) (expo y))))
		  (> k 1)
		  (> k1 1)
		  (> k2 1)
		  (exactp x (1- k1)))
	     (= (- x (sticky y k))
		(sticky (- x y) k2)))
  :rule-classes ())

(defthm sticky-lemma
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (integerp k)
		  (= k1 (+ k (- (expo x) (expo y))))
		  (= k2 (+ k (- (expo (+ x y)) (expo y))))
		  (> k 1)
		  (> k1 1)
		  (> k2 1)
		  (exactp x (1- k1)))
	     (= (+ x (sticky y k))
		(sticky (+ x y) k2)))
  :rule-classes ())


;;;**********************************************************************
;;;                              ODD 
;;;**********************************************************************

(defun odd (x n)
  (let ((z (fl (* (expt 2 (1- n)) (sig x)))))
    (if (evenp z)
	(* (sgn x) (1+ z) (expt 2 (- (1+ (expo x)) n)))
      (* (sgn x) z (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable odd))

(defthm odd-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (> (odd x n) 0))
  :rule-classes ())

(defthm odd>=trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (>= (odd x n) (trunc x n)))
  :rule-classes ())

(defthm odd-rewrite
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (equal (odd x n)
		    (let ((z (fl (* (expt 2 (- (1- n) (expo x))) x))))
		      (if (evenp z)
			  (* (1+ z) (expt 2 (- (1+ (expo x)) n)))
			(* z (expt 2 (- (1+ (expo x)) n)))))))
  :rule-classes ())

(defthm odd-other
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n) 
		  (> n 1))
	     (= (odd x n)
		(+ (trunc x (1- n))
		   (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ())

(defthm expo-odd
    (implies (and (rationalp x)
		  (integerp n)
		  (> x 0)
		  (> n 1))
	     (equal (expo (odd x n)) (expo x))))

(defthm exactp-odd
    (implies (and (rationalp x)
		  (integerp n)
		  (> x 0)
		  (> n 1))
	     (exactp (odd x n) n))
  :rule-classes ())

(defthm not-exactp-odd
    (implies (and (rationalp x)
		  (integerp n)
		  (> x 0)
		  (> n 1))
	     (not (exactp (odd x n) (1- n))))
  :rule-classes ())

(defthm trunc-odd
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (> n m))
	     (= (trunc (odd x n) m)
		(trunc x m)))
  :rule-classes ())

(defun kp (k x y)
  (+ k (- (expo (+ x y)) (expo y))))

(defthm odd-plus
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp k)
		  (> x 0)
		  (> y 0)
		  (> k 1)
		  (> (+ (1- k) (- (expo x) (expo y))) 0)
		  (exactp x (+ (1- k) (- (expo x) (expo y)))))
	     (= (+ x (odd y k))
		(odd (+ x y) (kp k x y))))
  :rule-classes ())

(defthm trunc-trunc-odd
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp m)
		  (integerp k)
		  (> x y)
		  (> y 0)
		  (> k 0)
		  (>= (- m 2) k))
	     (>= (trunc x k) (trunc (odd y m) k)))
  :rule-classes ())

(defthm away-away-odd
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp m)
		  (integerp k)
		  (> x y)
		  (> y 0)
		  (> k 0)
		  (>= (- m 2) k))
	     (>= (away x k) (away (odd y m) k)))
  :rule-classes ())

(defthm near-near-odd
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp m)
		  (integerp k)
		  (> x y)
		  (> y 0)
		  (> k 0)
		  (>= (- m 2) k))
	     (>= (near x k) (near (odd y m) k)))
  :rule-classes ())


;;;**********************************************************************
;;;        IEEE Rounding (most thms also allow away as a mode) 
;;;**********************************************************************

(defun inf (x n)
  (if (>= x 0)
      (away x n)
    (trunc x n)))

(defun minf (x n)
  (if (>= x 0)
      (trunc x n)
    (away x n)))

(defun ieee-mode-p (mode)
  (member mode '(trunc inf minf near)))

(defun rounding-mode-p (mode)
  (or (IEEE-mode-p mode) (equal mode 'away)))

(defthm ieee-mode-p-implies-rounding-mode-p
  (implies (IEEE-mode-p mode)
           (rounding-mode-p mode))
  :rule-classes (:rewrite; :forward-chaining
))

(in-theory (disable ieee-mode-p))

(defun rnd (x mode n)
  (case mode
    (away (away x n))
    (trunc (trunc x n))
    (inf (inf x n))
    (minf (minf x n))
    (near (near x n))))

(in-theory (disable rnd))

(defthm rnd-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode))
	     (> (rnd x mode n) 0))
  :rule-classes ())

(defthm rnd-neg
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode))
	     (< (rnd x mode n) 0))
  :rule-classes ())

(defthm rnd-0
    (implies (and (rounding-mode-p m))
	     (equal (rnd 0 m n) 0)))

(defthm sgn-rnd
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (rnd x mode n))
		    (sgn x))))

(in-theory (disable sgn-rnd))

(defun flip (m)
  (case m
    (inf 'minf)
    (minf 'inf)
    (t m)))

(in-theory (disable flip))

(defthm ieee-mode-p-flip
    (implies (ieee-mode-p m)
	     (ieee-mode-p (flip m))))

(defthm rounding-mode-p-flip
    (implies (rounding-mode-p m)
	     (rounding-mode-p (flip m))))

(defthm rnd-flip
    (implies (and (rounding-mode-p m))
	     (= (rnd (* -1 x) (flip m) n)
		(* -1 (rnd x m n))))
  :rule-classes ())

(defthm rnd-exactp-a
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (rnd x mode n))
		  (exactp x n)))
  :rule-classes ())

(defthm rnd-exactp-b
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (exactp (rnd x mode n) n)))

(defthm rnd-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x))
	     (>= a (rnd x mode n))))

(in-theory (disable rnd-exactp-c))

(defthm rnd-exactp-d
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (rnd x mode n))))

(in-theory (disable rnd-exactp-d))

(defthm rnd<=away
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (<= (rnd x mode n) (away x n)))
  :rule-classes ())

(defthm rnd>=trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (>= (rnd x mode n) (trunc x n)))
  :rule-classes ())

(defthm monotone-rnd
    (implies (and (rationalp x)
		  (rationalp y)
		  (< 0 x)
		  (<= x y)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (<= (rnd x mode n) (rnd y mode n))))

(in-theory (disable monotone-rnd))

(defthm exactp-rnd
    (implies (and (rationalp x)
		  (rounding-mode-p mode)
		  (integerp n)
		  (> n 0))
	     (exactp (rnd x mode n) n)))

(defthm rnd-shift
    (implies (and (rationalp x)
		  (integerp n)
		  (rounding-mode-p mode)
		  (integerp k))
	     (= (rnd (* x (expt 2 k)) mode n)
		(* (rnd x mode n) (expt 2 k))))
  :rule-classes ())

(defthm expo-rnd
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode)
		  (not (= (abs (rnd x mode n))
			  (expt 2 (1+ (expo x))))))
	     (= (expo (rnd x mode n))
		(expo x)))
  :rule-classes ())

(defthm expo-rnd-bnd
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0)
		  (rounding-mode-p mode))
	     (>= (expo (rnd x mode n))
		 (expo x)))
  :rule-classes ())

(defun rnd-const (e mode n)
  (case mode
    (near (expt 2 (- e n)))
    (inf (1- (expt 2 (1+ (- e n)))))
    (otherwise 0)))

(defthm rnd-const-thm
    (implies (and (ieee-mode-p mode)
		  (integerp n)
		  (> n 1)
		  (integerp x)
		  (> x 0)
		  (>= (expo x) n))
	     (= (rnd x mode n)
		(if (and (eql mode 'near)
			 (exactp x (1+ n))
			 (not (exactp x n)))
		    (trunc (+ x (rnd-const (expo x) mode n)) (1- n))
		  (trunc (+ x (rnd-const (expo x) mode n)) n))))
  :rule-classes ())

(defthm rnd-sticky
    (implies (and (rounding-mode-p mode)
		  (rationalp x) (> x 0)
		  (integerp k) (> k 0)
		  (integerp n) (> n (1+ k)))
	     (= (rnd x mode k)
		(rnd (sticky x n) mode k)))
  :rule-classes ())

(defthm rnd-diff
  (implies (and (rationalp x)
                (integerp n)
                (> n 0)
                (rounding-mode-p mode))
           (< (abs (- x (rnd x mode n))) (expt 2 (- (1+ (expo x)) n)))))

(in-theory (disable rnd-diff))

(defthm plus-rnd
  (implies (and (rationalp x)
                (>= x 0)
                (rationalp y)
                (>= y 0)
                (integerp k)
                (exactp x (+ -1 k (- (expo x) (expo y))))
                (rounding-mode-p mode))
           (= (+ x (rnd y mode k))
              (rnd (+ x y)
                   mode
                   (+ k (- (expo (+ x y)) (expo y))))))
  :rule-classes nil)



;;;**********************************************************************
;;;                         Denormal Rounding 
;;;**********************************************************************

(defun drnd (x mode n k)
  (- (rnd (+ x (* (sgn x) (expt 2 (- 2 (expt 2 (1- k)))))) mode n)
     (* (sgn x) (expt 2 (- 2 (expt 2 (1- k)))))))

(in-theory (disable drnd))

(defthm drnd-0
    (implies (rounding-mode-p m)
	     (equal (drnd 0 m n k) 0)))

(defthm drnd-flip
    (implies (and (rationalp x)
		  (rounding-mode-p m)
		  (integerp n)
		  (integerp k))
	     (= (drnd (* -1 x) (flip m) n k)
		(* -1 (drnd x m n k))))
  :rule-classes ())

(defthm drnd-sticky
    (implies (and (rounding-mode-p mode)
		  (natp n)
		  (> n 0)
		  (natp m)
		  (> m 1)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (<= (expo x) (- 1 (expt 2 (1- k))))
		  (<= (expo x) (- m (+ n (expt 2 (1- k))))))
	     (equal (drnd (sticky x m) mode n k)
		    (drnd x mode n k)))
  :rule-classes ())

(defthm drnd-tiny-equal
    (implies (and (ieee-mode-p m)
		  (natp n)
		  (> n 0)
		  (natp k)
		  (> k 0)
		  (rationalp x)
		  (< (abs x) (expt 2 (- 2 (+ n (expt 2 (1- k))))))
		  (rationalp y)
		  (< (abs y) (expt 2 (- 2 (+ n (expt 2 (1- k))))))
		  (equal (sgn x) (sgn y)))
	     (equal (drnd x m n k)
		    (drnd y m n k)))
  :rule-classes ())


(defun smallest-positive-normal (k)
  (expt 2 (- 1 (bias k))))

(in-theory (disable smallest-positive-normal))

;;these next three show that smallest-positive-normal really is what it claims to be

;was in rewrite to; put back there?
(defthm positive-spn
  (> (smallest-positive-normal k) 0)
  :rule-classes ( :linear))

(defthm nrepp-spn
  (implies (and (integerp n)
                (> n 0)
                (integerp k)
                (> k 1))
           (nrepp (smallest-positive-normal k) n k)))

(defthm smallest-spn
  (implies (and (nrepp x n k)
                (integerp n)
                (> n 0)
                (integerp k)
                (> k 1)
                )
           (>= (abs x) (smallest-positive-normal k)))
  :RULE-CLASSES
  ((:REWRITE :MATCH-FREE :ONCE)))

(defun smallest-positive-denormal (n k)
     (expt 2 (+ 2 (- (bias k)) (- n))))

(in-theory (disable smallest-positive-denormal))

;;these next three show that smallest-positive-denormal really is what it claims to be

(defthm positive-spd
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0))
           (> (smallest-positive-denormal n k) 0)))

(defthm drepp-spd
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0))
           (drepp (smallest-positive-denormal n k) n k)))

(defthm smallest-spd
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (drepp x n k))
           (>= (abs x) (smallest-positive-denormal n k))))

;drnd returns a denormal, or zero, or the smallest normal

(defthm drnd-type
  (implies (and (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rounding-mode-p mode))
           (or (drepp (drnd x mode n k) n k)
               (= (drnd x mode n k) 0)
               (= (drnd x mode n k) (* (sgn x) (smallest-positive-normal k)))))
  :rule-classes nil)


(defthm drnd-rewrite
  (implies (and (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (rounding-mode-p mode)
                (integerp n)
                (> n 1)
                (integerp k)
                (> k 0))
           (equal (drnd x mode n k)
                  (rnd x 
                       mode
                       (+ n 
                          (- (expo (smallest-positive-normal k))) 
                          (expo x))))))

(in-theory (disable drnd-rewrite))

(defthm drnd-of-drepp-is-NOP
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (drepp x n k)
                (rounding-mode-p mode))
           (= (drnd x mode n k)
              x)))
#|
(defthm drnd-spn-is-spn-general
  (implies (and (rounding-mode-p mode)
                (integerp n)
                (>= n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (= (abs x) (smallest-positive-normal k)))
           (= (drnd x mode n k)
              x)))
|#

;new
(DEFTHM DRND-SPN-IS-SPN-GENERAL
  (IMPLIES (AND (= (ABS X) (SMALLEST-POSITIVE-NORMAL K))
                (ROUNDING-MODE-P MODE)
                (INTEGERP N)
                (>= N 1)
                (INTEGERP K)
                (> K 0)
                (RATIONALP X))
           (= (DRND X MODE N K) X)))

(defthm drnd-trunc-never-goes-away-from-zero
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k)))
           (<= (abs (drnd x 'trunc n k))
               (abs x))))

(defthm drnd-away-never-goes-toward-zero
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k)))
           (>= (abs (drnd x 'away n k))
               (abs x))))

(defthm drnd-inf-never-goes-down
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k)))
           (>= (drnd x 'inf n k)
               x)))

(defthm drnd-minf-never-goes-up
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k)))
           (<= (drnd x 'minf n k)
               x)))

(defthm drnd-trunc-skips-no-denormals
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (drepp a n k)
                (<= (abs a) (abs x))
                )
           (<= (abs a)
               (abs (drnd x 'trunc n k)))))

(defthm drnd-away-skips-no-denormals
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (drepp a n k)
                (>= (abs a) (abs x))
                )
           (>= (abs a) (abs (drnd x 'away n k)))))


(defthm drnd-inf-skips-no-denormals
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (drepp a n k)
                (>= a x))
           (>= a (drnd x 'inf n k))))

(defthm drnd-minf-skips-no-denormals
  (implies (and (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (drepp a n k)
                (<= a x))
           (<= a (drnd x 'minf n k))))

(defthm drnd-diff
  (implies (and (rationalp x)
                (<= (ABS X) (SMALLEST-POSITIVE-NORMAL K))
                (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (rounding-mode-p mode))
           (< (abs (- x (drnd x mode n k))) (smallest-positive-denormal n k))))

(defun next-denormal (x n k)
  (+ x (smallest-positive-denormal n k)))

(in-theory (disable next-denormal))

;shows that next-denormal behaves as expected
(defthm denormal-spacing
  (implies (and (integerp n)
                (integerp k)
                (> k 0)
                (> n 1)
                (drepp x n k)
                (drepp x+ n k)
                (>= x 0)
                (> x+ x))
           (>= x+ (next-denormal x n k))))

(defthm no-denormal-is-closer-than-what-drnd-near-returns
  (implies (and (rationalp x)
                (<= (abs x) (smallest-positive-normal k))
                (integerp n)
                (> n 1)
                (integerp k)
                (> k 0)
                (drepp a n k))
           (>= (abs (- x a)) (abs (- x (drnd x 'near n k))))))



