/* ------------------------------------------------------------------------
 * $Id: InterTraxDevice.cc,v 1.3 2001/06/27 12:48:57 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-21 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

// -- Intertrax Includes
#include "itrax.h"

// -- 3Dwm Includes
#include "Celsius/Math.hh"
#include "Polhem/LogGroup.hh"
#include "Polhem/InterTraxDevice.hh"

// Device filename
#define DEVICE "/dev/input/tracker0"

// -- Code Segment

InterTraxDevice::InterTraxDevice(InputEventQueue *eventQueue)
    : InputDevice(eventQueue) 
{    
    char name[100];
    
    // Open the tracker device
    if ((_tracker = open(DEVICE, O_NONBLOCK)) < 0)
	throw Exception("InterTraxDevice: open()");

    // Retrieve device name
    if (ioctl(_tracker, ITRAXIOCGNAME(100), &name) < 0)
	throw Exception("InterTraxDevice: ioctl(GNAME)");
    _name = name;

    // Configure filter control
    struct filtercontrol filter;
    float coeff[] = { 0.2 , 0.2 , 0.2 , 0.2 , 0.2 };
    filter.type = ITRAX_FILTER_FASTMEAN;
    filter.size = 5;
    filter.axes = 0;
    filter.coeff = coeff;
    if (ioctl(_tracker, ITRAXIOCSFILTER, &filter ) < 0) 
	throw Exception("InterTraxDevice: ioctl(SFILTER)");

#ifdef DEBUG
    short id[4];
    int abs[5];
    
    // Retrieve device information
    if (ioctl(_tracker, ITRAXIOCGID, &id) < 0)
	throw Exception("InterTraxDevice: ioctl(GID)");
    Logger::log(LogGroup::Platform)
	<< "Bus: " << id[0] << " Vendor: " << id[1] << " Product: " << id[2]
	<< " Version: " << id[3] << std::endl;
    
    // Retrieve device absolute values
    if (ioctl(_tracker, ITRAXIOCGABS(5), &abs) < 0)
	throw Exception("InterTraxDevice: ioctl(GABS)");
    Logger::log(LogGroup::Platform)
	<< "Abs 0: " << abs[0] << " Abs 1: " << abs[1] << " Abs 2: " << abs[2]
	<< " Abs 3: " << abs[3] << " Abs 4: " << abs[4] << std::endl;
    
    // Retrieve filter values
    if (ioctl(_tracker, ITRAXIOCGFILTER, &filter) < 0)
	throw Exception("InterTraxDevice: ioctl(GFILTER)");
    Logger::log(LogGroup::Platform)
	<< "Filter axes: " << filter.axes << " type: " << filter.type
	<< " size: " << filter.size << std::endl;
#endif /* DEBUG */
}

InterTraxDevice::~InterTraxDevice()
{
    // empty
    // @@@ Should we shut down the tracker device somehow?
}

void InterTraxDevice::run() 
{
    const double max_yaw = 360.0f;
    const double max_pitch = 360.0f;
    const double max_roll = 360.0f;
    struct trackerposition data;
    double yaw = .0, pitch = .0, roll = .0;
	
    while (true) {
	
	// Retrieve tracker input
	read(_tracker, &data, sizeof(trackerposition));
	
	// Create the 3Dwm input event
	InputEvent *ie = new InputEvent();

	// Make sure we don't get any NaN:s!
	yaw = Math::abs(data.raw[0]) < max_yaw ? data.raw[0] : yaw;
	pitch = Math::abs(data.raw[1]) < max_pitch ? data.raw[1] : pitch;
	roll = Math::abs(data.raw[2]) < max_roll ? data.raw[2] : roll;
	
	// Build the event
	ie->add("headtracker");
	ie->add("headtracker.yaw", yaw);
	ie->add("headtracker.pitch", pitch);
	ie->add("headtracker.roll", roll);
	
	// Add the event to the event queue
	addInputEvent(ie);
	
	// @@@We need to improve this! Naive!
	const timespec ts = { 0, 1000 };
	nanosleep(&ts, NULL);
    }
}
