/* ------------------------------------------------------------------------
 * $Id: Quaternion.cc,v 1.3 2001/06/18 21:55:14 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * Reference from
 * <http://www.gamasutra.com/features/19980703/Quaternionernions_01.htm>
 * ------------------------------------------------------------------------
 * File created 2001-06-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- Local Includes
#include "Math.hh"
#include "Quaternion.hh"

// -- Code Segment

void Quaternion::fromMatrix(float matrix[4][4])
{
    float tr, s, q[4];
    int i, j, k;
    int nxt[3] = {1, 2, 0};
    
    // Add the diagonal elements
    tr = matrix[0][0] + matrix[1][1] + matrix[2][2];
    
    // Do we have a positive diagonal?
    if (tr > 0.0) {

        // Yes, do some computation
        s = (float) sqrt(tr + 1.0);
        _w = s / 2.0f;
        s = 0.5f / s;
        _v.x() = (matrix[1][2] - matrix[2][1]) * s;
        _v.y() = (matrix[2][0] - matrix[0][2]) * s;
        _v.z() = (matrix[0][1] - matrix[1][0]) * s;
    } 
    else {
	
        // No, diagonal is negative
        i = 0;
        if (matrix[1][1] > matrix[0][0]) i = 1;
        if (matrix[2][2] > matrix[i][i]) i = 2;
        j = nxt[i];
        k = nxt[j];
	
        s = (float) sqrt((matrix[i][i] - (matrix[j][j] + matrix[k][k])) + 1.0);
        q[i] = s * 0.5f;
        
        // Check for zero
        if (s != 0.0f) s = 0.5f / s;
	
        // Fill in the temporary fields
        q[3] = (matrix[j][k] - matrix[k][j]) * s;
        q[j] = (matrix[i][j] + matrix[j][i]) * s;
        q[k] = (matrix[i][k] + matrix[k][i]) * s;
	
        // Fill out the quaternion
        _v.x() = q[0];
        _v.y() = q[1];
        _v.z() = q[2];
        _w = q[3];
    }
}

/*
 M = {{1-2(yy+zz),  2(xy-wz),  2(xz+wy)},
             {  2(xy+wz),1-2(xx+zz),  2(yz-wx)},
             {  2(xz-wy),  2(yz+wx),1-2(xx+yy)}}.
*/

void Quaternion::toMatrix(float matrix[4][4]) const
{
    float wx, wy, wz, xx, yy, yz, xy, xz, zz, x2, y2, z2;
    
    // Calculate coefficients (to minimize redundancy)
    x2 = _v.x() + _v.x(); y2 = _v.y() + _v.y(); z2 = _v.z() + _v.z();
    
    xx = _v.x() * x2;   xy = _v.x() * y2;   xz = _v.x() * z2;
    yy = _v.y() * y2;   yz = _v.y() * z2;   zz = _v.z() * z2;
    
    wx = _w * x2;   wy = _w * y2;   wz = _w * z2;

    // Fill out the matrix 
    matrix[0][0] = 1.0f - (yy + zz);  matrix[0][1] = xy - wz;
    matrix[0][2] = xz + wy;           matrix[0][3] = 0.0f;
    
    matrix[1][0] = xy + wz;           matrix[1][1] = 1.0f - (xx + zz);
    matrix[1][2] = yz - wx;           matrix[1][3] = 0.0f;
    
    matrix[2][0] = xz - wy;           matrix[2][1] = yz + wx;
    matrix[2][2] = 1.0f - (xx + yy);  matrix[2][3] = 0.0f;
    
    matrix[3][0] = 0.0f;              matrix[3][1] = 0.0f;
    matrix[3][2] = 0.0f;              matrix[3][3] = 1.0f;
}

Matrix3D Quaternion::toMatrix() const
{
    Matrix3D matrix;
    float wx, wy, wz, xx, yy, yz, xy, xz, zz, x2, y2, z2;
    
    // Calculate coefficients (to minimize redundancy)
    x2 = _v.x() + _v.x(); y2 = _v.y() + _v.y(); z2 = _v.z() + _v.z();
    
    xx = _v.x() * x2;   xy = _v.x() * y2;   xz = _v.x() * z2;
    yy = _v.y() * y2;   yz = _v.y() * z2;   zz = _v.z() * z2;
    
    wx = _w * x2;   wy = _w * y2;   wz = _w * z2;

    // Fill out the matrix 
    matrix(0, 0) = 1.0f - (yy + zz);  matrix(0, 1) = xy - wz;
    matrix(0, 2) = xz + wy;           matrix(0, 3) = 0.0f;
    
    matrix(1, 0) = xy + wz;           matrix(1, 1) = 1.0f - (xx + zz);
    matrix(1, 2) = yz - wx;           matrix(1, 3) = 0.0f;
    
    matrix(2, 0) = xz - wy;           matrix(2, 1) = yz + wx;
    matrix(2, 2) = 1.0f - (xx + yy);  matrix(2, 3) = 0.0f;
    
    matrix(3, 0) = 0.0f;              matrix(3, 1) = 0.0f;
    matrix(3, 2) = 0.0f;              matrix(3, 3) = 1.0f;

    return matrix;
}

void Quaternion::toAngleAxis(float &angle, Vector3D &axis) const
{
    float scale, tw;

    tw = (float) acos(_w) * 2.0f;
    scale = (float) sin(tw / 2.0f);
    axis.x() = _v.x() / scale;
    axis.y() = _v.y() / scale;
    axis.z() = _v.z() / scale;
    angle = Math::radToDeg(tw);
}

void Quaternion::fromEuler(float x, float y, float z)
{
    float rx, ry, rz, ti, tj, tk;
    Quaternion qx, qy, qz;

    // Convert angles to radians
    rx = Math::degToRad(x);
    ry = Math::degToRad(y);
    rz = Math::degToRad(z);

    // Get the "half" angles
    ti = rx * 0.5f;
    tj = ry * 0.5f;
    tk = rz * 0.5f;

    // Calculate intermediate quaternions
    qx._v.x() = (float) sin(ti); qx._v.y() = 0.0f; qx._v.z() = 0.0f; qx._w = (float) cos(ti);
    qy._v.x() = 0.0f; qy._v.y() = (float) sin(tj); qy._v.z() = 0.0f; qy._w = (float) cos(tj);
    qz._v.x() = 0.0f; qz._v.y() = 0.0f; qz._v.z() = (float) sin(tk); qz._w = (float) cos(tk);

    // Compute the final quaternion (here, the order is Y, X, Z)
    // @@@ Configure this? 
    *this = qy * qx;
    *this = *this * qz;
}

Quaternion Quaternion::computeSLERP(const Quaternion &from,
				    const Quaternion &to, float t)
{
    Quaternion result;
    float tempTo[4];
    float scale0, scale1;

    // Calculate the cosine using the dot product
    float cosom = Vector3D::dot(from._v, to._v) + from._w * to._w;
    
    // Adjust signs if necessary
    if (cosom < 0.0f) {

	// Negate signs
	cosom = -cosom;
	tempTo[0] = -to._v.x();
	tempTo[1] = -to._v.y();
	tempTo[2] = -to._v.z();
	tempTo[3] = -to._w;
    }
    else {
	// Keep the signs
	tempTo[0] = to._v.x();
	tempTo[1] = to._v.y();
	tempTo[2] = to._v.z();
	tempTo[3] = to._w;
    }

    // Calculate coefficients differenly depending on how far away we are
    if ((1.0 - cosom) > Math::epsilon) {
		
	// Far away, do a SLERP
	float omega = (float) acos(cosom);
	float sinom = (float) sin(omega);
	scale0 = (float) sin((1.0 - t) * omega) / sinom;
	scale1 = (float) sin(t * omega) / sinom;
    }
    else {

	// Very close, we can do a linear interpolation
	scale0 = 1.0f - t;
	scale1 = t;
    }

    // Now we compute the final values
    result._v.x() = scale0 * from._v.x() + scale1 * tempTo[0];
    result._v.y() = scale0 * from._v.y() + scale1 * tempTo[1];
    result._v.z() = scale0 * from._v.z() + scale1 * tempTo[2];
    result._w     = scale0 * from._w     + scale1 * tempTo[3];

    return result;
}

std::ostream &operator << (std::ostream &f, const Quaternion &q)
{
    f << "(" << q._v << ", " << q._w << ")";
    return f;
}
