/* ------------------------------------------------------------------------
 * server.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <iostream>
#include <stdlib.h>

// -- 3Dwm Includes
#include "config.h"
#include "Celsius/Mutex.hh"
#include "Celsius/Exception.hh"
#include "Celsius/Thread.hh"
#include "Celsius/Logger.hh"
#include "Celsius/Path.hh"
#include "Celsius/User.hh"
#include "Polhem/SceneManager.hh"
#include "Polhem/EventManager.hh"
#include "Polhem/ServerImpl.hh"
#include "Polhem/LogGroup.hh"
#include "Polhem/Platform.hh"
#include "Polhem/RendererImpl.hh"
#include "Polhem/View.hh"
#include "Polhem/SystemCommand.hh"

#include "Config.hh"

//-- Class Declarations

/**
 * Display server class. 
 **/
class DisplayServer {
public:

    /**
     * Constructor. Initializes all the components of the display
     * server.
     **/
    DisplayServer(int &argc, char **argv);

    /**
     * Destructor. 
     **/
    ~DisplayServer();

    /**
     * Run the display server. 
     **/
    void runServer();
    
private:    
    CORBA::ORB_var _orb;
    PortableServer::POA_var _poa;    
    View *_view;
    ServerImpl *_server;
    SceneManager *_smanager;
    EventManager *_emanager;
};

// -- Code Segment

DisplayServer::DisplayServer(int &argc, char **argv) 
    : _orb(CORBA::ORB_init(argc, argv, "omniORB3")),
      _emanager(new EventManager)
{
    Logger::subscribe(LogGroup::Config);

    // Retrieve the config filename and read the configuration file
    const char *cfgFile = getenv("TDWMRC");
    if (cfgFile) Config::parse(Path::expandUser(cfgFile));
    else Config::parse(User().getHome() + "/.tdwmrc");
    
    // Subscribe to the predefined message groups
    for (int i = 0; i < Config::getLogGroups().size(); i++)
	Logger::subscribe(Config::getLogGroups()[i]);

    // Emit some initial output
    Logger::log(LogGroup::General) << "Logging service started." << std::endl;
    Logger::log(LogGroup::General) << "This is " << PACKAGE << " " << VERSION 
				   << "." << std::endl;
    Logger::log(LogGroup::General) << "Initializing server..." << std::endl;

    // Resolve root portable object adapter
    Logger::log(LogGroup::Corba) << "Resolving root POA." << std::endl;
    _poa = resolve_init<PortableServer::POA>(_orb, "RootPOA");
    
    // Activate the POA
    PortableServer::POAManager_var pman = _poa->the_POAManager();
    pman->activate();
    
    // Initialize the platform abstraction
    Logger::log(LogGroup::Platform) << "Initializing platform ("
				    << Config::getScreenWidth() << "x" 
				    << Config::getScreenHeight() << ", "
				    << Config::getScreenDepth() << " bpp)."
				    << std::endl;
    Platform::init(argc, argv, Config::getScreenWidth(),
		   Config::getScreenHeight(), Config::getScreenDepth());
    
    // Create the renderer
    Logger::log(LogGroup::Platform) << "Platform creating renderer..." 
				    << std::endl;
    RendererImpl *r = Platform::createRenderer();
    if (!r) {
	// When multiple renderers are available it might be possible
	// to create another enumerated renderer rather than
	// aborting right away.
	throw Exception("Backplane::Backplane - cannot create renderer");
    }
    Logger::log(LogGroup::Renderer) << "Renderer: " << r->getName() 
				    << std::endl;
    Logger::log(LogGroup::Renderer) << "Vendor: " << r->getVendor() 
				    << std::endl;
    Logger::log(LogGroup::Renderer) << "Version: " << r->getVersion()
				    << std::endl;
    
    // Create the initial view passing in the the renderer returned
    // from the platform.
    Logger::log(LogGroup::General) << "Creating main view." << std::endl;
    _view = new View(r);
    
    // Create the scene manager, passing in the initial view
    Logger::log(LogGroup::General) << "Creating scene manager." << std::endl;
    _smanager = new SceneManager(_view);

    // Create and activate the server instance
    _server = new ServerImpl(_smanager);
    Logger::log(LogGroup::Server) << "Activating server CORBA object." 
				  << std::endl;
    PortableServer::ObjectId_var oid = _poa->activate_object(_server);
    
    // Create command mapping object and pass it to the event manager
    _emanager->addEventMappingFunctions(new SystemCommand(_smanager));

    // Load event mapping rule files into the event manager
    for (int i = 0; i < Config::getRuleFiles().size(); i++)
	_emanager->loadEventMappingRuleFile
	    (Path::expandUser(Config::getRuleFiles()[i]));

    // Bind the server in the naming service
    Logger::log(LogGroup::Corba) << "Registering \"" 
				 << Nobel::Server::_PD_repoId 
				 << "\" with the name service."
				 << std::endl;
    bind_name(_orb, Nobel::Server_var(_server->_this()),
	      Nobel::Server::_PD_repoId);
    
    Logger::log(LogGroup::General) << "Server initialized." << std::endl;
}

DisplayServer::~DisplayServer() 
{
    // Deactive CORBA-exposed objects
    deactivate(_server);
    
    // Delete all objects
    delete _view;
    delete _server;
    delete _smanager;
    delete _emanager;
}

void DisplayServer::runServer()
{
    Logger::log(LogGroup::General) << "Starting threads..." << std::endl;
    
    // Create the threads
    Thread server_thread(_server);
    Thread event_thread(_emanager);
    
    // Start the different threads
    //server_thread.start();
    event_thread.start();

    // Run the scene manager until it returns (=server shutdown)
    _smanager->run();

    // Terminate the server and event manager
    // @@@ There is a bug related to the server thread here!
    //_server->terminate();
    _emanager->terminate();
    
    // Wait until they are terminated
    server_thread.join();
    event_thread.join();
    
    Logger::log(LogGroup::General) << "3Dwm halted." << std::endl;
}

int main(int argc, char **argv)
{
    try {

	// Create the display server
	DisplayServer displayserver(argc, argv);

	// Run it until termination
	displayserver.runServer();
    }
    catch (const Exception &e) {
	Logger::log(LogGroup::General) << e << std::endl;
	return EXIT_FAILURE;
    }
    catch (...) {
	Logger::log(LogGroup::General) << "Exception thrown. Exiting." 
					<< std::endl;	
	return EXIT_FAILURE;
    }
    
    return EXIT_SUCCESS;
}
