/* ------------------------------------------------------------------------
 * $Id: FunctionSet.hh,v 1.1 2001/06/19 22:48:00 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _FunctionSet_hh_
#define _FunctionSet_hh_

// -- System Includes
#include <map>
#include <string>

// -- Zorn Includes
#include "Instance.hh"

namespace Zorn {

    // -- Forward Declarations
    class FunctionObject;

    // -- Class Declarations
    
    /**
     * Zorn function set class.
     **/
    class FunctionSet {
    public:
    
	/**
	 * Memory-management instance class.
	 **/
	ZORN_INSTANCE_CLASS(FunctionSet);

        /**
	 * Singleton instance function.
	 **/
	static FunctionSet *instance() {
	    if (_instance == 0) _instance = new FunctionSet();
	    return _instance;
	}
    
	/**
	 * Add a function object to the function set. All implemented
	 * Zorn functions in the function object will be mapped into
	 * the function set, and the reference count for the function
	 * object will be bumped for each.
	 *
	 * @param fobj function object to add.
	 **/
	void addFunctionObject(FunctionObject *fobj);

	/**
	 * Retrieve the number of Zorn functions contained in the
	 * function set.
	 *
	 * @return the number of Zorn functions.
	 **/
	int getFunctionCount() const {
	    return _functions.size();
	}

	/**
	 * Lookup a Zorn function given its name and return the
	 * corresponding function object (if it is found). 
	 *
	 * @param id Zorn function name.
	 * @return function object implementing the Zorn function.
	 **/
	FunctionObject *find(const std::string &id) const;

    private:

	/**
	 * Constructor (private, singleton class).
	 **/
	FunctionSet();
	
	/**
	 * Destructor (private, singleton class).
	 **/
	~FunctionSet();

	/**
	 * Add a function object implementing a Zorn function
	 * identified by a name to the function set.
	 *
	 * @param id Zorn function identifier.
	 * @param fobj function object that implements the function.
	 **/ 
	void addFunction(const std::string &id, FunctionObject *fobj);
	
	static FunctionSet *_instance;
	std::map<std::string, FunctionObject *> _functions;
    };

};

#endif /* FunctionSet.hh */
