/* ------------------------------------------------------------------------
 * $Id: Solid.hh,v 1.7 2001/08/20 14:18:28 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Solid_hh_
#define _Solid_hh_

// -- 3Dwm Includes
#include "Nobel/CORBA.hh"
#include "Nobel/Types.hh"
#include "Polhem/EmitterImpl.hh"
#include "Nobel/Solid.hh"

// -- Forward Declarations
class TransformImpl;

// -- Class Declarations

namespace SolidImpl {

    /**
     * C++ implementation of the base solid node interface.
     **/ 
    class Node : public virtual POA_Nobel::Solid::Node,
		 public EmitterImpl {
    public:
	
	/**
	 * Constructor.
	 **/
	Node() { }
	
	/**
	 * Destructor.
	 **/
	virtual ~Node() { }
	
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v) = 0;
	virtual void receive(const Nobel::Event &e);
    protected:	
	void emitChangedEvent();
	Mutex _mutex;
    };

    /**
     * C++ implementation of the unary set operator. Contains one
     * solid node that it operates on.
     **/    
    class Unary : public virtual POA_Nobel::Solid::Unary,
		  public Node {
    public:

	/**
	 * Constructor.
	 **/
	Unary() { }
	
	/**
	 * Destructor.
	 **/
	virtual ~Unary() { }
	
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v) = 0;
	virtual Nobel::Solid::Node_ptr body();
	virtual void body(Nobel::Solid::Node_ptr b);
    private:
	Nobel::Solid::Node_var _body;
    };
    
    /**
     * C++ implementation of the binary set operator. Contains two
     * solid nodes that it operates on.
     **/
    class Binary : public virtual POA_Nobel::Solid::Binary,
		   public Node {
    public:

	/**
	 * Constructor.
	 **/
	Binary() { }
	
	/**
	 * Destructor.
	 **/
	virtual ~Binary() { }
	    
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v) = 0;
	virtual Nobel::Solid::Node_ptr left();
	virtual void left(Nobel::Solid::Node_ptr l);
	virtual Nobel::Solid::Node_ptr right();
	virtual void right(Nobel::Solid::Node_ptr r);
    private:
	Nobel::Solid::Node_var _left, _right;
    };

    /**
     * C++ implementation of the geometry container. Used for
     * including arbitrary geometry (b-reps) in a solid tree.
     **/    
    class Geometry : public virtual POA_Nobel::Solid::Geometry,
		     public Node {
    public:
	
	/**
	 * Constructor.
	 *
	 * @param g geometry to contain inside the node.
	 **/
	Geometry(Nobel::TriangleGeometry_ptr g);

	/**
	 * Destructor. 
	 **/
	virtual ~Geometry();

	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
	virtual Nobel::TriangleGeometry_ptr geo();
	virtual void geo(Nobel::TriangleGeometry_ptr g);
	virtual void receive(const Nobel::Event &e, Nobel::Emitter_ptr o);
    private:
	Nobel::TriangleGeometry_var _geo;
    };

    /**
     * C++ implementation of the primitive container. Used for
     * including 3D primitives in a solid tree.
     **/    
    class Primitive : public virtual POA_Nobel::Solid::Primitive,
		      public Node {
    public:
	
	/**
	 * Constructor.
	 *
	 * @param p primitive to contain inside the node.
	 **/
	Primitive(Nobel::Primitive_ptr p);

	/**
	 * Destructor. 
	 **/
	virtual ~Primitive();

	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
	virtual Nobel::Primitive_ptr prim();
	virtual void prim(Nobel::Primitive_ptr p);
    private:
	Nobel::Primitive_var _prim;
    };
    
    /**
     * C++ implementation of the transformation node. Used for adding
     * affine transformations to a solid tree. Children are contained
     * in the body of the node.
     **/
    class Transform : public virtual POA_Nobel::Solid::Transform,
		      public Unary {
    public:
	
	/**
	 * Constructor.
	 **/ 
	Transform();
	
	/**
	 * Destructor.
	 **/
	virtual ~Transform();
	
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
	virtual void identity();
	virtual void scale(const Nobel::Vertex3D &s);
	virtual void translate(const Nobel::Vertex3D &t);
	virtual void rotate(CORBA::Float angle, const Nobel::Vertex3D &axis);
	virtual void getTransformationMatrix(Nobel::Matrix m);
    private:
	TransformImpl *_transform;
    };

    /**
     * C++ implementation of the appearance container.
     **/
    class Appearance : public virtual POA_Nobel::Solid::Appearance,
		       public Unary {
    public:
	
	/**
	 * Constructor.
	 *
	 * @param a appearance object to use.
	 **/
	Appearance(Nobel::Appearance_ptr a);

	/**
	 * Destructor.
	 **/ 
	virtual ~Appearance() { }
	
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
	virtual Nobel::Appearance_ptr app();
	virtual void app(Nobel::Appearance_ptr);
    private:
	Nobel::Appearance_var _app;
    };

    /**
     * Complement bit set operator. 
     **/    
    class Complement : public Unary {
    public:
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
    };

    /**
     * Union bit set operator. 
     **/    
    class Union : public Binary {
    public:
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
    };
    
    /**
     * Intersection bit set operator. 
     **/    
    class Intersection : public Binary {
    public:
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
    };
    
    /**
     * Subtraction bit set operator. 
     **/    
    class Subtraction : public Binary {
    public:
	// IDL functions
	virtual void accept(Nobel::Solid::Visitor_ptr v);
    };

};

#endif /* Solid.hh */ 
